package org.ojrandom.paiesque.rhr;

import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.time.LocalDate;
import java.util.*;

public class SimpleRHRCalculator {
    private static final int MIN_READINGS = 30;
    private static final String TAG = "SimpleRHRCalculator";

    public RhrResult calculateDailyRHR(List<Map<String, String>> data, LocalDate date,
                                       int windowStartMinutes, int windowEndMinutes,
                                       HistoricalBaseline baseline) {

        // Filter to user-defined window and apply baseline filtering
        List<Integer> validHRs = new ArrayList<>();
        for (Map<String, String> entry : data) {
            try {
                int hr = Integer.parseInt(entry.get("HEART_RATE"));
                int minutesOfDay = Integer.parseInt(entry.get("MINUTES_OF_DAY"));

                if (isInTimeWindow(minutesOfDay, windowStartMinutes, windowEndMinutes) &&
                        isWithinBaseline(hr, baseline)) {
                    validHRs.add(hr);
                }
            } catch (Exception e) {
                continue;
            }
        }

        if (validHRs.size() < MIN_READINGS) {
            AppLogger.d(TAG, "Insufficient data in RHR window: " + validHRs.size() + " readings");
            return null;
        }

        // Use adaptive percentile based on data quality
        int percentile = determineOptimalPercentile(validHRs);
        int rhr = calculatePercentileRHR(validHRs, percentile);

        AppLogger.d(TAG, date + ": RHR " + rhr + " bpm (P" + percentile + ") from " + validHRs.size() +
                " readings in window " + formatTime(windowStartMinutes) + "-" + formatTime(windowEndMinutes));

        // UPDATED: Remove strategy string from constructor
        return new RhrResult(rhr, validHRs.size());
    }

    private int determineOptimalPercentile(List<Integer> heartRates) {
        int count = heartRates.size();
        if (count >= 300) {
            return 5;   // High-quality data: match device behavior
        } else if (count >= 150) {
            return 8;   // Good data: slightly conservative
        } else if (count >= 75) {
            return 10;  // Moderate data: original approach
        } else {
            return 15;  // Sparse data: more conservative
        }
    }

    private boolean isInTimeWindow(int minutesOfDay, int windowStart, int windowEnd) {
        if (windowStart <= windowEnd) {
            return minutesOfDay >= windowStart && minutesOfDay <= windowEnd;
        } else {
            return minutesOfDay >= windowStart || minutesOfDay <= windowEnd;
        }
    }

    private boolean isWithinBaseline(int hr, HistoricalBaseline baseline) {
        return hr >= baseline.personalizedMin && hr <= baseline.personalizedMax;
    }

    private int calculatePercentileRHR(List<Integer> heartRates, int percentile) {
        Collections.sort(heartRates);
        int index = (int) Math.ceil(percentile / 100.0 * heartRates.size()) - 1;
        index = Math.max(0, Math.min(index, heartRates.size() - 1));
        return heartRates.get(index);
    }

    private String formatTime(int totalMinutes) {
        int hours = totalMinutes / 60;
        int minutes = totalMinutes % 60;
        return String.format("%02d:%02d", hours, minutes);
    }
}