package org.ojrandom.paiesque.rhr;

import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class HeartRateQuery {
    private static final String TAG = "HeartRateQuery";
    private static final boolean DEBUG = false; // Set to true for debug output

    private final List<Map<String, String>> sortedData;
    private final String timestampKey;

    public HeartRateQuery(List<Map<String, String>> heartRateData, String timestampKey) {
        if (DEBUG) System.out.println(TAG + ": Step 1: Initializing HeartRateQuery with SECONDS only");
        this.timestampKey = timestampKey;

        if (DEBUG) System.out.println(TAG + ": Step 2: Validating and sorting data");
        validateTimestampKey();

        // Create a copy and sort the data
        List<Map<String, String>> dataToSort = heartRateData != null ?
                new ArrayList<>(heartRateData) : new ArrayList<>();
        this.sortedData = sortDataByTimestampOptimized(dataToSort);

        if (DEBUG) System.out.println(TAG + ": Step 3: HeartRateQuery initialization completed");
    }

    public List<Map<String, String>> getEntriesInTimeRange(long startTimestampSeconds, long endTimestampSeconds) {
        if (DEBUG) System.out.println(TAG + ": Step 4: Starting query for timestamp between " + startTimestampSeconds + " and " + endTimestampSeconds + " (SECONDS)");

        if (sortedData.isEmpty()) {
            if (DEBUG) System.out.println(TAG + ": Step 6: No data available, returning empty list");
            return Collections.emptyList();
        }

        if (DEBUG) System.out.println(TAG + ": Step 7: Performing binary search on " + sortedData.size() + " entries");
        int startIndex = binarySearchFirstIndex(startTimestampSeconds);
        int endIndex = binarySearchLastIndex(endTimestampSeconds);

        if (startIndex == -1 || endIndex == -1 || startIndex > endIndex) {
            if (DEBUG) System.out.println(TAG + ": Step 8: No entries found in range [" + startTimestampSeconds + "-" + endTimestampSeconds + "] (SECONDS)");
            return Collections.emptyList();
        }

        int resultSize = endIndex - startIndex + 1;
        if (DEBUG) System.out.println(TAG + ": Step 9: Found " + resultSize + " entries from index " + startIndex + " to " + endIndex);
        return new ArrayList<>(sortedData.subList(startIndex, endIndex + 1));
    }

    public List<Map<String, String>> getEntriesFromTimestamp(long timestampSeconds) {
        if (DEBUG) System.out.println(TAG + ": Step 4: Starting query for timestamp >= " + timestampSeconds + " (SECONDS)");

        if (sortedData.isEmpty()) {
            if (DEBUG) System.out.println(TAG + ": Step 6: No data available, returning empty list");
            return Collections.emptyList();
        }

        if (DEBUG) System.out.println(TAG + ": Step 7: Performing binary search on " + sortedData.size() + " entries");
        int startIndex = binarySearchFirstIndex(timestampSeconds);

        if (startIndex == -1) {
            if (DEBUG) System.out.println(TAG + ": Step 8: No entries found for timestamp >= " + timestampSeconds + " (SECONDS)");
            return Collections.emptyList();
        }

        int resultSize = sortedData.size() - startIndex;
        if (DEBUG) System.out.println(TAG + ": Step 9: Found " + resultSize + " entries starting from index " + startIndex);
        return new ArrayList<>(sortedData.subList(startIndex, sortedData.size()));
    }

    private int binarySearchLastIndex(long timestampSeconds) {
        if (DEBUG) System.out.println(TAG + ": Step 11: Starting binary search for last index <= " + timestampSeconds);
        int low = 0;
        int high = sortedData.size() - 1;
        int result = -1;
        int iterations = 0;

        while (low <= high) {
            iterations++;
            int mid = (low + high) >>> 1;
            Long midTimestamp = getTimestampSafe(sortedData.get(mid));

            if (midTimestamp == null) {
                if (DEBUG) System.out.println(TAG + ": Step 12: Invalid timestamp at index " + mid + ", searching around it");
                int rightResult = binarySearchLastIndexInRange(timestampSeconds, mid + 1, high);
                if (rightResult != -1) {
                    if (DEBUG) System.out.println(TAG + ": Step 13: Found valid entry in right partition at index " + rightResult);
                    return rightResult;
                }
                if (DEBUG) System.out.println(TAG + ": Step 14: No valid entry in right partition, searching left");
                return binarySearchLastIndexInRange(timestampSeconds, low, mid - 1);
            }

            if (DEBUG) System.out.println(TAG + ": Binary search iteration " + iterations + ": low=" + low + ", high=" + high +
                    ", mid=" + mid + ", midTimestamp=" + midTimestamp);

            if (midTimestamp <= timestampSeconds) {
                if (DEBUG) System.out.println(TAG + ": Step 15: Candidate found at index " + mid + " (timestamp: " + midTimestamp + ")");
                result = mid;
                low = mid + 1;
            } else {
                if (DEBUG) System.out.println(TAG + ": Step 16: Too high at index " + mid + " (timestamp: " + midTimestamp + ")");
                high = mid - 1;
            }
        }

        if (DEBUG) System.out.println(TAG + ": Step 17: Binary search completed in " + iterations + " iterations, result: " + result);
        return result;
    }

    private int binarySearchLastIndexInRange(long timestampSeconds, int low, int high) {
        if (DEBUG) System.out.println(TAG + ": Step 18: Starting partition search for last index from " + low + " to " + high);
        int result = -1;
        int iterations = 0;

        while (low <= high) {
            iterations++;
            int mid = (low + high) >>> 1;
            Long midTimestamp = getTimestampSafe(sortedData.get(mid));

            if (midTimestamp == null) {
                if (DEBUG) System.out.println(TAG + ": Step 19: Skipping invalid entry at index " + mid);
                high = mid - 1;
                continue;
            }

            if (DEBUG) System.out.println(TAG + ": Partition search iteration " + iterations + ": low=" + low + ", high=" + high +
                    ", mid=" + mid + ", midTimestamp=" + midTimestamp);

            if (midTimestamp <= timestampSeconds) {
                if (DEBUG) System.out.println(TAG + ": Step 20: Valid candidate found at index " + mid + " in partition");
                result = mid;
                low = mid + 1;
            } else {
                if (DEBUG) System.out.println(TAG + ": Step 21: Too high at index " + mid + " in partition");
                high = mid - 1;
            }
        }

        if (DEBUG) System.out.println(TAG + ": Step 22: Partition search completed, result: " + result);
        return result;
    }

    private int binarySearchFirstIndex(long timestampSeconds) {
        if (DEBUG) System.out.println(TAG + ": Step 11: Starting binary search for timestamp " + timestampSeconds);
        int low = 0;
        int high = sortedData.size() - 1;
        int result = -1;
        int iterations = 0;

        while (low <= high) {
            iterations++;
            int mid = (low + high) >>> 1;
            Long midTimestamp = getTimestampSafe(sortedData.get(mid));

            if (midTimestamp == null) {
                if (DEBUG) System.out.println(TAG + ": Step 12: Invalid timestamp at index " + mid + ", searching around it");
                int leftResult = binarySearchFirstIndexInRange(timestampSeconds, low, mid - 1);
                if (leftResult != -1) {
                    if (DEBUG) System.out.println(TAG + ": Step 13: Found valid entry in left partition at index " + leftResult);
                    return leftResult;
                }
                if (DEBUG) System.out.println(TAG + ": Step 14: No valid entry in left partition, searching right");
                return binarySearchFirstIndexInRange(timestampSeconds, mid + 1, high);
            }

            if (DEBUG) System.out.println(TAG + ": Binary search iteration " + iterations + ": low=" + low + ", high=" + high +
                    ", mid=" + mid + ", midTimestamp=" + midTimestamp);

            if (midTimestamp >= timestampSeconds) {
                if (DEBUG) System.out.println(TAG + ": Step 15: Candidate found at index " + mid + " (timestamp: " + midTimestamp + ")");
                result = mid;
                high = mid - 1;
            } else {
                if (DEBUG) System.out.println(TAG + ": Step 16: Too low at index " + mid + " (timestamp: " + midTimestamp + ")");
                low = mid + 1;
            }
        }

        if (DEBUG) System.out.println(TAG + ": Step 17: Binary search completed in " + iterations + " iterations, result: " + result);
        return result;
    }

    private int binarySearchFirstIndexInRange(long timestampSeconds, int low, int high) {
        if (DEBUG) System.out.println(TAG + ": Step 18: Starting partition search from " + low + " to " + high);
        int result = -1;
        int iterations = 0;

        while (low <= high) {
            iterations++;
            int mid = (low + high) >>> 1;
            Long midTimestamp = getTimestampSafe(sortedData.get(mid));

            if (midTimestamp == null) {
                if (DEBUG) System.out.println(TAG + ": Step 19: Skipping invalid entry at index " + mid);
                low = mid + 1;
                continue;
            }

            if (DEBUG) System.out.println(TAG + ": Partition search iteration " + iterations + ": low=" + low + ", high=" + high +
                    ", mid=" + mid + ", midTimestamp=" + midTimestamp);

            if (midTimestamp >= timestampSeconds) {
                if (DEBUG) System.out.println(TAG + ": Step 20: Valid candidate found at index " + mid + " in partition");
                result = mid;
                high = mid - 1;
            } else {
                if (DEBUG) System.out.println(TAG + ": Step 21: Too low at index " + mid + " in partition");
                low = mid + 1;
            }
        }

        if (DEBUG) System.out.println(TAG + ": Step 22: Partition search completed, result: " + result);
        return result;
    }

    private Long getTimestampSafe(Map<String, String> entry) {
        if (entry == null) {
            if (DEBUG) System.out.println(TAG + ": Step 23: Entry is null");
            return null;
        }

        String timestampStr = entry.get(timestampKey);
        if (timestampStr == null) {
            if (DEBUG) System.out.println(TAG + ": Step 24: Timestamp key '" + timestampKey + "' not found in entry");
            return null;
        }

        try {
            long timestamp = Long.parseLong(timestampStr.trim());
            if (DEBUG) System.out.println(TAG + ": Step 25: Successfully parsed timestamp: " + timestamp);
            return timestamp;
        } catch (NumberFormatException e) {
            if (DEBUG) System.out.println(TAG + ": Step 26: Invalid timestamp format: '" + timestampStr + "'");
            return null;
        }
    }

    public void printDataQualityReport() {
        if (DEBUG) System.out.println(TAG + ": Step 42: Generating data quality report");

        if (sortedData.isEmpty()) {
            System.out.println(TAG + ": Data quality report - No data available");
            return;
        }

        int totalEntries = sortedData.size();
        int validTimestamps = 0;
        int missingKey = 0;
        int invalidFormat = 0;

        if (DEBUG) System.out.println(TAG + ": Step 44: Analyzing " + totalEntries + " entries");
        for (int i = 0; i < sortedData.size(); i++) {
            Map<String, String> entry = sortedData.get(i);
            String timestampStr = entry.get(timestampKey);
            if (timestampStr == null) {
                missingKey++;
            } else {
                try {
                    Long.parseLong(timestampStr.trim());
                    validTimestamps++;
                } catch (NumberFormatException e) {
                    invalidFormat++;
                }
            }

            if (totalEntries > 1000 && (i + 1) % 1000 == 0) {
                if (DEBUG) System.out.println(TAG + ": Step 45: Processed " + (i + 1) + "/" + totalEntries + " entries for quality report");
            }
        }

        int validPercent = (validTimestamps * 100) / totalEntries;
        int missingPercent = (missingKey * 100) / totalEntries;
        int invalidPercent = (invalidFormat * 100) / totalEntries;

        System.out.println(TAG + ": Data Quality Report:");
        System.out.println(TAG + ": Total entries: " + totalEntries);
        System.out.println(TAG + ": Valid timestamps: " + validTimestamps + " (" + validPercent + "%)");
        System.out.println(TAG + ": Missing key: " + missingKey + " (" + missingPercent + "%)");
        System.out.println(TAG + ": Invalid format: " + invalidFormat + " (" + invalidPercent + "%)");
        System.out.println(TAG + ": All timestamps are in SECONDS");

        if (validPercent < 50) {
            System.out.println(TAG + ": WARNING: Less than 50% of timestamps are valid");
        }
    }

    private void validateTimestampKey() {
        if (DEBUG) System.out.println(TAG + ": Step 27: Validating timestamp key configuration");

        if (timestampKey == null || timestampKey.trim().isEmpty()) {
            String error = "Timestamp key cannot be null or empty";
            if (DEBUG) System.out.println(TAG + ": Step 28: " + error);
            throw new IllegalArgumentException(error);
        }

        if (DEBUG) System.out.println(TAG + ": Step 29: Timestamp key is valid: '" + timestampKey + "'");
    }

    private List<Map<String, String>> sortDataByTimestampOptimized(List<Map<String, String>> data) {
        if (data.size() <= 1) return data;

        // Precompute timestamps once to avoid repeated parsing
        List<TimestampedEntry> entries = new ArrayList<>(data.size());

        for (Map<String, String> entry : data) {
            Long timestamp = getTimestampSafe(entry);
            entries.add(new TimestampedEntry(entry, timestamp));
        }

        // Sort using precomputed timestamps
        Collections.sort(entries, (e1, e2) -> {
            if (e1.timestamp == null && e2.timestamp == null) return 0;
            if (e1.timestamp == null) return -1;
            if (e2.timestamp == null) return 1;
            return Long.compare(e1.timestamp, e2.timestamp);
        });

        // Extract sorted entries
        List<Map<String, String>> result = new ArrayList<>(data.size());
        for (TimestampedEntry entry : entries) {
            result.add(entry.data);
        }

        return result;
    }

    // Helper class to cache parsed timestamps
    private static class TimestampedEntry {
        final Map<String, String> data;
        final Long timestamp;

        TimestampedEntry(Map<String, String> data, Long timestamp) {
            this.data = data;
            this.timestamp = timestamp;
        }
    }

    public boolean verifyDataIsSorted() {
        if (DEBUG) System.out.println(TAG + ": Verifying data is sorted...");
        if (sortedData.size() < 2) {
            if (DEBUG) System.out.println(TAG + ": Not enough data to verify sorting");
            return true;
        }

        Long prev = getTimestampSafe(sortedData.get(0));
        for (int i = 1; i < Math.min(100, sortedData.size()); i++) {
            Long current = getTimestampSafe(sortedData.get(i));
            if (prev != null && current != null && current < prev) {
                System.out.println(TAG + ": Data NOT sorted at index " + i + ": " + prev + " -> " + current);
                return false;
            }
            prev = current;
        }
        if (DEBUG) System.out.println(TAG + ": Data appears to be sorted correctly");
        return true;
    }
}