package org.ojrandom.paiesque.pai;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.data.SettingsManager;
import org.ojrandom.paiesque.logging.AppLogger;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeMap;

public class HealthMetricsAnalyzer {
    private static final String TAG = "HealthMetricsAnalyzer";

    public static class HealthInsights {
        public final int measuredRHR;
        public final int assumedRHR;
        public final int measuredMaxHR;
        public final int assumedMaxHR;
        public final String rhrInsight;
        public final String maxHrInsight;
        public final String educationalNote;

        // Updated constructor to accept string parameters
        public HealthInsights(int measuredRHR, int assumedRHR,
                              int measuredMaxHR, int assumedMaxHR,
                              String rhrInsight, String maxHrInsight,
                              String educationalNote) {
            this.measuredRHR = measuredRHR;
            this.assumedRHR = assumedRHR;
            this.measuredMaxHR = measuredMaxHR;
            this.assumedMaxHR = assumedMaxHR;
            this.rhrInsight = rhrInsight;
            this.maxHrInsight = maxHrInsight;
            this.educationalNote = educationalNote;
        }

    }

    public HealthInsights analyzeHealthMetrics(TreeMap<LocalDate, Integer> historicalRHR,
                                               SettingsManager settingsManager,
                                               SQLiteDatabase database,
                                               HealthInsightsStrings strings) {  // Add strings parameter
        int assumedRHR = settingsManager.getRHR();
        int assumedMaxHR = settingsManager.getMaxHR();

        // Get measured values
        int measuredRHR = calculateMeasuredRHR(historicalRHR);
        int measuredMaxHR = calculateMeasuredMaxHR(database);

        AppLogger.d(TAG, String.format("Health metrics - RHR: %d vs %d, MaxHR: %d vs %d",
                measuredRHR, assumedRHR, measuredMaxHR, assumedMaxHR));

        // Generate insights using provided strings
        String rhrInsight = generateRHRInsight(measuredRHR, assumedRHR, strings);
        String maxHrInsight = generateMaxHRInsight(measuredMaxHR, assumedMaxHR, strings);
        String educationalNote = generateEducationalNote(measuredMaxHR, assumedMaxHR, settingsManager, strings);

        return new HealthInsights(
                measuredRHR,
                assumedRHR,
                measuredMaxHR,
                assumedMaxHR,
                rhrInsight,
                maxHrInsight,
                educationalNote
        );
    }

    private String generateRHRInsight(int measured, int assumed, HealthInsightsStrings strings) {
        if (measured <= 0) return strings.rhrNoData;

        int diff = assumed - measured;
        if (diff == 0) {
            return String.format(strings.rhrMatches, measured);
        } else if (diff > 0) {
            return String.format(strings.rhrLower, diff, measured, assumed);
        } else {
            return String.format(strings.rhrHigher, -diff, measured, assumed);
        }
    }

    private String generateMaxHRInsight(int measured, int assumed, HealthInsightsStrings strings) {
        if (measured <= 0) return strings.maxHrNoData;

        int diff = measured - assumed;
        if (diff == 0) {
            return String.format(strings.maxHrMatches, measured);
        } else if (diff > 0) {
            return String.format(strings.maxHrHigher, diff, measured, assumed);
        } else {
            return String.format(strings.maxHrLower, -diff, measured, assumed);
        }
    }

    private String generateEducationalNote(int measuredMaxHR, int assumedMaxHR, SettingsManager settings, HealthInsightsStrings strings) {
        boolean isManual = settings.isManualMaxHR();

        if (measuredMaxHR < assumedMaxHR - 5) {
            if (isManual) {
                return strings.educationalNoteLowerManual;
            } else {
                return strings.educationalNoteLowerFormula;
            }
        } else if (measuredMaxHR > assumedMaxHR + 5) {
            if (isManual) {
                return strings.educationalNoteHigherManual;
            } else {
                return strings.educationalNoteHigherFormula;
            }
        } else {
            if (isManual) {
                return strings.educationalNoteMatchesManual;
            } else {
                return strings.educationalNoteMatchesFormula;
            }
        }
    }

    private int calculateMeasuredRHR(TreeMap<LocalDate, Integer> historicalRHR) {
        if (historicalRHR == null || historicalRHR.isEmpty()) {
            return 0;
        }

        LocalDate today = LocalDate.now();
        LocalDate startDate = today.minusDays(30);

        double sum = 0;
        int count = 0;

        for (java.util.Map.Entry<LocalDate, Integer> entry : historicalRHR.entrySet()) {
            LocalDate entryDate = entry.getKey();
            if (!entryDate.isBefore(startDate) && !entryDate.isAfter(today)) {
                sum += entry.getValue();
                count++;
            }
        }

        return count > 0 ? (int) Math.round(sum / count) : 0;
    }

    public int calculateMeasuredMaxHR(SQLiteDatabase database) {
        if (database == null || !database.isOpen()) {
            return 0;
        }

        try {
            String sql = "SELECT HEART_RATE FROM " + AppConstants.PaiesqueTables.HEART_RATES +
                    " WHERE HEART_RATE > 100 ORDER BY HEART_RATE DESC LIMIT 100";

            List<Integer> topHRs = new ArrayList<>();
            try (Cursor cursor = database.rawQuery(sql, null)) {
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        topHRs.add(cursor.getInt(0));
                    }
                }
            }

            if (!topHRs.isEmpty()) {
                int index = (int) (topHRs.size() * 0.05);
                int measuredMaxHR = topHRs.get(Math.max(0, index));
                AppLogger.d(TAG, "Measured max HR (95th percentile): " + measuredMaxHR);
                return measuredMaxHR;
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error calculating measured max HR", e);
        }

        return 0;
    }

    // Helper class to pass string resources
    public static class HealthInsightsStrings {
        public final String rhrNoData;
        public final String rhrMatches;
        public final String rhrLower;
        public final String rhrHigher;
        public final String maxHrNoData;
        public final String maxHrMatches;
        public final String maxHrHigher;
        public final String maxHrLower;
        public final String educationalNoteLowerManual;
        public final String educationalNoteLowerFormula;
        public final String educationalNoteHigherManual;
        public final String educationalNoteHigherFormula;
        public final String educationalNoteMatchesManual;
        public final String educationalNoteMatchesFormula;

        public HealthInsightsStrings(String rhrNoData, String rhrMatches, String rhrLower, String rhrHigher,
                                     String maxHrNoData, String maxHrMatches, String maxHrHigher, String maxHrLower,
                                     String educationalNoteLowerManual, String educationalNoteLowerFormula,
                                     String educationalNoteHigherManual, String educationalNoteHigherFormula,
                                     String educationalNoteMatchesManual, String educationalNoteMatchesFormula) {
            this.rhrNoData = rhrNoData;
            this.rhrMatches = rhrMatches;
            this.rhrLower = rhrLower;
            this.rhrHigher = rhrHigher;
            this.maxHrNoData = maxHrNoData;
            this.maxHrMatches = maxHrMatches;
            this.maxHrHigher = maxHrHigher;
            this.maxHrLower = maxHrLower;
            this.educationalNoteLowerManual = educationalNoteLowerManual;
            this.educationalNoteLowerFormula = educationalNoteLowerFormula;
            this.educationalNoteHigherManual = educationalNoteHigherManual;
            this.educationalNoteHigherFormula = educationalNoteHigherFormula;
            this.educationalNoteMatchesManual = educationalNoteMatchesManual;
            this.educationalNoteMatchesFormula = educationalNoteMatchesFormula;
        }
    }
}