package org.ojrandom.paiesque.data.sync;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.data.DeviceDiscoveryService;
import org.ojrandom.paiesque.data.GadgetbridgeSyncSource;
import org.ojrandom.paiesque.data.SyncCheckpointManager;
import org.ojrandom.paiesque.data.repositories.HeartRateRepository;
import org.ojrandom.paiesque.logging.AppLogger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;

public class HeartRateSyncService {
    private static final String TAG = "HeartRateSyncService";
    private static final int BATCH_SIZE = 1000;

    private final GadgetbridgeSyncSource gadgetbridgeSyncSource;
    private final HeartRateRepository heartRateRepository;
    private final SyncCheckpointManager checkpointManager;
    private final ExecutorService executorService;

    public HeartRateSyncService(GadgetbridgeSyncSource gadgetbridgeSyncSource,
                                HeartRateRepository heartRateRepository,
                                SyncCheckpointManager checkpointManager,
                                ExecutorService executorService) {
        this.gadgetbridgeSyncSource = gadgetbridgeSyncSource; // Updated parameter
        this.heartRateRepository = heartRateRepository;
        this.checkpointManager = checkpointManager;
        this.executorService = executorService;
    }

    public SyncSummary syncAllHeartRateTablesOptimized(DeviceDiscoveryService.DiscoveryResult discoveryResult) {
        AppLogger.i(TAG, "=== SYNC DEBUG: Starting sync ===");

        // Log cache stats at start
        AppLogger.d(TAG, TimestampUnitDetector.getCacheStats());

        Map<String, Long> deviceMap = discoveryResult.deviceMap;
        Map<String, String> sourceDeviceNames = discoveryResult.deviceNames;

        long globalLastTimestamp = checkpointManager.getGlobalLastSyncTimestamp();
        AppLogger.d(TAG, "=== SYNC DEBUG: globalLastTimestamp = " + globalLastTimestamp +
                " (" + (globalLastTimestamp == 0 ? "FULL SYNC" :
                "INCREMENTAL since " +
                        new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss", java.util.Locale.getDefault())
                                .format(new java.util.Date(globalLastTimestamp))) + ") ===");

        AppLogger.i(TAG, "Starting comprehensive heart rate sync");
        Map<String, AppConstants.Manufacturers.ManufacturerConfig> configs =
                AppConstants.Manufacturers.getManufacturerConfigs();

        // Build targetId -> deviceName map
        Map<Long, String> targetDeviceNames = new HashMap<>();
        for (Map.Entry<String, Long> entry : deviceMap.entrySet()) {
            String sourceId = entry.getKey();
            Long targetId = entry.getValue();
            String deviceName = sourceDeviceNames.get(sourceId);
            if (deviceName != null && targetId != null) {
                targetDeviceNames.put(targetId, deviceName);
            }
        }

        List<CompletableFuture<TableSyncResult>> futures = new ArrayList<>();

        for (Map.Entry<String, AppConstants.Manufacturers.ManufacturerConfig> entry : configs.entrySet()) {
            String manufacturer = entry.getKey();
            AppConstants.Manufacturers.ManufacturerConfig config = entry.getValue();

            for (String tableName : config.getTableNames()) {
                CompletableFuture<TableSyncResult> future = CompletableFuture.supplyAsync(() -> {
                    return syncHeartRateTable(tableName, manufacturer, config, deviceMap, targetDeviceNames);
                }, executorService);
                futures.add(future);
            }
        }

        // Wait for all futures to complete
        CompletableFuture<Void> allFutures = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));

        // Get all results after completion
        List<TableSyncResult> results = new ArrayList<>();
        try {
            // Wait for completion first
            allFutures.get();

            // Then collect all results
            for (CompletableFuture<TableSyncResult> future : futures) {
                TableSyncResult result = future.get();
                results.add(result);
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting sync results", e);
        }

        // Find maximum timestamp across all results
        long overallMaxTimestamp = globalLastTimestamp;
        for (TableSyncResult result : results) {
            if (result.getNewMaxTimestamp() > overallMaxTimestamp) {
                overallMaxTimestamp = result.getNewMaxTimestamp();
                AppLogger.d(TAG, "=== SYNC DEBUG: Updated overallMaxTimestamp to " + overallMaxTimestamp +
                        " from table " + result.getTableName());
            }
        }

        // Update global timestamp ONCE after all tables are done
        if (overallMaxTimestamp > globalLastTimestamp) {
            checkpointManager.updateGlobalSyncTimestamp(overallMaxTimestamp);
            AppLogger.i(TAG, "=== SYNC DEBUG: FINAL - Updated global sync timestamp from " +
                    globalLastTimestamp + " to " + overallMaxTimestamp +
                    " after all " + results.size() + " tables processed");
        } else {
            AppLogger.i(TAG, "=== SYNC DEBUG: FINAL - No timestamp update needed. " +
                    "overallMaxTimestamp=" + overallMaxTimestamp +
                    ", globalLastTimestamp=" + globalLastTimestamp);
        }

        // Log cache stats at end
        AppLogger.d(TAG, TimestampUnitDetector.getCacheStats());

        // Generate summary
        return generateSyncSummary(results);
    }

    private TableSyncResult syncHeartRateTable(String tableName, String manufacturer,
                                               AppConstants.Manufacturers.ManufacturerConfig config,
                                               Map<String, Long> deviceMap,
                                               Map<Long, String> deviceNames) {
        TableSyncResult result = new TableSyncResult(tableName, manufacturer);

        try {
            // Pre-flight checks
            if (!performPreFlightChecks(tableName, config, result)) {
                return result;
            }

            // Get query handler
            QueryHandler queryHandler = createQueryHandler(config);
            if (queryHandler == null) {
                result.addError("Failed to create query handler: " + config.getQueryHandler().getSimpleName());
                return result;
            }

            long globalLastTimestamp = checkpointManager.getGlobalLastSyncTimestamp();

            // No need to build query template here
            result.setQueryUsed("Dynamic multi-table query");
            result.setLastSyncTimestamp(globalLastTimestamp);

            // Pass the config for table names
            executeTableSync(tableName, queryHandler, "", deviceMap, deviceNames,
                    globalLastTimestamp, result, config);

        } catch (Exception e) {
            String errorMsg = "Unexpected error syncing table: " + e.getMessage();
            result.addError(errorMsg);
            AppLogger.e(TAG, errorMsg, e);
        }

        return result;
    }

    private boolean performPreFlightChecks(String tableName,
                                           AppConstants.Manufacturers.ManufacturerConfig config,
                                           TableSyncResult result) {
        // Check table existence
        if (!tableExists(tableName)) {
            result.addError("Table does not exist");
            return false;
        }

        // Validate schema
        SchemaValidationResult schemaResult = validateTableSchema(tableName, config.getRequiredColumns());
        if (!schemaResult.isValid()) {
            result.addError("Schema validation failed: " + schemaResult.getErrorMessage());
            return false;
        }

        result.setSchemaValid(true);
        return true;
    }

    private QueryHandler createQueryHandler(AppConstants.Manufacturers.ManufacturerConfig config) {
        try {
            Class<? extends QueryHandler> handlerClass = config.getQueryHandler();

            // Special case for abstract base class - should never be instantiated directly
            if (handlerClass == BaseQueryHandler.class) {
                AppLogger.w(TAG, "BaseQueryHandler should not be used directly, falling back to StandardQueryHandler");
                return new StandardQueryHandler();
            }

            // Use reflection for flexible instantiation
            return handlerClass.getDeclaredConstructor().newInstance();

        } catch (Exception e) {
            AppLogger.e(TAG, "Error creating query handler: " + config.getQueryHandler().getSimpleName(), e);

            // Graceful fallback to standard handler
            AppLogger.w(TAG, "Falling back to StandardQueryHandler");
            return new StandardQueryHandler();
        }
    }

    private void executeTableSync(String tableName, QueryHandler queryHandler, String query,
                                  Map<String, Long> deviceMap, Map<Long, String> deviceNames,
                                  long globalLastTimestamp, TableSyncResult result,
                                  AppConstants.Manufacturers.ManufacturerConfig config) {

        AppLogger.d(TAG, "=== SYNC DEBUG: Table " + tableName + " starting - Querying with timestamp " +
                globalLastTimestamp + " (" +
                new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss", java.util.Locale.getDefault())
                        .format(new java.util.Date(globalLastTimestamp)) + ") ===");

        // Check database availability
        SQLiteDatabase database = gadgetbridgeSyncSource.getDatabase();
        if (database == null || !database.isOpen()) {
            result.addError("Database is null or closed - cannot execute sync");
            AppLogger.e(TAG, "Database is null or closed when executing sync for table: " + tableName);
            return;
        }

        // Auto-detect timestamp units - use first table for detection
        String[] tableNames = config.getTableNames();
        String timestampDetectionTable = tableNames[0];
        long normalizedQueryTimestamp = TimestampUnitDetector.normalizeForQuery(
                globalLastTimestamp, database, timestampDetectionTable);

        AppLogger.d(TAG, "=== SYNC DEBUG: Normalized timestamp for " + timestampDetectionTable +
                ": " + globalLastTimestamp + "ms -> " + normalizedQueryTimestamp +
                " (" + (globalLastTimestamp != normalizedQueryTimestamp ? "CONVERTED to seconds" : "using milliseconds") + ")");

        Map<Long, List<Object[]>> deviceBatches = new HashMap<>();
        long maxTimestampInThisSync = globalLastTimestamp;

        // ELEGANT SOLUTION: Always use the multi-table query method
        String finalQuery = queryHandler.buildQuery(tableNames, normalizedQueryTimestamp);

        AppLogger.d(TAG, "Sync " + tableNames.length + " table(s) - Query: " + finalQuery);

        try (Cursor cursor = database.rawQuery(finalQuery, null)) {
            if (cursor == null) {
                result.addError("Null cursor returned from query");
                AppLogger.e(TAG, "Null cursor for query: " + finalQuery);
                return;
            }

            result.setTotalRecords(cursor.getCount());
            AppLogger.i(TAG, String.format("Table %s: Found %d records to process", tableName, cursor.getCount()));

            while (cursor.moveToNext()) {
                QueryHandler.HeartRateRecord record = queryHandler.extractRecord(
                        cursor, tableName, deviceMap, gadgetbridgeSyncSource);

                if (!record.isValid) {
                    result.incrementSkippedRecords();
                    continue;
                }

                // Convert timestamp to milliseconds
                long recordTimestampInMillis = TimestampUnitDetector.toMilliseconds(
                        record.timestamp, database, timestampDetectionTable);

                if (recordTimestampInMillis > maxTimestampInThisSync) {
                    maxTimestampInThisSync = recordTimestampInMillis;
                }

                deviceBatches.computeIfAbsent(record.deviceId, k -> new ArrayList<>())
                        .add(new Object[]{record.timestamp, record.deviceId, record.heartRate});

                result.incrementProcessedRecords();

                // Process batch when full
                int totalBatchSize = deviceBatches.values().stream().mapToInt(List::size).sum();
                if (totalBatchSize >= BATCH_SIZE) {
                    AppLogger.v(TAG, String.format("Processing batch of %d records for table %s", totalBatchSize, tableName));
                    int batchInserted = heartRateRepository.processHeartRateBatches(deviceBatches, deviceNames);
                    result.incrementSyncedRecords(batchInserted);
                    deviceBatches.clear();
                }
            }

            // Process final batch
            if (!deviceBatches.isEmpty()) {
                int finalBatchSize = deviceBatches.values().stream().mapToInt(List::size).sum();
                AppLogger.d(TAG, String.format("Processing final batch of %d records for table %s", finalBatchSize, tableName));
                int batchInserted = heartRateRepository.processHeartRateBatches(deviceBatches, deviceNames);
                result.incrementSyncedRecords(batchInserted);
            }

            AppLogger.d(TAG, "=== SYNC DEBUG: Table " + tableName + " completed - " +
                    "Query used timestamp: " + normalizedQueryTimestamp + ", " +
                    "Found max timestamp: " + maxTimestampInThisSync + " (" +
                    (maxTimestampInThisSync > globalLastTimestamp ? "NEWER" : "SAME or OLDER") + ")");

            // Track the max timestamp
            if (maxTimestampInThisSync > globalLastTimestamp) {
                result.setNewMaxTimestamp(maxTimestampInThisSync);
                AppLogger.i(TAG, String.format("Table %s: Tracked max timestamp %d (greater than %d)",
                        tableName, maxTimestampInThisSync, globalLastTimestamp));
            } else {
                result.setNewMaxTimestamp(globalLastTimestamp);
            }

            AppLogger.i(TAG, String.format("Table %s sync completed: %d processed, %d synced, %d skipped",
                    tableName, result.getProcessedRecords(), result.getSyncedRecords(), result.getSkippedRecords()));

        } catch (Exception e) {
            String errorMsg = "Error during table sync execution: " + e.getMessage();
            result.addError(errorMsg);
            AppLogger.e(TAG, errorMsg, e);
            AppLogger.e(TAG, "Problematic query: " + finalQuery);
            result.setNewMaxTimestamp(globalLastTimestamp);
        }
    }

    // Schema validation
    private SchemaValidationResult validateTableSchema(String tableName, String[] requiredColumns) {
        try {
            SQLiteDatabase database = gadgetbridgeSyncSource.getDatabase();
            if (database == null || !database.isOpen()) {
                return SchemaValidationResult.invalid("Database is not available");
            }

            String sql = "PRAGMA table_info(" + tableName + ")";
            try (Cursor cursor = database.rawQuery(sql, null)) {
                if (cursor == null) {
                    return SchemaValidationResult.invalid("Cannot read table schema");
                }

                Map<String, String> columnTypes = new HashMap<>();
                while (cursor.moveToNext()) {
                    String columnName = cursor.getString(1);
                    String columnType = cursor.getString(2);
                    columnTypes.put(columnName.toUpperCase(), columnType);
                }

                // Check required columns
                for (String requiredColumn : requiredColumns) {
                    if (!columnTypes.containsKey(requiredColumn.toUpperCase())) {
                        return SchemaValidationResult.invalid("Missing required column: " + requiredColumn);
                    }
                }

                return SchemaValidationResult.valid();
            }
        } catch (Exception e) {
            return SchemaValidationResult.invalid("Error validating schema: " + e.getMessage());
        }
    }

    private boolean tableExists(String tableName) {
        // if database isn't ready, return false
        if (!gadgetbridgeSyncSource.isSourceInitialized()) {
            AppLogger.w(TAG, "Gadgetbridge sync source not initialized for table check: " + tableName);
            return false;
        }
        try {
            // Check if database is available
            SQLiteDatabase database = gadgetbridgeSyncSource.getDatabase();
            if (database == null) {
                AppLogger.e(TAG, "Database is null when checking table existence: " + tableName);
                return false;
            }

            // Verify database is open
            if (!database.isOpen()) {
                AppLogger.e(TAG, "Database is closed when checking table existence: " + tableName);
                return false;
            }

            String sql = "SELECT name FROM sqlite_master WHERE type='table' AND name=?";
            try (Cursor cursor = database.rawQuery(sql, new String[]{tableName})) {
                return cursor != null && cursor.getCount() > 0;
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error checking table existence: " + tableName, e);
            return false;
        }
    }

    private SyncSummary generateSyncSummary(List<TableSyncResult> results) {
        SyncSummary summary = new SyncSummary();

        for (TableSyncResult result : results) {
            summary.addTableResult(result);

            // Log each table result
            if (result.hasErrors()) {
                AppLogger.w(TAG, String.format("Table %s: %d processed, %d synced, %d skipped, ERRORS: %s",
                        result.getTableName(), result.getProcessedRecords(),
                        result.getSyncedRecords(), result.getSkippedRecords(),
                        String.join(", ", result.getErrors())));
            } else {
                AppLogger.i(TAG, String.format("Table %s: %d processed, %d synced, %d skipped",
                        result.getTableName(), result.getProcessedRecords(),
                        result.getSyncedRecords(), result.getSkippedRecords()));
            }
        }

        AppLogger.i(TAG, summary.generateSummaryReport());
        return summary;
    }

    // Result classes
    public static class TableSyncResult {
        private final String tableName;
        private final String manufacturer;
        private final List<String> errors = new ArrayList<>();
        private String queryUsed;
        private long lastSyncTimestamp;
        private long newMaxTimestamp;  // Track max timestamp found in this table
        private boolean schemaValid = false;
        private int totalRecords = 0;
        private int processedRecords = 0;
        private int syncedRecords = 0;
        private int skippedRecords = 0;

        public TableSyncResult(String tableName, String manufacturer) {
            this.tableName = tableName;
            this.manufacturer = manufacturer;
        }

        // Getters and setters
        public String getTableName() { return tableName; }
        public String getManufacturer() { return manufacturer; }
        public List<String> getErrors() { return errors; }
        public boolean hasErrors() { return !errors.isEmpty(); }
        public void setQueryUsed(String queryUsed) { this.queryUsed = queryUsed; }
        public long getLastSyncTimestamp() { return lastSyncTimestamp; }
        public void setLastSyncTimestamp(long lastSyncTimestamp) { this.lastSyncTimestamp = lastSyncTimestamp; }
        public long getNewMaxTimestamp() { return newMaxTimestamp; }  // ADD THIS GETTER
        public void setNewMaxTimestamp(long newMaxTimestamp) { this.newMaxTimestamp = newMaxTimestamp; }
        public void setSchemaValid(boolean schemaValid) { this.schemaValid = schemaValid; }
        public void setTotalRecords(int totalRecords) { this.totalRecords = totalRecords; }
        public int getProcessedRecords() { return processedRecords; }
        public void incrementProcessedRecords() { this.processedRecords++; }
        public int getSyncedRecords() { return syncedRecords; }
        public void incrementSyncedRecords(int count) { this.syncedRecords += count; }
        public int getSkippedRecords() { return skippedRecords; }
        public void incrementSkippedRecords() { this.skippedRecords++; }
        public void addError(String error) { this.errors.add(error); }
    }

    public static class SyncSummary {
        private final List<TableSyncResult> tableResults = new ArrayList<>();
        private int totalTables = 0;
        private int successfulTables = 0;
        private int failedTables = 0;
        private int totalProcessed = 0;
        private int totalSynced = 0;
        private int totalSkipped = 0;

        public void addTableResult(TableSyncResult result) {
            tableResults.add(result);
            totalTables++;

            if (result.hasErrors()) {
                failedTables++;
            } else {
                successfulTables++;
            }

            totalProcessed += result.getProcessedRecords();
            totalSynced += result.getSyncedRecords();
            totalSkipped += result.getSkippedRecords();
        }

        public String generateSummaryReport() {
            return String.format(
                    "Sync Summary: %d tables (%d successful, %d failed), %d records processed, %d synced, %d skipped",
                    totalTables, successfulTables, failedTables, totalProcessed, totalSynced, totalSkipped
            );
        }
    }

    private static class SchemaValidationResult {
        private final boolean valid;
        private final String errorMessage;

        private SchemaValidationResult(boolean valid, String errorMessage) {
            this.valid = valid;
            this.errorMessage = errorMessage;
        }

        public static SchemaValidationResult valid() {
            return new SchemaValidationResult(true, null);
        }

        public static SchemaValidationResult invalid(String errorMessage) {
            return new SchemaValidationResult(false, errorMessage);
        }

        public boolean isValid() { return valid; }
        public String getErrorMessage() { return errorMessage; }
    }
}