package org.ojrandom.paiesque.data.repositories;

import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.util.Log;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class HeartRateRepository {
    private static final String TAG = "HeartRateRepository";
    private final SQLiteDatabase database;

    // Timestamp threshold: if greater than this, it's likely milliseconds that need conversion to seconds
    private static final long MILLIS_THRESHOLD = 10000000000L; // 10 billion

    // Heart rate validation
    private static final int MIN_VALID_HR = 30;
    private static final int MAX_VALID_HR = 250;

    public HeartRateRepository(SQLiteDatabase database) {
        this.database = database;
    }

    public synchronized int processHeartRateBatches(Map<Long, List<Object[]>> deviceBatches, Map<Long, String> deviceNames) {
        int totalInserted = 0;
        int invalidHRCount = 0;

        // Simple log: what we're processing
        StringBuilder batchInfo = new StringBuilder("Processing batches: ");
        for (Map.Entry<Long, List<Object[]>> entry : deviceBatches.entrySet()) {
            if (batchInfo.length() > "Processing batches: ".length()) {
                batchInfo.append(", ");
            }
            Long deviceId = entry.getKey();
            String deviceName = deviceNames.getOrDefault(deviceId, "Device-" + deviceId);
            batchInfo.append(deviceName).append(" (").append(deviceId).append("): ").append(entry.getValue().size()).append(" records");
        }
        AppLogger.d(TAG, batchInfo.toString());

        database.beginTransaction();
        SQLiteStatement statement = null;
        try {
            String sql = "INSERT OR REPLACE INTO " + AppConstants.PaiesqueTables.HEART_RATES + " (" +
                    AppConstants.Columns.TIMESTAMP + ", " +
                    AppConstants.Columns.DEVICE_ID + ", " +
                    AppConstants.Columns.HEART_RATE + ", " +
                    "LOCAL_DATE, " +
                    "MINUTE_OF_DAY" +
                    ") VALUES (?, ?, ?, ?, ?)";

            statement = database.compileStatement(sql);

            for (Map.Entry<Long, List<Object[]>> entry : deviceBatches.entrySet()) {
                Long deviceId = entry.getKey();
                String deviceName = deviceNames.getOrDefault(deviceId, "Device-" + deviceId);
                List<Object[]> records = entry.getValue();
                int deviceRecordCount = 0;
                int deviceInvalidCount = 0;

                for (Object[] record : records) {
                    long timestamp = (Long) record[0];
                    int heartRate = (Integer) record[2];

                    // Skip invalid heart rates
                    if (heartRate < MIN_VALID_HR || heartRate > MAX_VALID_HR) {
                        invalidHRCount++;
                        deviceInvalidCount++;
                        continue;
                    }

                    // NORMALIZE: Convert milliseconds to seconds, keep seconds as-is
                    long normalizedTimestamp = normalizeToSeconds(timestamp);

                    // Calculate date fields using normalized timestamp (now in seconds)
                    int localDate = convertSecondsToLocalDate(normalizedTimestamp);
                    int minuteOfDay = convertSecondsToMinuteOfDay(normalizedTimestamp);

                    // Store the normalized timestamp (SECONDS)
                    statement.bindLong(1, normalizedTimestamp);
                    statement.bindLong(2, deviceId);
                    statement.bindLong(3, heartRate);
                    statement.bindLong(4, localDate);
                    statement.bindLong(5, minuteOfDay);

                    statement.execute();
                    statement.clearBindings();
                    totalInserted++;
                    deviceRecordCount++;
                }

                // One log per device with final count - now with device name
                if (deviceInvalidCount > 0) {
                    AppLogger.d(TAG, String.format("Device %s (%d): %d inserted, %d invalid skipped",
                            deviceName, deviceId, deviceRecordCount, deviceInvalidCount));
                } else {
                    AppLogger.i(TAG, String.format("Device %s (%d): %d inserted",
                            deviceName, deviceId, deviceRecordCount));
                }
            }
            database.setTransactionSuccessful();

            // One final summary log
            AppLogger.v(TAG, String.format("Completed: %d total records inserted, %d invalid skipped",
                    totalInserted, invalidHRCount));

        } catch (Exception e) {
            AppLogger.e(TAG, "Error in batch processing: " + e.getMessage(), e);
            try {
                database.endTransaction(); // This rolls back if not successful
            } catch (Exception inner) {
                AppLogger.e(TAG, "Error ending transaction", inner);
            }
            throw e; // Re-throw after cleanup
        } finally {
            if (statement != null) {
                try {
                    statement.close();
                } catch (Exception e) {
                    AppLogger.e(TAG, "Error closing statement", e);
                }
            }
            // ENSURE transaction is always ended:
            try {
                if (database.inTransaction()) {
                    database.endTransaction();
                }
            } catch (Exception e) {
                AppLogger.e(TAG, "Error in transaction cleanup", e);
            }
        }
        return totalInserted;
    }

    /**
     * Normalize timestamp: ensure ALL timestamps are stored as SECONDS
     */
    private long normalizeToSeconds(long timestamp) {
        // If timestamp is large (likely milliseconds), convert to seconds
        // If timestamp is small (likely seconds), use as-is
        if (timestamp > MILLIS_THRESHOLD) {
            // This is likely milliseconds - convert to seconds
            return timestamp / 1000;
        } else {
            // This is likely already seconds - use as-is
            return timestamp;
        }
    }

    /**
     * Convert seconds timestamp to local date in YYYYMMDD format
     */
    private int convertSecondsToLocalDate(long timestampSeconds) {
        try {
            Instant instant = Instant.ofEpochSecond(timestampSeconds);
            LocalDate localDate = instant.atZone(ZoneId.systemDefault()).toLocalDate();
            return localDate.getYear() * 10000 + localDate.getMonthValue() * 100 + localDate.getDayOfMonth();
        } catch (Exception e) {
            AppLogger.e(TAG, "Error converting seconds to local date: " + timestampSeconds, e);
            LocalDate today = LocalDate.now();
            return today.getYear() * 10000 + today.getMonthValue() * 100 + today.getDayOfMonth();
        }
    }

    /**
     * Convert seconds timestamp to minute of day (0-1439)
     */
    private int convertSecondsToMinuteOfDay(long timestampSeconds) {
        try {
            Instant instant = Instant.ofEpochSecond(timestampSeconds);
            LocalTime localTime = instant.atZone(ZoneId.systemDefault()).toLocalTime();
            return localTime.getHour() * 60 + localTime.getMinute();
        } catch (Exception e) {
            AppLogger.e(TAG, "Error converting seconds to minute of day: " + timestampSeconds, e);
            LocalTime now = LocalTime.now();
            return now.getHour() * 60 + now.getMinute();
        }
    }
}