package org.ojrandom.paiesque.data;

import android.content.Context;

import org.ojrandom.paiesque.logging.AppLogger;

public class SettingsManager {
    private static final String TAG = "SettingsManager";
    private final Context context;
    private final PaiesqueDBManager dbManager;

    public SettingsManager(Context context, PaiesqueDBManager dbManager) {
        this.context = context;
        this.dbManager = dbManager;
    }

    // Device Selection Methods
    public synchronized String getDeviceSelection() {
        return getStringSetting(AppConstants.Settings.KEY_DEVICE_SELECTION, null,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized void setDeviceSelection(String deviceId) {
        setStringSetting(AppConstants.Settings.KEY_DEVICE_SELECTION, deviceId,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized int getSelectedDeviceId() {
        return getIntSetting(AppConstants.Columns.DEVICE_ID, -1
        );
    }

    public synchronized void setSelectedDeviceId(int deviceId) {
        setIntSetting(AppConstants.Columns.DEVICE_ID, deviceId,
                AppConstants.SettingsCategories.GENERAL);
    }

    // Sleep Settings Methods
    public synchronized int getSleepStartHour() {
        return getIntSetting(AppConstants.SleepSettings.KEY_SLEEP_START + "_hour", 0
        );
    }

    public synchronized int getSleepStartMinute() {
        return getIntSetting(AppConstants.SleepSettings.KEY_SLEEP_START + "_minute", 0
        );
    }

    public synchronized int getSleepEndHour() {
        return getIntSetting(AppConstants.SleepSettings.KEY_SLEEP_END + "_hour", 6
        );
    }

    public synchronized int getSleepEndMinute() {
        return getIntSetting(AppConstants.SleepSettings.KEY_SLEEP_END + "_minute", 0
        );
    }

    public synchronized void setSleepTimes(int startHour, int startMinute, int endHour, int endMinute) {
        setIntSetting(AppConstants.SleepSettings.KEY_SLEEP_START + "_hour", startHour,
                AppConstants.SettingsCategories.SLEEP);
        setIntSetting(AppConstants.SleepSettings.KEY_SLEEP_START + "_minute", startMinute,
                AppConstants.SettingsCategories.SLEEP);
        setIntSetting(AppConstants.SleepSettings.KEY_SLEEP_END + "_hour", endHour,
                AppConstants.SettingsCategories.SLEEP);
        setIntSetting(AppConstants.SleepSettings.KEY_SLEEP_END + "_minute", endMinute,
                AppConstants.SettingsCategories.SLEEP);
        markSleepSettingsAsChanged();
    }

    // Overloaded method for string format compatibility
    public synchronized void setSleepTimes(String sleepStart, String sleepEnd) {
        int startHour = parseHourFromTimeString(sleepStart);
        int startMinute = parseMinuteFromTimeString(sleepStart);
        int endHour = parseHourFromTimeString(sleepEnd);
        int endMinute = parseMinuteFromTimeString(sleepEnd);

        setSleepTimes(startHour, startMinute, endHour, endMinute);
    }

    // PAI Settings Methods
    public synchronized int getRHR() {
        return getIntSetting(AppConstants.Settings.KEY_RHR, 0
        );
    }

    public synchronized void setRHR(int rhr) {
        setIntSetting(AppConstants.Settings.KEY_RHR, rhr,
                AppConstants.SettingsCategories.PAI);
        markSettingsAsChanged();
    }

    public synchronized int getMaxHR() {
        return getIntSetting(AppConstants.Settings.KEY_MAX_HR, 0
        );
    }

    public synchronized void setMaxHR(int maxHR) {
        setIntSetting(AppConstants.Settings.KEY_MAX_HR, maxHR,
                AppConstants.SettingsCategories.PAI);
        markSettingsAsChanged();
    }

    public synchronized boolean isManualMaxHR() {
        return getBooleanSetting("manual_max_hr", false,
                AppConstants.SettingsCategories.PAI);
    }

    public synchronized void setManualMaxHR(boolean manualMaxHR) {
        setBooleanSetting("manual_max_hr", manualMaxHR,
                AppConstants.SettingsCategories.PAI);
    }

    public synchronized int getAge() {
        return getIntSetting("age", 0);
    }

    public synchronized void setAge(int age) {
        setIntSetting("age", age, AppConstants.SettingsCategories.PAI);
    }

    public synchronized String getGender() {
        return getStringSetting("gender", "male", AppConstants.SettingsCategories.PAI);
    }

    public synchronized void setGender(String gender) {
        setStringSetting("gender", gender, AppConstants.SettingsCategories.PAI);
    }

    public synchronized String getMaxHRFormula() {
        return getStringSetting("max_hr_formula", "standard", AppConstants.SettingsCategories.PAI);
    }

    public synchronized void setMaxHRFormula(String formula) {
        setStringSetting("max_hr_formula", formula, AppConstants.SettingsCategories.PAI);
    }

    // Settings Changed Flags
    public synchronized boolean hasSettingsChanged() {
        return getBooleanSetting(AppConstants.Settings.KEY_SETTINGS_CHANGED, false,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized void markSettingsAsChanged() {
        setBooleanSetting(AppConstants.Settings.KEY_SETTINGS_CHANGED, true,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized void markSettingsAsProcessed() {
        setBooleanSetting(AppConstants.Settings.KEY_SETTINGS_CHANGED, false,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized boolean hasSleepSettingsChanged() {
        return getBooleanSetting(AppConstants.SleepSettings.KEY_SLEEP_SETTINGS_CHANGED, false,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized void markSleepSettingsAsChanged() {
        setBooleanSetting(AppConstants.SleepSettings.KEY_SLEEP_SETTINGS_CHANGED, true,
                AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized void markSleepSettingsAsProcessed() {
        setBooleanSetting(AppConstants.SleepSettings.KEY_SLEEP_SETTINGS_CHANGED, false,
                AppConstants.SettingsCategories.GENERAL);
    }

    // Generic database methods
    private int getIntSetting(String key, int defaultValue) {
        try {
            String value = dbManager.getSetting(key);
            if (value != null) {
                // validation:
                value = value.trim();
                if (!value.isEmpty()) {
                    try {
                        return Integer.parseInt(value);
                    } catch (NumberFormatException e) {
                        AppLogger.w(TAG, "Invalid integer format for setting " + key + ": " + value);
                        // FALLBACK: Try to extract number from string
                        String numeric = value.replaceAll("[^0-9-]", "");
                        if (!numeric.isEmpty()) {
                            try {
                                return Integer.parseInt(numeric);
                            } catch (NumberFormatException ignored) {}
                        }
                    }
                }
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading int setting: " + key, e);
        }
        return defaultValue;
    }

    private String getStringSetting(String key, String defaultValue, String category) {
        try {
            String value = dbManager.getSetting(key);
            return value != null ? value : defaultValue;
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading string setting: " + key, e);
            return defaultValue;
        }
    }

    private boolean getBooleanSetting(String key, boolean defaultValue, String category) {
        try {
            String value = dbManager.getSetting(key);
            if (value != null) {
                return Boolean.parseBoolean(value);
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading boolean setting: " + key, e);
        }
        return defaultValue;
    }

    // Public methods without category parameter
    public synchronized void setIntSetting(String key, int value) {
        setIntSetting(key, value, AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized void setStringSetting(String key, String value) {
        setStringSetting(key, value, AppConstants.SettingsCategories.GENERAL);
    }

    public synchronized String getStringSetting(String key, String defaultValue) {
        return getStringSetting(key, defaultValue, AppConstants.SettingsCategories.GENERAL);
    }

    private void setIntSetting(String key, int value, String category) {
        dbManager.setSetting(key, String.valueOf(value),
                AppConstants.SettingTypes.INT, category);
    }

    private void setStringSetting(String key, String value, String category) {
        dbManager.setSetting(key, value, AppConstants.SettingTypes.STRING, category);
    }

    private void setBooleanSetting(String key, boolean value, String category) {
        dbManager.setSetting(key, String.valueOf(value),
                AppConstants.SettingTypes.BOOLEAN, category);
    }

    // Utility methods for time parsing
    private int parseHourFromTimeString(String timeString) {
        try {
            if (timeString != null && timeString.contains(":")) {
                String[] parts = timeString.split(":");
                return Integer.parseInt(parts[0]);
            }
        } catch (NumberFormatException e) {
            AppLogger.e(TAG, "Error parsing hour from: " + timeString, e);
        }
        return 22; // Default 10 PM
    }

    private int parseMinuteFromTimeString(String timeString) {
        try {
            if (timeString != null && timeString.contains(":")) {
                String[] parts = timeString.split(":");
                return Integer.parseInt(parts[1]);
            }
        } catch (NumberFormatException e) {
            AppLogger.e(TAG, "Error parsing minute from: " + timeString, e);
        }
        return 0;
    }

    public synchronized boolean isManualRHR() {
        return getBooleanSetting("manual_rhr", true, AppConstants.SettingsCategories.PAI);
    }

    public synchronized void setManualRHR(boolean manualRHR) {
        setBooleanSetting("manual_rhr", manualRHR, AppConstants.SettingsCategories.PAI);
    }

    public synchronized String getMaxHrMode() {
        return getStringSetting("max_hr_mode", "formula", AppConstants.SettingsCategories.PAI);
    }

    public synchronized void setMaxHrMode(String maxHrMode) {
        setStringSetting("max_hr_mode", maxHrMode, AppConstants.SettingsCategories.PAI);
    }
}