package org.ojrandom.paiesque.data;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;

import org.ojrandom.paiesque.data.database.DatabaseInitializer;
import org.ojrandom.paiesque.data.repositories.DatabaseStatsRepository;
import org.ojrandom.paiesque.data.repositories.DeviceRepository;
import org.ojrandom.paiesque.data.repositories.HeartRateRepository;
import org.ojrandom.paiesque.data.repositories.PaiRepository;
import org.ojrandom.paiesque.data.repositories.RhrRepository;
import org.ojrandom.paiesque.data.repositories.TimestampRepository;
import org.ojrandom.paiesque.data.sync.HeartRateSyncService;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.pai.PAIesqueCalculator;
import org.ojrandom.paiesque.rhr.RhrResult;
import org.ojrandom.paiesque.ui.ZoneDayData;

import java.io.File;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import android.content.ContentValues;

public class PaiesqueDBManager {
    private static final String TAG = "PaiesqueDBManager";
    private final GadgetbridgeSyncSource gadgetbridgeSyncSource; // Renamed field
    private SQLiteDatabase targetDb;
    private final String targetDbPath;
    private SyncCheckpointManager checkpointManager;
    private ExecutorService executorService;

    // Repository instances
    private final DatabaseInitializer databaseInitializer;
    private DeviceRepository deviceRepository;
    private HeartRateRepository heartRateRepository;
    private PaiRepository paiRepository;
    private RhrRepository rhrRepository;
    private TimestampRepository timestampRepository;
    private DatabaseStatsRepository databaseStatsRepository;
    private HeartRateSyncService heartRateSyncService;


    public PaiesqueDBManager(Context context, GadgetbridgeSyncSource gadgetbridgeSyncSource) {
        AppLogger.d(TAG, "PaiesqueDBManager constructor called - instance: " + this.hashCode());
        this.gadgetbridgeSyncSource = gadgetbridgeSyncSource; // Use shared instance
        this.targetDbPath = context.getDatabasePath(AppConstants.KEY_PAIESQUE_DB_NAME).getAbsolutePath();
        this.executorService = Executors.newFixedThreadPool(3);
        this.databaseInitializer = new DatabaseInitializer();
    }

    public void close() {
        gadgetbridgeSyncSource.closeSource(); // Updated method call
        if (targetDb != null && targetDb.isOpen()) {
            targetDb.close();
        }
        if (executorService != null && !executorService.isShutdown()) {
            executorService.shutdown();
        }
    }

    private void initializeRepositories() {
        if (targetDb == null || !targetDb.isOpen()) {
            AppLogger.e(TAG, "Cannot initialize repositories - target database is not open");
            return;
        }

        // Initialize ALL repositories
        if (deviceRepository == null) {
            deviceRepository = new DeviceRepository(targetDb);
        }
        if (heartRateRepository == null) {
            heartRateRepository = new HeartRateRepository(targetDb);
        }
        if (paiRepository == null) {
            paiRepository = new PaiRepository(targetDb);
        }
        if (rhrRepository == null) {
            rhrRepository = new RhrRepository(targetDb);
        }
        if (timestampRepository == null) {
            timestampRepository = new TimestampRepository(targetDb);
        }
        if (databaseStatsRepository == null) {
            databaseStatsRepository = new DatabaseStatsRepository(targetDb);
        }
        if (heartRateSyncService == null && checkpointManager != null) {
            heartRateSyncService = new HeartRateSyncService(
                    gadgetbridgeSyncSource, heartRateRepository, checkpointManager, executorService);
        }

        AppLogger.d(TAG, "All repositories initialized successfully");
    }

    private synchronized void createOrOpenTargetDatabase() {
        try {
            targetDb = SQLiteDatabase.openOrCreateDatabase(targetDbPath, null);
            if (targetDb != null && targetDb.isOpen()) {
                databaseInitializer.removeStrategyUsedColumn(targetDb);
                databaseInitializer.createTablesIfNotExist(targetDb);
                databaseInitializer.createIndexes(targetDb);
                this.checkpointManager = new SyncCheckpointManager(targetDb);
                initializeRepositories(); // This MUST be called
                AppLogger.d(TAG, "Target database opened/created successfully: " + targetDbPath);
            } else {
                AppLogger.e(TAG, "Failed to open/create target database");
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error opening/creating target database", e);
            targetDb = null;
        }
    }

    private boolean isDatabaseReady() {
        if (targetDb == null || !targetDb.isOpen()) {
            createOrOpenTargetDatabase();
        }
        // SIMPLE CHECK: Just verify database is open and repositories exist
        return targetDb != null && targetDb.isOpen() &&
                deviceRepository != null && heartRateRepository != null;
    }

    private boolean ensureDatabaseForSettings() {
        if (targetDb == null || !targetDb.isOpen()) {
            createOrOpenTargetDatabase();
        }
        return targetDb != null && targetDb.isOpen();
    }

    /**
     * Store daily PAI scores with zone breakdown
     */
    public void storeDailyPAIScoresWithZones(Map<LocalDate, PAIesqueCalculator.DailyPAIResult> dailyPAIResults, int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for storing PAI scores with zones");
            return;
        }
        if (paiRepository != null) {
            paiRepository.storeDailyPAIScoresWithZones(dailyPAIResults, deviceId);
        } else {
            AppLogger.e(TAG, "PAI Repository not initialized");
        }
    }

    /**
     * Loads ALL historical PAI scores for a device
     */
    public Map<LocalDate, Double> loadAllDailyPAIScores(int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for loading PAI scores");
            return new HashMap<>();
        }

        Map<LocalDate, Double> dailyPAI = new HashMap<>();

        // Use correct table name and column names
        String sql = "SELECT LOCAL_DATE, PAI_SCORE FROM " + AppConstants.PaiesqueTables.PAI_SCORES + " WHERE DEVICE_ID = ? ORDER BY LOCAL_DATE";

        try (Cursor cursor = targetDb.rawQuery(sql, new String[]{String.valueOf(deviceId)})) {
            if (cursor == null) {
                AppLogger.e(TAG, "Cursor is null for PAI scores query");
                return dailyPAI;
            }

            while (cursor.moveToNext()) {
                try {
                    // Get the date as integer in YYYYMMDD format
                    int dateInt = cursor.getInt(cursor.getColumnIndexOrThrow("LOCAL_DATE"));
                    double paiScore = cursor.getDouble(cursor.getColumnIndexOrThrow("PAI_SCORE"));

                    // Parse integer date (YYYYMMDD) to LocalDate
                    LocalDate date = intToLocalDate(dateInt);
                    if (date != null) {
                        dailyPAI.put(date, paiScore);
                    }
                } catch (Exception e) {
                    AppLogger.w(TAG, "Error parsing PAI score from database for date integer: " +
                            cursor.getInt(cursor.getColumnIndexOrThrow("LOCAL_DATE")), e);
                }
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error loading PAI scores from database", e);
        }

        return dailyPAI;
    }

    public TreeMap<LocalDate, Integer> loadHistoricalRHR(int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for loading historical RHR");
            return new TreeMap<>();
        }
        if (rhrRepository != null) {
            return rhrRepository.loadHistoricalRHR(deviceId);
        } else {
            AppLogger.e(TAG, "RHR Repository not initialized");
            return new TreeMap<>();
        }
    }

    public boolean isPaiesqueDbAvailable() {
        if (!isDatabaseReady()) {
            return false;
        }
        if (databaseStatsRepository != null) {
            return databaseStatsRepository.isPaiesqueDbAvailable();
        } else {
            AppLogger.e(TAG, "DatabaseStats Repository not initialized");
            return false;
        }
    }

    // ===== Sync Methods =====
    public synchronized void syncDatabases() {
        long startTime = System.currentTimeMillis();
        long startMemory = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();

        AppLogger.i(TAG, "Starting optimized database sync...");

        try {
            createOrOpenTargetDatabase();
            if (targetDb != null && targetDb.isOpen()) {
                // Only sync heart rate data
                syncAllHeartRateTablesOptimized();
            } else {
                AppLogger.e(TAG, "Cannot sync databases - target database not available");
            }
        } finally {
            long endTime = System.currentTimeMillis();
            long endMemory = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();

            AppLogger.i(TAG, String.format(
                    "Sync completed in %.2f seconds. Memory delta: %.2f MB",
                    (endTime - startTime) / 1000.0,
                    (endMemory - startMemory) / (1024.0 * 1024.0)
            ));
        }
    }

    private void syncAllHeartRateTablesOptimized() {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for heart rate sync");
            return;
        }

        if (deviceRepository != null && heartRateSyncService != null) {
            // Use enhanced device discovery
            DeviceDiscoveryService.DiscoveryResult discoveryResult =
                    DeviceDiscoveryService.discoverAndMapDevices(gadgetbridgeSyncSource, targetDb);

            // Pass the entire DiscoveryResult instead of just deviceMap
            HeartRateSyncService.SyncSummary syncSummary =
                    heartRateSyncService.syncAllHeartRateTablesOptimized(discoveryResult);

            AppLogger.i(TAG, syncSummary.generateSummaryReport());

        } else {
            AppLogger.e(TAG, "Repositories not initialized for heart rate sync");
        }
    }

    /**
     * Gets the last processed local date for PAI calculations
     */
    public int getLastProcessedDateForPAI(int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for getting last processed PAI date");
            return 0;
        }
        if (timestampRepository != null) {
            return timestampRepository.getLastProcessedDateForPAI(deviceId);
        } else {
            AppLogger.e(TAG, "Timestamp Repository not initialized");
            return 0;
        }
    }

    /**
     * Updates the last processed local date for PAI calculations
     */
    public void updateLastProcessedDateForPAI(int deviceId, int localDate) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for updating last processed PAI date");
            return;
        }
        if (timestampRepository != null) {
            timestampRepository.updateLastProcessedDateForPAI(deviceId, localDate);
        } else {
            AppLogger.e(TAG, "Timestamp Repository not initialized");
        }
    }

    /**
     * Gets the last processed local date for RHR calculations
     */
    public int getLastProcessedDateForRHR(int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for getting last processed RHR date");
            return 0;
        }
        if (timestampRepository != null) {
            return timestampRepository.getLastProcessedDateForRHR(deviceId);
        } else {
            AppLogger.e(TAG, "Timestamp Repository not initialized");
            return 0;
        }
    }

    /**
     * Updates the last processed local date for RHR calculations
     */
    public void updateLastProcessedDateForRHR(int deviceId, int localDate) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for updating last processed RHR date");
            return;
        }
        if (timestampRepository != null) {
            timestampRepository.updateLastProcessedDateForRHR(deviceId, localDate);
        } else {
            AppLogger.e(TAG, "Timestamp Repository not initialized");
        }
    }

    public void storeHistoricalRollingPAI(List<Map<String, String>> rollingPAI, int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for storing historical rolling PAI");
            return;
        }
        if (paiRepository != null) {
            paiRepository.storeHistoricalRollingPAI(rollingPAI, deviceId);
        } else {
            AppLogger.e(TAG, "PAI Repository not initialized");
        }
    }

    public synchronized Map<LocalDate, DailyHeartRates> getDailyHeartRates(int deviceId, int sinceLocalDate, int minHeartRate) {
        Map<LocalDate, DailyHeartRates> result = new TreeMap<>();

        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for getting daily heart rates");
            return result;
        }

        AppLogger.d(TAG, "Querying heart rates for device: " + deviceId +
                ", since local date: " + sinceLocalDate +
                ", min HR: " + minHeartRate);

        // Build query based on deviceId WITH MIN HR FILTER
        String sql;
        String[] selectionArgs;

        if (deviceId == -1) {
            // Use AVERAGE heart rate for duplicate timestamps
            sql = "SELECT MAX(LOCAL_DATE) as LOCAL_DATE, TIMESTAMP, " +
                    "    MAX(HEART_RATE) as HEART_RATE, " +
                    "    MAX(MINUTE_OF_DAY) as MINUTE_OF_DAY " +
                    "FROM " + AppConstants.PaiesqueTables.HEART_RATES + " " +
                    "WHERE LOCAL_DATE >= ? " +
                    "AND HEART_RATE >= ? " +
                    "GROUP BY TIMESTAMP " +  // Group duplicates
                    "ORDER BY TIMESTAMP";
            selectionArgs = new String[]{
                    String.valueOf(sinceLocalDate),
                    String.valueOf(minHeartRate)
            };
        } else {
            // Query for specific device (unchanged)
            sql = "SELECT LOCAL_DATE, TIMESTAMP, HEART_RATE, MINUTE_OF_DAY " +
                    "FROM " + AppConstants.PaiesqueTables.HEART_RATES + " " +
                    "WHERE DEVICE_ID = ? AND LOCAL_DATE >= ? " +
                    "AND HEART_RATE >= ? " +
                    "ORDER BY LOCAL_DATE, TIMESTAMP";
            selectionArgs = new String[]{
                    String.valueOf(deviceId),
                    String.valueOf(sinceLocalDate),
                    String.valueOf(minHeartRate)
            };
        }

        try (Cursor cursor = targetDb.rawQuery(sql, selectionArgs)) {
            if (cursor == null) {
                AppLogger.e(TAG, "Cursor is null for heart rates query");
                return result;
            }

            AppLogger.d(TAG, "Cursor count with HR filter: " + cursor.getCount());

            int rowCount = 0;
            LocalDate currentDate = null;
            List<DailyHeartRates.HeartRateRecord> currentRecords = null;

            while (cursor.moveToNext()) {
                rowCount++;
                try {
                    int localDateInt = cursor.getInt(0);
                    LocalDate date = intToLocalDate(localDateInt);
                    if (date == null) {
                        AppLogger.w(TAG, "Skipping row with invalid date: " + localDateInt);
                        continue;
                    }

                    long timestamp = cursor.getLong(1);
                    int heartRate = cursor.getInt(2);
                    int minuteOfDay = cursor.getInt(3);

                    if (rowCount <= 5) {
                        AppLogger.d(TAG, "Row " + rowCount + ": date=" + localDateInt + "(" + date + "), " +
                                "timestamp=" + timestamp + ", HR=" + heartRate + ", minute=" + minuteOfDay);
                    }

                    if (!date.equals(currentDate)) {
                        if (currentDate != null && currentRecords != null) {
                            result.put(currentDate, new DailyHeartRates(currentRecords));
                            AppLogger.d(TAG, "Completed day: " + currentDate + " with " + currentRecords.size() + " measurements");
                        }
                        currentDate = date;
                        currentRecords = new ArrayList<>();
                        AppLogger.d(TAG, "Starting new day: " + date);
                    }

                    currentRecords.add(new DailyHeartRates.HeartRateRecord(timestamp, heartRate, minuteOfDay));
                } catch (Exception e) {
                    AppLogger.w(TAG, "Error parsing heart rate record", e);
                }
            }

            if (currentDate != null && currentRecords != null && !currentRecords.isEmpty()) {
                result.put(currentDate, new DailyHeartRates(currentRecords));
                AppLogger.d(TAG, "Completed last day: " + currentDate + " with " + currentRecords.size() + " measurements");
            }

            AppLogger.d(TAG, "Total rows processed with HR filter: " + rowCount);

        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading daily heart rates", e);
        }

        AppLogger.d(TAG, "Retrieved " + result.size() + " days of heart rate data (min HR: " + minHeartRate + ")");
        return result;
    }

    /**
     * EXISTING METHOD: Keep for backward compatibility
     */
    public Map<LocalDate, DailyHeartRates> getDailyHeartRates(int deviceId, int sinceLocalDate) {
        return getDailyHeartRates(deviceId, sinceLocalDate, 40); // Default conservative filter
    }

    private LocalDate intToLocalDate(int dateInt) {
        try {
            int year = dateInt / 10000;
            int month = (dateInt % 10000) / 100;
            int day = dateInt % 100;
            return LocalDate.of(year, month, day);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error parsing date integer: " + dateInt, e);
            return null;
        }
    }

    public void storeEnhancedDailyRHRValues(Map<LocalDate, RhrResult> dailyEnhancedRHR, int deviceId) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for storing enhanced RHR values");
            return;
        }
        if (rhrRepository != null) {
            rhrRepository.storeEnhancedDailyRHRValues(dailyEnhancedRHR, deviceId);
        } else {
            AppLogger.e(TAG, "RHR Repository not initialized");
        }
    }

    public void resetDatabase() {
        AppLogger.d(TAG, "Resetting paiesque database");

        // Close current connections
        close();

        // Delete the database file
        File dbFile = new File(targetDbPath);
        if (dbFile.exists()) {
            boolean deleted = dbFile.delete();
            AppLogger.d(TAG, "Database file deletion: " + (deleted ? "successful" : "failed"));

            // Also delete related files (journal, wal, etc.)
            String[] suffixes = {"-journal", "-wal", "-shm"};
            for (String suffix : suffixes) {
                File relatedFile = new File(targetDbPath + suffix);
                if (relatedFile.exists()) {
                    boolean relatedDeleted = relatedFile.delete();
                    AppLogger.d(TAG, "Related file " + relatedFile.getName() + " deletion: " +
                            (relatedDeleted ? "successful" : "failed"));
                }
            }
        } else {
            AppLogger.d(TAG, "Database file does not exist, nothing to delete");
        }

        // Reset state
        this.targetDb = null;
        this.deviceRepository = null;
        this.heartRateRepository = null;
        this.paiRepository = null;
        this.rhrRepository = null;
        this.timestampRepository = null;
        this.databaseStatsRepository = null;
        this.heartRateSyncService = null;
        this.checkpointManager = null;

        // Recreate executor service
        if (executorService != null && !executorService.isShutdown()) {
            executorService.shutdown();
            try {
                if (!executorService.awaitTermination(5, TimeUnit.SECONDS)) {
                    executorService.shutdownNow();
                }
            } catch (InterruptedException e) {
                executorService.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        this.executorService = Executors.newFixedThreadPool(3);

        AppLogger.d(TAG, "Paiesque database reset completed");
    }

    /**
     * Get all devices from paiesque.db ONLY
     */
    public List<DeviceInfo> getDevicesFromPaiesqueDb() {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for reading devices");
            return new ArrayList<>();
        }

        List<DeviceInfo> devices = new ArrayList<>();
        String sql = "SELECT " +
                AppConstants.Columns.ID + ", " +
                AppConstants.Columns.NAME + ", " +
                AppConstants.Columns.MANUFACTURER + ", " +
                AppConstants.Columns.IDENTIFIER +
                " FROM " + AppConstants.Database.TABLE_DEVICE +
                " ORDER BY " + AppConstants.Columns.NAME;

        try (Cursor cursor = targetDb.rawQuery(sql, null)) {
            if (cursor == null) {
                AppLogger.e(TAG, "Cursor is null for devices query");
                return devices;
            }

            while (cursor.moveToNext()) {
                try {
                    long id = cursor.getLong(cursor.getColumnIndexOrThrow(AppConstants.Columns.ID));
                    String name = cursor.getString(cursor.getColumnIndexOrThrow(AppConstants.Columns.NAME));
                    String manufacturer = cursor.getString(cursor.getColumnIndexOrThrow(AppConstants.Columns.MANUFACTURER));
                    String identifier = cursor.getString(cursor.getColumnIndexOrThrow(AppConstants.Columns.IDENTIFIER));

                    devices.add(new DeviceInfo(id, name, manufacturer, identifier));
                } catch (Exception e) {
                    AppLogger.w(TAG, "Error parsing device from database", e);
                }
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading devices from paiesque database", e);
        }

        AppLogger.d(TAG, "Retrieved " + devices.size() + " devices from paiesque.db");
        return devices;
    }

    /**
     * Initialize devices in paiesque.db from gadgetbridge.db (one-time operation)
     */
    public void initializeDevicesFromSource(String sourceDbPath) {
        if (!ensureDatabaseForSettings()) {
            AppLogger.e(TAG, "Database not ready for device initialization");
            return;
        }

        // Check if we already have devices
        List<DeviceInfo> existingDevices = getDevicesFromPaiesqueDb();
        if (!existingDevices.isEmpty()) {
            AppLogger.d(TAG, "Devices already exist in paiesque.db, skipping initialization");
            return;
        }

        // Source database check
        if (!gadgetbridgeSyncSource.isSourceInitialized()) {
            AppLogger.e(TAG, "Gadgetbridge sync source not initialized for device initialization");
            return;
        }

        // Read devices from Gadgetbridge sync source
        String[] columns = {
                AppConstants.Columns.ID,
                AppConstants.Columns.NAME,
                AppConstants.Columns.MANUFACTURER,
                AppConstants.Columns.IDENTIFIER
        };

        List<Map<String, String>> sourceDevices = gadgetbridgeSyncSource.readDataFromTable(
                AppConstants.Database.TABLE_DEVICE, columns, null, null);

        AppLogger.d(TAG, "Found " + sourceDevices.size() + " devices in Gadgetbridge source for initialization");

        targetDb.beginTransaction();
        try {
            String insertSql = "INSERT INTO " + AppConstants.Database.TABLE_DEVICE + " (" +
                    AppConstants.Columns.NAME + ", " +
                    AppConstants.Columns.MANUFACTURER + ", " +
                    AppConstants.Columns.IDENTIFIER + ", " +
                    "SOURCE_DEVICE_ID" +
                    ") VALUES (?, ?, ?, ?)";

            android.database.sqlite.SQLiteStatement insertStatement = targetDb.compileStatement(insertSql);

            for (Map<String, String> device : sourceDevices) {
                String name = device.get(AppConstants.Columns.NAME);
                String manufacturer = device.get(AppConstants.Columns.MANUFACTURER);
                String identifier = device.get(AppConstants.Columns.IDENTIFIER);
                String sourceId = device.get(AppConstants.Columns.ID);

                if (name == null || manufacturer == null || identifier == null || sourceId == null) {
                    AppLogger.w(TAG, "Skipping device with null values: " + device);
                    continue;
                }

                // Insert into paiesque.db (initial population)
                insertStatement.bindString(1, name);
                insertStatement.bindString(2, manufacturer);
                insertStatement.bindString(3, identifier);
                insertStatement.bindString(4, sourceId);
                insertStatement.execute();
                insertStatement.clearBindings();

                AppLogger.d(TAG, "Initialized device in paiesque.db: " + name);
            }
            targetDb.setTransactionSuccessful();
            AppLogger.d(TAG, "Device initialization completed successfully");
            AppLogger.i(TAG, "found " + sourceDevices.size() + " devices");
        } catch (Exception e) {
            AppLogger.e(TAG, "Error in device initialization", e);
        } finally {
            targetDb.endTransaction();
        }
    }

    public SQLiteDatabase getWritableDatabase() {
        if (!isDatabaseReady()) {
            createOrOpenTargetDatabase();
        }
        return targetDb;
    }

    /**
     * Get zone data for a specific date
     */
    public Map<String, Double> getZoneDataForDate(int deviceId, LocalDate date) {
        Map<String, Double> zoneData = new HashMap<>();

        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for zone data query");
            return zoneData;
        }

        int dateInt = localDateToInt(date);
        String sql = "SELECT " +
                "PAI_ZONE_50_59, PAI_ZONE_60_69, PAI_ZONE_70_79, " +
                "PAI_ZONE_80_89, PAI_ZONE_90_PLUS " +
                "FROM " + AppConstants.PaiesqueTables.PAI_SCORES + " " +
                "WHERE LOCAL_DATE = ? AND DEVICE_ID = ?";

        try (Cursor cursor = targetDb.rawQuery(sql, new String[]{
                String.valueOf(dateInt), String.valueOf(deviceId)})) {

            if (cursor != null && cursor.moveToFirst()) {
                zoneData.put(PAIesqueCalculator.ZONE_50_59, cursor.getDouble(0));
                zoneData.put(PAIesqueCalculator.ZONE_60_69, cursor.getDouble(1));
                zoneData.put(PAIesqueCalculator.ZONE_70_79, cursor.getDouble(2));
                zoneData.put(PAIesqueCalculator.ZONE_80_89, cursor.getDouble(3));
                zoneData.put(PAIesqueCalculator.ZONE_90_PLUS, cursor.getDouble(4));
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting zone data for date: " + date, e);
        }

        return zoneData;
    }

    /**
     * Get zone analysis data for a device
     * @param deviceId The device ID
     * @param daysBack Number of days to look back (0 for all data)
     * @return List of ZoneDayData for analysis
     */
    public List<ZoneDayData> getZoneAnalysisData(int deviceId, int daysBack) {
        List<ZoneDayData> zoneDataList = new ArrayList<>();

        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for zone analysis");
            return zoneDataList;
        }

        String sql;
        String[] selectionArgs;

        if (daysBack > 0) {
            // Calculate cutoff date
            LocalDate cutoffDate = LocalDate.now().minusDays(daysBack);
            int cutoffDateInt = localDateToInt(cutoffDate);

            sql = "SELECT LOCAL_DATE, PAI_SCORE, " +
                    "PAI_ZONE_50_59, PAI_ZONE_60_69, PAI_ZONE_70_79, " +
                    "PAI_ZONE_80_89, PAI_ZONE_90_PLUS, " +
                    "TIME_ZONE_50_59, TIME_ZONE_60_69, TIME_ZONE_70_79, " +
                    "TIME_ZONE_80_89, TIME_ZONE_90_PLUS " +
                    "FROM " + AppConstants.PaiesqueTables.PAI_SCORES + " " +
                    "WHERE DEVICE_ID = ? AND LOCAL_DATE >= ? " +
                    "ORDER BY LOCAL_DATE DESC";
            selectionArgs = new String[]{String.valueOf(deviceId), String.valueOf(cutoffDateInt)};
        } else {
            sql = "SELECT LOCAL_DATE, PAI_SCORE, " +
                    "PAI_ZONE_50_59, PAI_ZONE_60_69, PAI_ZONE_70_79, " +
                    "PAI_ZONE_80_89, PAI_ZONE_90_PLUS, " +
                    "TIME_ZONE_50_59, TIME_ZONE_60_69, TIME_ZONE_70_79, " +
                    "TIME_ZONE_80_89, TIME_ZONE_90_PLUS " +
                    "FROM " + AppConstants.PaiesqueTables.PAI_SCORES + " " +
                    "WHERE DEVICE_ID = ? " +
                    "ORDER BY LOCAL_DATE DESC";
            selectionArgs = new String[]{String.valueOf(deviceId)};
        }

        try (Cursor cursor = targetDb.rawQuery(sql, selectionArgs)) {
            if (cursor == null) {
                AppLogger.e(TAG, "Cursor is null for zone analysis query");
                return zoneDataList;
            }

            while (cursor.moveToNext()) {
                try {
                    int localDateInt = cursor.getInt(0);
                    LocalDate date = intToLocalDate(localDateInt);
                    if (date == null) continue;

                    ZoneDayData zoneData = new ZoneDayData(date);
                    zoneData.setTotalPai(cursor.getDouble(1));

                    // Set PAI zone values
                    zoneData.setPaiZone50_59(cursor.getDouble(2));
                    zoneData.setPaiZone60_69(cursor.getDouble(3));
                    zoneData.setPaiZone70_79(cursor.getDouble(4));
                    zoneData.setPaiZone80_89(cursor.getDouble(5));
                    zoneData.setPaiZone90Plus(cursor.getDouble(6));

                    // Set time zone values
                    zoneData.setTimeZone50_59(cursor.getDouble(7));
                    zoneData.setTimeZone60_69(cursor.getDouble(8));
                    zoneData.setTimeZone70_79(cursor.getDouble(9));
                    zoneData.setTimeZone80_89(cursor.getDouble(10));
                    zoneData.setTimeZone90Plus(cursor.getDouble(11));

                    zoneDataList.add(zoneData);

                } catch (Exception e) {
                    AppLogger.w(TAG, "Error parsing zone data row", e);
                }
            }

            AppLogger.d(TAG, "Loaded " + zoneDataList.size() + " days of zone analysis data");

        } catch (Exception e) {
            AppLogger.e(TAG, "Error loading zone analysis data", e);
        }

        return zoneDataList;
    }

    /**
     * Get PAI score for a specific date
     */
    public double getPaiForDate(int deviceId, LocalDate date) {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for PAI query");
            return 0.0;
        }

        int dateInt = localDateToInt(date);
        String sql = "SELECT PAI_SCORE FROM " + AppConstants.PaiesqueTables.PAI_SCORES +
                " WHERE LOCAL_DATE = ? AND DEVICE_ID = ?";

        try (Cursor cursor = targetDb.rawQuery(sql, new String[]{
                String.valueOf(dateInt), String.valueOf(deviceId)})) {

            if (cursor != null && cursor.moveToFirst()) {
                return cursor.getDouble(0);
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting PAI for date: " + date, e);
        }

        return 0.0;
    }

    private int localDateToInt(LocalDate date) {
        return date.getYear() * 10000 + date.getMonthValue() * 100 + date.getDayOfMonth();
    }

    /**
     * Get measured max HR using 95th percentile to exclude outliers
     */
    public int getMeasuredMaxHR() {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for max HR calculation");
            return 0;
        }

        try {
            // Get top 100 heart rates to calculate 95th percentile
            String sql = "SELECT HEART_RATE FROM " + AppConstants.PaiesqueTables.HEART_RATES +
                    " WHERE HEART_RATE > 100 ORDER BY HEART_RATE DESC LIMIT 100";

            List<Integer> topHRs = new ArrayList<>();
            try (Cursor cursor = targetDb.rawQuery(sql, null)) {
                if (cursor != null) {
                    while (cursor.moveToNext()) {
                        topHRs.add(cursor.getInt(0));
                    }
                }
            }

            if (!topHRs.isEmpty()) {
                // Use 95th percentile (5th element from top 100) to exclude outliers
                int index = (int) (topHRs.size() * 0.05);
                int measuredMaxHR = topHRs.get(Math.max(0, index));
                AppLogger.d(TAG, "Measured max HR (95th percentile): " + measuredMaxHR);
                return measuredMaxHR;
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error calculating measured max HR", e);
        }

        return 0;
    }
    public String getSetting(String key) {
        // Use ensureDatabaseForSettings() instead of isDatabaseReady()
        if (!ensureDatabaseForSettings()) {
            AppLogger.e(TAG, "Database not ready for reading setting: " + key);
            return null;
        }

        Cursor cursor = null;
        try {
            cursor = targetDb.query(AppConstants.SettingsTables.APP_SETTINGS,
                    new String[]{AppConstants.SettingsColumns.SETTING_VALUE},
                    AppConstants.SettingsColumns.SETTING_KEY + " = ?",
                    new String[]{key}, null, null, null);

            if (cursor != null && cursor.moveToFirst()) {
                return cursor.getString(0);
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading setting: " + key, e);
        } finally {
            if (cursor != null) cursor.close();
        }
        return null;
    }

    public void setSetting(String key, String value, String type, String category) {
        // Use ensureDatabaseForSettings() instead of isDatabaseReady()
        if (!ensureDatabaseForSettings()) {
            AppLogger.e(TAG, "Database not ready for writing setting: " + key);
            return;
        }

        try {
            ContentValues values = new ContentValues();
            values.put(AppConstants.SettingsColumns.SETTING_KEY, key);
            values.put(AppConstants.SettingsColumns.SETTING_VALUE, value);
            values.put(AppConstants.SettingsColumns.SETTING_TYPE, type);
            values.put(AppConstants.SettingsColumns.CATEGORY, category);
            values.put(AppConstants.SettingsColumns.LAST_MODIFIED, System.currentTimeMillis() / 1000);

            targetDb.insertWithOnConflict(AppConstants.SettingsTables.APP_SETTINGS, null, values,
                    SQLiteDatabase.CONFLICT_REPLACE);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error writing setting: " + key, e);
        }
    }

    /**
     * Clear all settings from the settings table
     */
    public void clearAllSettings() {
        if (!isDatabaseReady()) {
            AppLogger.e(TAG, "Database not ready for clearing settings");
            return;
        }

        try {
            // Check if database is writable before attempting delete
            if (targetDb != null && targetDb.isOpen() && !targetDb.isReadOnly()) {
                targetDb.delete(AppConstants.SettingsTables.APP_SETTINGS, null, null);
                AppLogger.i(TAG, "All settings cleared from database");
            } else {
                AppLogger.w(TAG, "Database is not writable, cannot clear settings");
                // Reset the database connection
                resetDatabase();
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error clearing settings", e);
            // If we get a readonly error, reset the database
            if (e instanceof android.database.sqlite.SQLiteReadOnlyDatabaseException) {
                AppLogger.w(TAG, "Database is readonly, resetting database connection");
                resetDatabase();
            }
        }
    }

    public SyncCheckpointManager getCheckpointManager() {
        return checkpointManager;
    }

}