package org.ojrandom.paiesque.data;

import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * Represents a single heart rate measurement with timestamp and metadata
 * Can be used by both PAI and RHR calculation systems
 */
public class HRPoint {
    private final ZonedDateTime time;
    private final int heartRate;
    private final Map<String, String> metadata;
    private final int minutesOfDay; // Pre-calculated for performance

    public HRPoint(ZonedDateTime time, int heartRate, Map<String, String> metadata) {
        this.time = time;
        this.heartRate = heartRate;
        this.metadata = metadata != null ? new HashMap<>(metadata) : new HashMap<>();
        this.minutesOfDay = time.getHour() * 60 + time.getMinute();
    }

    // Constructor that accepts pre-calculated minutes of day (for RHR optimization)
    public HRPoint(long timestamp, int heartRate, int minutesOfDay) {
        this.time = Instant.ofEpochSecond(timestamp).atZone(ZoneId.systemDefault());
        this.heartRate = heartRate;
        this.metadata = new HashMap<>();
        this.metadata.put("MINUTES_OF_DAY", String.valueOf(minutesOfDay));
        this.minutesOfDay = minutesOfDay;
    }

    // Getters
    public ZonedDateTime getTime() { return time; }
    public int getHeartRate() { return heartRate; }

    @Override
    public String toString() {
        return String.format("HRPoint[time=%s, hr=%d, minute=%d]",
                time, heartRate, minutesOfDay);
    }
}