package org.ojrandom.paiesque.data;

import android.content.Context;
import android.net.Uri;
import android.util.Log;

import androidx.documentfile.provider.DocumentFile;

import org.ojrandom.paiesque.logging.AppLogger;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;

public class DatabaseFileCopier {
    private static final String TAG = "DatabaseFileCopier";

    private final Context context;
    private final Uri uri;


    public static class CopyResult {
        public final boolean success;
        public final String dbPath;
        public final String errorMessage;

        public CopyResult(boolean success, String dbPath, String errorMessage) {
            this.success = success;
            this.dbPath = dbPath;
            this.errorMessage = errorMessage;
        }
    }

    public DatabaseFileCopier(Context context, Uri uri) {
        this.context = context;
        this.uri = uri;
    }

    private long getLastModifiedFromContentUri() {
        try {
            DocumentFile docFile = DocumentFile.fromSingleUri(context, uri);
            if (docFile != null && docFile.exists()) {
                long lastModified = docFile.lastModified();
                return lastModified;
            }
        } catch (SecurityException e) {
            AppLogger.e(TAG, "SecurityException when accessing DocumentFile: " + uri, e);
            // Fallback: return current time to force copy
            return System.currentTimeMillis();
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting last modified time: " + e.getMessage());
        }
        return 0;
    }

    private long getLastModifiedFromFile() {
        try {
            File finalFile = new File(context.getFilesDir(), AppConstants.KEY_DB_NAME);
            if (finalFile.exists()) {
                return finalFile.lastModified();
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading last modified time from file", e);
        }
        return 0;
    }

    public CopyResult copyDbToInternal() {
        File tempFile = null;
        File finalFile = null;

        try {
            long lastKnownModified = getLastModifiedFromFile();
            long currentModified = getLastModifiedFromContentUri();

            finalFile = new File(context.getFilesDir(), AppConstants.KEY_DB_NAME);
            String dbPath = finalFile.getAbsolutePath();

            boolean found = finalFile.exists();
            if (currentModified <= lastKnownModified && found) {
                AppLogger.d(TAG, "Database is up to date, using existing file");
                return new CopyResult(true, dbPath, null);
            }

            AppLogger.d(TAG, "Database needs update - copying from source");

            // FAILURE POINT 1: DocumentFile access
            DocumentFile docFile = DocumentFile.fromSingleUri(context, uri);
            if (docFile == null || !docFile.exists()) {
                return new CopyResult(false, null,
                        "Cannot access the selected file. Please check if the file still exists.");
            }

            // FAILURE POINT 2: InputStream creation
            InputStream input;
            try {
                input = context.getContentResolver().openInputStream(uri);
                if (input == null) {
                    return new CopyResult(false, null,
                            "Cannot read the database file. Please ensure the app has file permissions.");
                }
            } catch (SecurityException e) {
                return new CopyResult(false, null,
                        "Permission denied. Please reselect the database file to grant permissions.");
            }

            // FAILURE POINT 3: File copy process
            tempFile = new File(context.getFilesDir(), AppConstants.KEY_DB_NAME + ".tmp");
            if (tempFile.exists()) {
                tempFile.delete();
            }

            try (InputStream inputStream = input;
                 FileOutputStream output = new FileOutputStream(tempFile)) {

                byte[] buffer = new byte[1024];
                int length;
                while ((length = inputStream.read(buffer)) > 0) {
                    output.write(buffer, 0, length);
                }
            }

            // FAILURE POINT 4: Temp file validation
            if (!tempFile.exists()) {
                return new CopyResult(false, null,
                        "Temporary file was not created. Storage might be full.");
            }

            if (tempFile.length() == 0) {
                return new CopyResult(false, null,
                        "Temporary file is empty. The source file might be corrupted.");
            }

            // FAILURE POINT 5: Final file preparation
            if (finalFile.exists()) {
                if (!finalFile.delete()) {
                    AppLogger.w(TAG, "Could not delete old database file, will attempt overwrite");
                }
            }

            // FAILURE POINT 6: Rename operation
            if (!tempFile.renameTo(finalFile)) {
                return new CopyResult(false, null,
                        "Failed to save database file. Storage might be full.");
            }

            AppLogger.d(TAG, "Database successfully copied");
            return new CopyResult(true, finalFile.getAbsolutePath(), null);

        } catch (Exception e) {
            AppLogger.e(TAG, "Error copying database", e);

            // Clean up temp file on failure
            if (tempFile != null && tempFile.exists()) {
                tempFile.delete();
            }

            String errorMessage = "Unexpected error during database import.";
            if (e instanceof SecurityException) {
                errorMessage = "File access permission was denied.";
            } else if (e instanceof java.io.IOException) {
                errorMessage = "File copy failed. Storage might be full.";
            }

            return new CopyResult(false, null, errorMessage);
        }
    }
}