
package org.ojrandom.paiesque.data;

import android.content.Context;
import android.net.Uri;

import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.pai.PAIesqueCalculator;
import org.ojrandom.paiesque.rhr.RHRService;
import org.ojrandom.paiesque.rhr.RHRSummary;
import org.ojrandom.paiesque.rhr.RHRSummaryCalculator;
import org.ojrandom.paiesque.rhr.RhrResult;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;

public class DataManager {
    private static final String TAG = "DataManager";
    private static volatile DataManager instance;

    private final Context appContext;
    private GadgetbridgeFileImporter gadgetbridgeFileImporter;
    private final PaiesqueDBManager paiesqueDBManager;
    private final SettingsManager settingsManager;
    private final GadgetbridgeSyncSource gadgetbridgeSyncSource;
    private ExecutorService backgroundExecutor;

    // Private constructor for singleton
    private DataManager(Context context) {
        this.appContext = context.getApplicationContext();
        this.gadgetbridgeSyncSource = new GadgetbridgeSyncSource();
        this.paiesqueDBManager = new PaiesqueDBManager(appContext, gadgetbridgeSyncSource);
        this.settingsManager = new SettingsManager(appContext, paiesqueDBManager);
        AppLogger.d(TAG, "DataManager singleton instance created");
    }

    public static DataManager getInstance(Context context) {
        if (instance == null) {
            synchronized (DataManager.class) {
                if (instance == null) {
                    instance = new DataManager(context);
                }
            }
        }
        return instance;
    }

    public interface DatabaseProgressListener {
        void onProgress(String status);
        void onComplete(boolean success);
        void onError(String errorMessage);
    }

    public synchronized boolean initializeDatabase(Uri databaseUri, DatabaseProgressListener listener) {
        try {
            if (listener != null) listener.onProgress("Importing Gadgetbridge export...");

            this.gadgetbridgeFileImporter = new GadgetbridgeFileImporter(appContext, databaseUri);

            // Use enhanced copy with error reporting
            DatabaseFileCopier.CopyResult copyResult = gadgetbridgeFileImporter.importToInternalStorage();

            if (!copyResult.success) {
                // REPORT ERROR but continue with existing flow
                if (listener != null) {
                    listener.onError(copyResult.errorMessage);
                }
                // Still return false like original code
                if (listener != null) listener.onComplete(false);
                return false;
            }

            String sourceDbPath = copyResult.dbPath;

            if (listener != null) listener.onProgress("Initializing sync source...");

            boolean sourceInitialized = gadgetbridgeSyncSource.initializeSource(sourceDbPath);
            if (!sourceInitialized) {
                if (listener != null) listener.onComplete(false);
                return false;
            }

            if (listener != null) listener.onProgress("Setting up devices...");
            paiesqueDBManager.initializeDevicesFromSource(sourceDbPath);

            if (listener != null) listener.onProgress("Synchronizing data...");
            paiesqueDBManager.syncDatabases();

            gadgetbridgeSyncSource.closeSource();

            boolean isReady = paiesqueDBManager.isPaiesqueDbAvailable();
            if (listener != null) listener.onComplete(isReady);

            AppLogger.d(TAG, "Database initialization " + (isReady ? "successful" : "failed"));
            return isReady;

        } catch (Exception e) {
            AppLogger.e(TAG, "Error initializing database", e);
            if (listener != null) {
                listener.onError("Database initialization failed: " + e.getMessage());
                listener.onComplete(false);
            }
            return false;
        }
    }

    /**
     * Re-initializes the source database connection for sync operations
     * Used by swipe refresh to ensure source database is available
     */
    public synchronized String reinitializeSourceConnection(Uri databaseUri) {
        // Return null on success, error message on failure
        try {
            AppLogger.d(TAG, "Re-initializing Gadgetbridge source database connection");

            // Close existing source connection if any
            if (gadgetbridgeSyncSource != null) {
                gadgetbridgeSyncSource.closeSource();
            }

            // Create a new importer - this will check if file needs to be copied
            gadgetbridgeFileImporter = new GadgetbridgeFileImporter(appContext, databaseUri);

            // Use enhanced copy with error reporting
            DatabaseFileCopier.CopyResult copyResult = gadgetbridgeFileImporter.importToInternalStorage();
            if (!copyResult.success) {
                AppLogger.e(TAG, "Failed to import database for source re-initialization: " + copyResult.errorMessage);
                return copyResult.errorMessage; // Return the specific error
            }

            String sourceDbPath = copyResult.dbPath;

            boolean sourceInitialized = gadgetbridgeSyncSource.initializeSource(sourceDbPath);
            if (!sourceInitialized) {
                String error = "Cannot open database file - it might be corrupted";
                AppLogger.e(TAG, error);
                return error; // Return the specific error
            }

            AppLogger.d(TAG, "Gadgetbridge source database re-initialized successfully");
            return null; // Success - no error

        } catch (Exception e) {
            AppLogger.e(TAG, "Error re-initializing source database connection", e);
            return "Re-initialization failed: " + e.getMessage();
        }
    }

    public Map<LocalDate, DailyHeartRates> loadDailyHeartRatesForPAI(int deviceId) {
        int lastProcessedDate = 0;
        if (!hasSettingsChanged()) {
            lastProcessedDate = paiesqueDBManager.getLastProcessedDateForPAI(deviceId);
        }

        // Calculate minimum HR for database filtering
        int restingHR = settingsManager.getRHR();
        int maxHR = settingsManager.getMaxHR();
        int minHRForPAI = PAIesqueCalculator.getDatabaseFilterHR(restingHR, maxHR);

        AppLogger.d(TAG, "Using minimum HR filter for PAI: " + minHRForPAI +
                " bpm (restingHR=" + restingHR + ", maxHR=" + maxHR + ")");

        return paiesqueDBManager.getDailyHeartRates(deviceId, lastProcessedDate, minHRForPAI);
    }

    public Map<LocalDate, DailyHeartRates> loadDailyHeartRatesForRHR(int deviceId) {
        int lastProcessedDate = 0;
        if (!hasSleepSettingsChanged()) {
            lastProcessedDate = paiesqueDBManager.getLastProcessedDateForRHR(deviceId);
        }
        return paiesqueDBManager.getDailyHeartRates(deviceId, lastProcessedDate);
    }

    public List<Map<String, String>> calculatePAI(Map<LocalDate, DailyHeartRates> dailyData, int deviceId) {
        int restingHR = settingsManager.getRHR();
        int maxHR = settingsManager.getMaxHR();

        AppLogger.d(TAG, "Starting optimized PAI calculation for " + dailyData.size() + " days");

        // Load ALL historical PAI scores
        Map<LocalDate, Double> allDailyPAI = paiesqueDBManager.loadAllDailyPAIScores(deviceId);
        AppLogger.d(TAG, "Loaded " + allDailyPAI.size() + " historical PAI scores");

        if (!dailyData.isEmpty()) {
            // Calculate PAI for new days with zone breakdown
            Map<LocalDate, Double> newDailyPAI = calculatePAIFromDailyDataWithZones(dailyData, restingHR, maxHR, deviceId);

            AppLogger.d(TAG, "Calculated PAI for " + newDailyPAI.size() + " new days");

            // Update historical PAI with new calculations
            allDailyPAI.putAll(newDailyPAI);

            // Update timestamp
            if (!newDailyPAI.isEmpty()) {
                LocalDate latestDate = newDailyPAI.keySet().stream().max(LocalDate::compareTo).orElse(null);
                if (latestDate != null) {
                    int latestDateInt = localDateToInt(latestDate);
                    paiesqueDBManager.updateLastProcessedDateForPAI(deviceId, latestDateInt);
                    AppLogger.d(TAG, "Updated PAI last processed date to: " + latestDate);
                }
            }
        }

        // Calculate rolling PAI using complete dataset
        PAIesqueCalculator rollingCalculator = new PAIesqueCalculator(restingHR, maxHR, false);
        List<Map<String, String>> rollingPAI = rollingCalculator.calculate7DayRollingPAI(allDailyPAI);

        if (!rollingPAI.isEmpty()) {
            paiesqueDBManager.storeHistoricalRollingPAI(rollingPAI, deviceId);
            AppLogger.d(TAG, "Stored " + rollingPAI.size() + " historical rolling PAI records");
        }

        markSettingsAsProcessed();

        AppLogger.d(TAG, "PAI calculation complete: " + rollingPAI.size() + " rolling PAI records");
        return rollingPAI;
    }

    private int localDateToInt(LocalDate date) {
        return date.getYear() * 10000 + date.getMonthValue() * 100 + date.getDayOfMonth();
    }

    /**
     * Enhanced PAI calculation with zone tracking stored in PAI_SCORES table
     */
    private Map<LocalDate, Double> calculatePAIFromDailyDataWithZones(
            Map<LocalDate, DailyHeartRates> dailyData,
            int restingHR, int maxHR, int deviceId) {

        Map<LocalDate, Double> dailyPAI = new TreeMap<>();
        Map<LocalDate, PAIesqueCalculator.DailyPAIResult> dailyPAIResults = new TreeMap<>();
        PAIesqueCalculator PAIesqueCalculator = new PAIesqueCalculator(restingHR, maxHR, false);

        for (Map.Entry<LocalDate, DailyHeartRates> entry : dailyData.entrySet()) {
            LocalDate date = entry.getKey();
            DailyHeartRates dailyRates = entry.getValue();

            if (dailyRates.getCount() < 10) { // Minimum readings per day
                AppLogger.d(TAG, "Skipping " + date + " - insufficient data: " + dailyRates.getCount());
                continue;
            }

            // Convert to HeartRatePoint format using pre-calculated minutes from database
            List<HRPoint> hrPoints = dailyRates.getRecords().stream()
                    .map(record -> new HRPoint(
                            record.getTimestamp(),
                            record.getHeartRate(),
                            record.getMinutesOfDay()  // Use pre-calculated minutes from database
                    ))
                    .collect(Collectors.toList());

            // Use the new zone-based calculation
            PAIesqueCalculator.DailyPAIResult result = PAIesqueCalculator.calculateDailyPAIWithZones(hrPoints);
            dailyPAI.put(date, result.getTotalPAI());
            dailyPAIResults.put(date, result);

            AppLogger.d(TAG, date + ": " + result.getTotalPAI() + " PAI from " + dailyRates.getCount() + " readings");
            AppLogger.d(TAG, date + " zone breakdown: " + result.getPaiByZone());
        }

        // Store zone breakdown in PAI_SCORES table
        if (!dailyPAIResults.isEmpty()) {
            paiesqueDBManager.storeDailyPAIScoresWithZones(dailyPAIResults, deviceId);
            AppLogger.d(TAG, "Stored zone breakdown for " + dailyPAIResults.size() + " days in PAI_SCORES table");
        }

        return dailyPAI;
    }

    public RhrResult calculateRHR(Map<LocalDate, DailyHeartRates> dailyData, int deviceId) {
        AppLogger.d(TAG, "Starting enhanced RHR calculation for " + dailyData.size() + " days");

        long startTime = System.currentTimeMillis();

        // Load historical RHR ONCE
        TreeMap<LocalDate, Integer> historicalRHR = paiesqueDBManager.loadHistoricalRHR(deviceId);
        AppLogger.d(TAG, "Loaded " + historicalRHR.size() + " historical RHR records");

        // Calculate enhanced daily RHR
        Map<LocalDate, RhrResult> dailyEnhancedRhr = calculateEnhancedDailyRHR(dailyData, historicalRHR);

        // Store enhanced results AND update local historicalRHR map
        if (!dailyEnhancedRhr.isEmpty()) {
            paiesqueDBManager.storeEnhancedDailyRHRValues(dailyEnhancedRhr, deviceId);
            AppLogger.d(TAG, "Stored " + dailyEnhancedRhr.size() + " enhanced RHR values with metadata");

            // Update timestamp for last RHR calculation
            LocalDate latestDate = dailyEnhancedRhr.keySet().stream().max(LocalDate::compareTo).orElse(null);
            if (latestDate != null) {
                int latestDateInt = localDateToInt(latestDate);
                paiesqueDBManager.updateLastProcessedDateForRHR(deviceId, latestDateInt);
                AppLogger.d(TAG, "Updated RHR last processed date to: " + latestDate);
            }

            // Update the local historicalRHR map with new calculations instead of reloading from database
            dailyEnhancedRhr.forEach((date, result) -> {
                historicalRHR.put(date, result.getRhrValue());
            });
            AppLogger.d(TAG, "Updated historical RHR map with " + dailyEnhancedRhr.size() + " new values");
        }

        // Use the UPDATED historicalRHR map instead of reloading from database
        RHRSummaryCalculator summaryCalculator = new RHRSummaryCalculator();
        RHRSummary rhrSummary = summaryCalculator.getSummary(historicalRHR, deviceId);

        int restingHeartRate = rhrSummary.currentBaselineRHR;

        // Extract sample size from the most recent result
        int sampleSize = historicalRHR.size();
        if (!dailyEnhancedRhr.isEmpty()) {
            RhrResult latestResult = dailyEnhancedRhr.values().stream()
                    .reduce((first, second) -> second) // Get last element
                    .orElse(null);
            if (latestResult != null) {
                sampleSize = latestResult.getSampleSize();
            }
        }

        AppLogger.d(TAG, "Enhanced RHR calculation complete: " + restingHeartRate + " bpm" +
                ". Total historical records: " + historicalRHR.size());

        markSleepSettingsAsProcessed();

        long endTime = System.currentTimeMillis();
        AppLogger.d(TAG, "RHR calculation completed in " + (endTime - startTime) + "ms");

        // Return simplified RhrResult with today's RHR value
        RhrResult latestDailyResult = dailyEnhancedRhr.isEmpty() ? null :
                dailyEnhancedRhr.values().stream()
                        .reduce((first, second) -> second)
                        .orElse(null);

        if (latestDailyResult != null) {
            return latestDailyResult; // Return the most recent daily result
        } else {
            // Fallback: create a basic result with the baseline RHR - UPDATED CONSTRUCTOR
            return new RhrResult(restingHeartRate, sampleSize);
        }
    }

    public boolean hasDatabasePath() {
        // Store database path in settings table instead of SharedPreferences
        String dbPath = settingsManager.getStringSetting(AppConstants.PREFS_NAME + "_db_path", null);
        return dbPath != null;
    }

    public Uri getDatabaseUri() {
        String dbPath = settingsManager.getStringSetting(AppConstants.PREFS_NAME + "_db_path", null);
        return dbPath != null ? Uri.parse(dbPath) : null;
    }

    public void saveDatabasePath(Uri uri) {
        settingsManager.setStringSetting(AppConstants.PREFS_NAME + "_db_path", uri.toString());
    }

    public boolean hasSleepSettingsChanged() {
        return settingsManager.hasSleepSettingsChanged();
    }

    public void markSleepSettingsAsProcessed() {
        settingsManager.markSleepSettingsAsProcessed();
    }

    public boolean hasSettingsChanged() {
        return settingsManager.hasSettingsChanged();
    }

    public void markSettingsAsProcessed() {
        settingsManager.markSettingsAsProcessed();
    }

    public int getRestingHR() {
        return settingsManager.getRHR();
    }

    // Database operations go through PaiesqueDBManager
    public PaiesqueDBManager getPaiesqueDBManager() {
        return paiesqueDBManager;
    }

    public void closeDatabase() {
        if (gadgetbridgeFileImporter != null) {
            gadgetbridgeFileImporter.close();
        }
        if (paiesqueDBManager != null) {
            paiesqueDBManager.close();
        }
        if (gadgetbridgeSyncSource != null) {
            gadgetbridgeSyncSource.closeSource();
        }
    }

    public void cleanup() {
        AppLogger.d(TAG, "Cleaning up DataManager resources");
        closeDatabase();
        if (backgroundExecutor != null && !backgroundExecutor.isShutdown()) {
            backgroundExecutor.shutdown();
        }
    }

    public void cleanupAfterDelete() {
        // Reset the paiesque database completely
        if (this.paiesqueDBManager != null) {
            this.paiesqueDBManager.resetDatabase();
        }

        // Close gadgetbridgeFileImporter
        if (this.gadgetbridgeFileImporter != null) {
            this.gadgetbridgeFileImporter.close();
            this.gadgetbridgeFileImporter = null;
        }

        // Close sync source
        if (this.gadgetbridgeSyncSource != null) {
            this.gadgetbridgeSyncSource.closeSource();
        }

        AppLogger.d(TAG, "DataManager cleanup completed");
    }

    /**
     * Enhanced RHR calculation using simple calculator
     * This replaces the old strategy-based approach
     */
    private Map<LocalDate, RhrResult> calculateEnhancedDailyRHR(
            Map<LocalDate, DailyHeartRates> dailyData,
            Map<LocalDate, Integer> historicalRHR) {

        // Pass settingsManager to RHRService constructor
        RHRService rhrService = new RHRService(historicalRHR, settingsManager);
        return rhrService.calculateEnhancedDailyRHR(dailyData);
    }

    public SettingsManager getSettingsManager() {
        return settingsManager;
    }
}