package org.ojrandom.paiesque;

import android.os.Bundle;
import android.util.Log;
import android.widget.Button;
import android.widget.TimePicker;
import android.widget.Toast;

import androidx.activity.OnBackPressedCallback;
import androidx.appcompat.app.AppCompatActivity;

import org.ojrandom.paiesque.data.SettingsRepository;
import org.ojrandom.paiesque.logging.AppLogger;

public class SleepSettingsActivity extends AppCompatActivity {
    private static final String TAG = "SleepSettingsActivity";

    private SettingsRepository settingsRepository;
    private TimePicker timePickerStart;
    private TimePicker timePickerEnd;
    private Button buttonSave;
    private Button buttonReset;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_sleep_settings);

        PaiesqueApplication app = (PaiesqueApplication) getApplication();
        settingsRepository = new SettingsRepository(app.getDataManager());

        initializeViews();
        loadCurrentSettings();
        setupClickListeners();
        setupBackPressedHandler();
    }

    private void initializeViews() {
        timePickerStart = findViewById(R.id.timePickerSleepStart);
        timePickerEnd = findViewById(R.id.timePickerSleepEnd);
        buttonSave = findViewById(R.id.btnSaveSleepTime);
        buttonReset = findViewById(R.id.btnResetDefault);

        // Set 24-hour format
        timePickerStart.setIs24HourView(true);
        timePickerEnd.setIs24HourView(true);
    }

    private void loadCurrentSettings() {
        // Load current sleep settings from repository
        SettingsRepository.SleepSettings sleepSettings = settingsRepository.getSleepSettings();

        int startHour = sleepSettings.getStartHour();
        int startMinute = sleepSettings.getStartMinute();
        int endHour = sleepSettings.getEndHour();
        int endMinute = sleepSettings.getEndMinute();

        AppLogger.d(TAG, "Loaded sleep settings - Start: " + startHour + ":" + startMinute +
                ", End: " + endHour + ":" + endMinute);

        // Set time pickers to current values
        timePickerStart.setHour(startHour);
        timePickerStart.setMinute(startMinute);
        timePickerEnd.setHour(endHour);
        timePickerEnd.setMinute(endMinute);
    }

    private void setupClickListeners() {
        buttonSave.setOnClickListener(v -> saveSleepSettings());
        buttonReset.setOnClickListener(v -> resetToDefault());
    }

    private void setupBackPressedHandler() {
        // Modern replacement for onBackPressed()
        getOnBackPressedDispatcher().addCallback(this, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                // Show message that changes won't be saved
                Toast.makeText(SleepSettingsActivity.this, "Sleep settings not saved", Toast.LENGTH_SHORT).show();
                setResult(RESULT_CANCELED);
                finish();
            }
        });
    }

    private void saveSleepSettings() {
        try {
            int startHour = timePickerStart.getHour();
            int startMinute = timePickerStart.getMinute();
            int endHour = timePickerEnd.getHour();
            int endMinute = timePickerEnd.getMinute();

            AppLogger.d(TAG, "Saving sleep settings - Start: " + startHour + ":" + startMinute +
                    ", End: " + endHour + ":" + endMinute);

            // Validate times
            if (!isValidSleepTime(startHour, startMinute, endHour, endMinute)) {
                Toast.makeText(this, "Invalid sleep times. Please check your inputs.", Toast.LENGTH_LONG).show();
                return;
            }

            // Save to repository
            SettingsRepository.SleepSettings sleepSettings = new SettingsRepository.SleepSettings(
                    startHour, startMinute, endHour, endMinute
            );
            settingsRepository.saveSleepSettings(sleepSettings);

            Toast.makeText(this, "Sleep settings saved successfully", Toast.LENGTH_SHORT).show();

            // Set result and finish
            setResult(RESULT_OK);
            finish();

        } catch (Exception e) {
            AppLogger.e(TAG, "Error saving sleep settings", e);
            Toast.makeText(this, "Error saving sleep settings", Toast.LENGTH_SHORT).show();
        }
    }

    private void resetToDefault() {
        timePickerStart.setHour(0);
        timePickerStart.setMinute(0);
        timePickerEnd.setHour(6);
        timePickerEnd.setMinute(0);

        Toast.makeText(this, "Reset to default sleep times", Toast.LENGTH_SHORT).show();
    }

    private boolean isValidSleepTime(int startHour, int startMinute, int endHour, int endMinute) {
        // Basic validation - you can enhance this as needed
        if (startHour < 0 || startHour > 23 || endHour < 0 || endHour > 23) {
            return false;
        }
        if (startMinute < 0 || startMinute > 59 || endMinute < 0 || endMinute > 59) {
            return false;
        }

        // Check if end time is after start time (prevent going over midnight)
        int startTotalMinutes = startHour * 60 + startMinute;
        int endTotalMinutes = endHour * 60 + endMinute;

        // End time must be after start time to prevent going over midnight
        if (endTotalMinutes <= startTotalMinutes) {
            return false;
        }

        // Check if we have at least 3 hours of sleep time
        int sleepDurationMinutes = endTotalMinutes - startTotalMinutes;
        int minimumSleepMinutes = 3 * 60; // 3 hours in minutes

        return sleepDurationMinutes >= minimumSleepMinutes;
    }
}