package org.ojrandom.paiesque;

import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.Spinner;
import android.widget.TableLayout;
import android.widget.TableRow;
import android.widget.TextView;
import android.widget.Toast;

import androidx.appcompat.app.AppCompatActivity;

import com.google.android.material.floatingactionbutton.FloatingActionButton;

import org.ojrandom.paiesque.data.DataManager;
import org.ojrandom.paiesque.data.SettingsRepository;

public class SettingsActivity extends AppCompatActivity {

    private static final String TAG = "SettingsActivity";

    private enum HrMode { CALCULATED, FORMULA, MANUAL }
    private HrMode currentRhrMode = HrMode.MANUAL;
    private HrMode currentMaxHrMode = HrMode.MANUAL;

    private SettingsRepository settingsRepository;
    private int currentDeviceId = -1;

    // Views
    private View rhrInputContainer, maxHrInputContainer;
    private TextView rhrCalculatedValue, maxHrCalculatedValue;
    private LinearLayout formulaOptionsLayout;
    private EditText inputRhr, inputMaxHr, inputAge;
    private Spinner spinnerGender;
    private RadioGroup radioGroupFormula;
    private TableLayout intensityTable;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_settings);

        // Get device ID from intent
        currentDeviceId = getIntent().getIntExtra("DEVICE_ID", -1);

        PaiesqueApplication app = (PaiesqueApplication) getApplication();
        DataManager dataManager = app.getDataManager();
        settingsRepository = new SettingsRepository(dataManager);

        initializeViews();
        setupViews();
        loadCurrentSettings();
    }

    private void initializeViews() {
        // RHR Section
        rhrInputContainer = findViewById(R.id.rhr_input_layout);
        rhrCalculatedValue = findViewById(R.id.rhr_calculated_value);
        inputRhr = findViewById(R.id.input_rhr);

        // Max HR Section
        maxHrInputContainer = findViewById(R.id.maxhr_input_layout);
        maxHrCalculatedValue = findViewById(R.id.maxhr_calculated_value);
        formulaOptionsLayout = findViewById(R.id.formula_options_layout);
        inputMaxHr = findViewById(R.id.input_max_hr);
        inputAge = findViewById(R.id.input_age);
        spinnerGender = findViewById(R.id.spinner_gender);
        radioGroupFormula = findViewById(R.id.radio_group_formula);

        // Intensity Table
        intensityTable = findViewById(R.id.intensity_table);

        // Save Button
        FloatingActionButton fabSave = findViewById(R.id.fab_save);
        fabSave.setOnClickListener(v -> saveSettings());
    }

    private void setupViews() {
        setupRhrSection();
        setupMaxHrSection();
        setupGenderSpinner();
        setupFormulaListeners();
        setupValidation();
    }

    private void setupRhrSection() {
        RadioGroup rhrModeGroup = findViewById(R.id.rhr_mode_group);

        rhrModeGroup.setOnCheckedChangeListener((group, checkedId) -> {
            if (checkedId == R.id.rhr_calculated) {
                currentRhrMode = HrMode.CALCULATED;
                rhrInputContainer.setVisibility(View.GONE);

                // Load and show calculated value
                SettingsRepository.HealthData healthData =
                        settingsRepository.getCalculatedHealthData(currentDeviceId);

                if (healthData.hasRHRData) {
                    rhrCalculatedValue.setText(getString(R.string.calculated_value, healthData.restingHR));
                    rhrCalculatedValue.setVisibility(View.VISIBLE);
                    inputRhr.setText(String.valueOf(healthData.restingHR));
                } else {
                    rhrCalculatedValue.setText(R.string.no_data_available);
                    rhrCalculatedValue.setVisibility(View.VISIBLE);
                    inputRhr.setText("");
                }
            } else {
                currentRhrMode = HrMode.MANUAL;
                rhrInputContainer.setVisibility(View.VISIBLE);
                rhrCalculatedValue.setVisibility(View.GONE);
            }
            updateIntensityTable();
        });
    }

    private void setupMaxHrSection() {
        RadioGroup maxHrModeGroup = findViewById(R.id.maxhr_mode_group);

        maxHrModeGroup.setOnCheckedChangeListener((group, checkedId) -> {
            if (checkedId == R.id.maxhr_calculated) {
                currentMaxHrMode = HrMode.CALCULATED;
                maxHrInputContainer.setVisibility(View.GONE);
                formulaOptionsLayout.setVisibility(View.GONE);

                SettingsRepository.HealthData healthData =
                        settingsRepository.getCalculatedHealthData(currentDeviceId);

                if (healthData.hasMaxHRData) {
                    maxHrCalculatedValue.setText(getString(R.string.calculated_value, healthData.maxHR));
                    maxHrCalculatedValue.setVisibility(View.VISIBLE);
                    inputMaxHr.setText(String.valueOf(healthData.maxHR));
                } else {
                    maxHrCalculatedValue.setText(R.string.no_data_available);
                    maxHrCalculatedValue.setVisibility(View.VISIBLE);
                    inputMaxHr.setText("");
                }
            } else if (checkedId == R.id.maxhr_formula) {
                currentMaxHrMode = HrMode.FORMULA;
                maxHrInputContainer.setVisibility(View.GONE);
                formulaOptionsLayout.setVisibility(View.VISIBLE);
                maxHrCalculatedValue.setVisibility(View.GONE);
                calculateMaxHrFromFormula();
            } else {
                currentMaxHrMode = HrMode.MANUAL;
                maxHrInputContainer.setVisibility(View.VISIBLE);
                formulaOptionsLayout.setVisibility(View.GONE);
                maxHrCalculatedValue.setVisibility(View.GONE);
            }
            updateIntensityTable();
        });
    }

    private void setupGenderSpinner() {
        ArrayAdapter<CharSequence> adapter = ArrayAdapter.createFromResource(
                this,
                R.array.gender_options,
                android.R.layout.simple_spinner_item  // Use default instead of custom dark
        );
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item); // Use default
        spinnerGender.setAdapter(adapter);
    }

    private void setupFormulaListeners() {
        // Age input listener
        inputAge.addTextChangedListener(new TextWatcher() {
            @Override public void beforeTextChanged(CharSequence s, int start, int count, int after) {}
            @Override public void afterTextChanged(Editable s) {}

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                if (currentMaxHrMode == HrMode.FORMULA) {
                    calculateMaxHrFromFormula();
                }
            }
        });

        // Gender spinner listener
        spinnerGender.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                if (currentMaxHrMode == HrMode.FORMULA) {
                    calculateMaxHrFromFormula();
                }
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {}
        });

        // Formula radio group listener
        radioGroupFormula.setOnCheckedChangeListener((group, checkedId) -> {
            if (currentMaxHrMode == HrMode.FORMULA) {
                calculateMaxHrFromFormula();
            }
        });
    }

    private void setupValidation() {
        // Add text watchers for real-time validation
        inputRhr.addTextChangedListener(new SimpleTextWatcher() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                updateIntensityTable();
            }
        });

        inputMaxHr.addTextChangedListener(new SimpleTextWatcher() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                updateIntensityTable();
            }
        });
    }

    private void loadCurrentSettings() {
        SettingsRepository.PaiSettings settings = settingsRepository.getPaiSettings();

        // Set initial modes based on saved settings - FIXED: Use saved max HR mode
        String savedMaxHrMode = settings.getMaxHrMode();
        if ("calculated".equals(savedMaxHrMode)) {
            currentMaxHrMode = HrMode.CALCULATED;
            ((RadioButton) findViewById(R.id.maxhr_calculated)).setChecked(true);
        } else if ("manual".equals(savedMaxHrMode)) {
            currentMaxHrMode = HrMode.MANUAL;
            ((RadioButton) findViewById(R.id.maxhr_manual)).setChecked(true);
        } else {
            currentMaxHrMode = HrMode.FORMULA;
            ((RadioButton) findViewById(R.id.maxhr_formula)).setChecked(true);
        }

        // Set RHR mode based on saved settings
        if (settings.isManualRHR()) {
            currentRhrMode = HrMode.MANUAL;
            ((RadioButton) findViewById(R.id.rhr_manual)).setChecked(true);
        } else {
            currentRhrMode = HrMode.CALCULATED;
            ((RadioButton) findViewById(R.id.rhr_calculated)).setChecked(true);
        }

        // Load values
        inputRhr.setText(String.valueOf(settings.getRestingHR()));
        inputMaxHr.setText(String.valueOf(settings.getMaxHR()));
        inputAge.setText(settings.getAge() > 0 ? String.valueOf(settings.getAge()) : "");

        // Load gender
        setGenderSelection(settings.getGender());

        // Load formula
        setFormulaSelection(settings.getMaxHRFormula());

        // Update UI based on initial modes
        updateUiForCurrentModes();
        updateIntensityTable();
    }

    private void updateUiForCurrentModes() {
        // Update RHR section
        if (currentRhrMode == HrMode.CALCULATED) {
            rhrInputContainer.setVisibility(View.GONE);
            SettingsRepository.HealthData healthData = settingsRepository.getCalculatedHealthData(currentDeviceId);
            if (healthData.hasRHRData) {
                rhrCalculatedValue.setText(getString(R.string.calculated_value, healthData.restingHR));
                rhrCalculatedValue.setVisibility(View.VISIBLE);
            } else {
                rhrCalculatedValue.setText(R.string.no_data_available);
                rhrCalculatedValue.setVisibility(View.VISIBLE);
            }
        } else {
            rhrInputContainer.setVisibility(View.VISIBLE);
            rhrCalculatedValue.setVisibility(View.GONE);
        }

        // Update Max HR section
        if (currentMaxHrMode == HrMode.CALCULATED) {
            maxHrInputContainer.setVisibility(View.GONE);
            formulaOptionsLayout.setVisibility(View.GONE);
            SettingsRepository.HealthData healthData = settingsRepository.getCalculatedHealthData(currentDeviceId);
            if (healthData.hasMaxHRData) {
                maxHrCalculatedValue.setText(getString(R.string.calculated_value, healthData.maxHR));
                maxHrCalculatedValue.setVisibility(View.VISIBLE);
            } else {
                maxHrCalculatedValue.setText(R.string.no_data_available);
                maxHrCalculatedValue.setVisibility(View.VISIBLE);
            }
        } else if (currentMaxHrMode == HrMode.FORMULA) {
            maxHrInputContainer.setVisibility(View.GONE);
            formulaOptionsLayout.setVisibility(View.VISIBLE);
            maxHrCalculatedValue.setVisibility(View.GONE);
        } else {
            maxHrInputContainer.setVisibility(View.VISIBLE);
            formulaOptionsLayout.setVisibility(View.GONE);
            maxHrCalculatedValue.setVisibility(View.GONE);
        }
    }

    private void calculateMaxHrFromFormula() {
        if (currentMaxHrMode != HrMode.FORMULA) {
            return;
        }

        try {
            String ageText = inputAge.getText().toString();
            if (ageText.isEmpty()) {
                inputMaxHr.setText("");
                return;
            }

            int age = Integer.parseInt(ageText);
            String gender = getSelectedGender();
            String formula = getSelectedFormula();

            int calculatedMaxHR = calculateMaxHR(age, gender, formula);
            inputMaxHr.setText(String.valueOf(calculatedMaxHR));

        } catch (NumberFormatException e) {
            inputMaxHr.setText("");
        }
    }

    private int calculateMaxHR(int age, String gender, String formula) {
        switch (formula) {
            case "tanaka":
                return (int) (208 - (0.7 * age));
            case "gulati":
                if ("female".equals(gender)) {
                    return (int) (206 - (0.88 * age));
                } else {
                    // Show warning for men using Gulati
                    Toast.makeText(this, R.string.settings_gulati_warning, Toast.LENGTH_LONG).show();
                    return (int) (208 - (0.7 * age)); // Use Tanaka for men
                }
            case "hunt":
                return (int) (211 - (0.64 * age));
            case "standard":
            default:
                if ("female".equals(gender)) {
                    return 226 - age;
                } else {
                    return 220 - age;
                }
        }
    }

    private void updateIntensityTable() {
        try {
            String rhrText = inputRhr.getText().toString();
            String maxHrText = inputMaxHr.getText().toString();

            if (rhrText.isEmpty() || maxHrText.isEmpty()) {
                return;
            }

            int rhr = Integer.parseInt(rhrText);
            int maxHR = Integer.parseInt(maxHrText);

            if (rhr >= 30 && rhr <= 120 && maxHR >= 100 && maxHR <= 220) {
                populateIntensityTable(rhr, maxHR);
            }
        } catch (NumberFormatException e) {
            // Ignore - table will remain empty
        }
    }

    private void populateIntensityTable(int rhr, int maxHR) {
        // Clear existing rows (except header)
        int childCount = intensityTable.getChildCount();
        for (int i = 1; i < childCount; i++) {
            intensityTable.removeViewAt(1);
        }

        // Define zones
        IntensityZone[] zones = {
                new IntensityZone("50-59%", 0.50, 0.60, 0.2),
                new IntensityZone("60-69%", 0.60, 0.70, 0.5),
                new IntensityZone("70-79%", 0.70, 0.80, 1.0),
                new IntensityZone("80-89%", 0.80, 0.90, 1.5),
                new IntensityZone("90%+", 0.90, 1.00, 2.0)
        };

        // Add rows for each intensity zone
        for (IntensityZone zone : zones) {
            TableRow row = new TableRow(this);
            row.setLayoutParams(new TableRow.LayoutParams(
                    TableRow.LayoutParams.MATCH_PARENT,
                    TableRow.LayoutParams.WRAP_CONTENT
            ));

            // Calculate HR range using Karvonen formula
            int minHR = calculateTrainingHR(rhr, maxHR, zone.minPercentage);
            int maxHRZone = calculateTrainingHR(rhr, maxHR, zone.maxPercentage);

            // Intensity column
            TextView intensityText = createTableCell(zone.name);
            row.addView(intensityText);

            // HR Range column
            String hrRange = minHR + " - " + maxHRZone + " bpm";
            TextView hrRangeText = createTableCell(hrRange);
            row.addView(hrRangeText);

            // PAI/min column
            String paiRate = String.format("%.1f", zone.paiRate);
            TextView paiText = createTableCell(paiRate);
            row.addView(paiText);

            intensityTable.addView(row);
        }
    }

    private int calculateTrainingHR(int rhr, int maxHR, double percentage) {
        return (int) Math.round(((maxHR - rhr) * percentage) + rhr);
    }

    private TextView createTableCell(String text) {
        TextView textView = new TextView(this);
        textView.setLayoutParams(new TableRow.LayoutParams(
                0,
                TableRow.LayoutParams.WRAP_CONTENT,
                1f
        ));
        textView.setPadding(8, 8, 8, 8);
        textView.setText(text);
        // Let the system handle text coloring through theme
        return textView;
    }

    private String getSelectedGender() {
        int position = spinnerGender.getSelectedItemPosition();
        if (position == 0) return "male";
        if (position == 1) return "female";
        return "male"; // default
    }

    private void setGenderSelection(String gender) {
        if ("female".equals(gender)) {
            spinnerGender.setSelection(1);
        } else {
            spinnerGender.setSelection(0);
        }
    }

    private String getSelectedFormula() {
        int checkedId = radioGroupFormula.getCheckedRadioButtonId();
        if (checkedId == R.id.radio_tanaka) return "tanaka";
        if (checkedId == R.id.radio_gulati) return "gulati";
        if (checkedId == R.id.radio_trained) return "hunt";
        return "standard";
    }

    private void setFormulaSelection(String formula) {
        if ("tanaka".equals(formula)) {
            radioGroupFormula.check(R.id.radio_tanaka);
        } else if ("gulati".equals(formula)) {
            radioGroupFormula.check(R.id.radio_gulati);
        } else if ("hunt".equals(formula)) {
            radioGroupFormula.check(R.id.radio_trained);
        } else {
            radioGroupFormula.check(R.id.radio_standard);
        }
    }

    private void saveSettings() {
        try {
            // Validate required fields
            if (inputRhr.getText().toString().trim().isEmpty()) {
                Toast.makeText(this, R.string.settings_enter_resting_hr, Toast.LENGTH_LONG).show();
                return;
            }

            if (inputMaxHr.getText().toString().trim().isEmpty()) {
                Toast.makeText(this, R.string.settings_enter_max_hr, Toast.LENGTH_LONG).show();
                return;
            }

            int rhr = Integer.parseInt(inputRhr.getText().toString());
            int maxHR = Integer.parseInt(inputMaxHr.getText().toString());

            // Validate ranges
            if (rhr < 30 || rhr > 120) {
                Toast.makeText(this, R.string.settings_valid_resting_hr, Toast.LENGTH_LONG).show();
                return;
            }

            if (maxHR < 100 || maxHR > 220) {
                Toast.makeText(this, R.string.settings_valid_max_hr, Toast.LENGTH_LONG).show();
                return;
            }

            // Validate age if using formula
            String maxHrMode;
            switch (currentMaxHrMode) {
                case CALCULATED:
                    maxHrMode = "calculated";
                    break;
                case MANUAL:
                    maxHrMode = "manual";
                    break;
                case FORMULA:
                default:
                    maxHrMode = "formula";
                    break;
            }

            // Create and save settings - FIXED: Include RHR mode
            SettingsRepository.PaiSettings settings = new SettingsRepository.PaiSettings(
                    rhr,
                    maxHR,
                    maxHrMode,
                    currentRhrMode == HrMode.MANUAL, // Save RHR mode
                    inputAge.getText().toString().trim().isEmpty() ? 0 : Integer.parseInt(inputAge.getText().toString()),
                    getSelectedGender(),
                    getSelectedFormula()
            );

            settingsRepository.savePaiSettings(settings);
            Toast.makeText(this, R.string.settings_saved_successfully, Toast.LENGTH_SHORT).show();
            setResult(RESULT_OK);
            finish();


        } catch (NumberFormatException e) {
            Toast.makeText(this, R.string.settings_valid_numbers, Toast.LENGTH_LONG).show();
        }
    }

    // Helper classes
    private abstract class SimpleTextWatcher implements TextWatcher {
        @Override public void beforeTextChanged(CharSequence s, int start, int count, int after) {}
        @Override public void afterTextChanged(Editable s) {}
    }

    private static class IntensityZone {
        String name;
        double minPercentage;
        double maxPercentage;
        double paiRate;

        IntensityZone(String name, double minPercentage, double maxPercentage, double paiRate) {
            this.name = name;
            this.minPercentage = minPercentage;
            this.maxPercentage = maxPercentage;
            this.paiRate = paiRate;
        }
    }
}