package org.ojrandom.paiesque;

import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Spinner;
import android.widget.Toast;

import androidx.activity.EdgeToEdge;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.content.ContextCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import org.ojrandom.paiesque.calculation.CalculationManager;
import org.ojrandom.paiesque.calculation.CalculationProgressListener;
import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.data.DailyHeartRates;
import org.ojrandom.paiesque.data.DataManager;
import org.ojrandom.paiesque.data.DeviceInfo;
import org.ojrandom.paiesque.data.SettingsManager;
import org.ojrandom.paiesque.data.SettingsRepository;
import org.ojrandom.paiesque.data.backup.BackupExportManager;
import org.ojrandom.paiesque.data.backup.BackupImportManager;
import org.ojrandom.paiesque.io.FileOperationsManager;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.rhr.RHRSummary;
import org.ojrandom.paiesque.rhr.RHRSummaryCalculator;
import org.ojrandom.paiesque.rhr.RhrResult;
import org.ojrandom.paiesque.ui.MenuManager;
import org.ojrandom.paiesque.ui.PaiChartManager;
import org.ojrandom.paiesque.ui.PaiViewManager;
import org.ojrandom.paiesque.ui.ProgressManager;
import org.ojrandom.paiesque.ui.RhrChartManager;
import org.ojrandom.paiesque.ui.RhrViewManager;
import org.ojrandom.paiesque.ui.ZoneAnalysisActivity;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

public class MainActivity extends AppCompatActivity implements
        CalculationProgressListener, MenuManager.MenuActionListener {

    private static final String TAG = "MainActivity";
    public static final String CURRENT_DEVICE_ALL_MERGED = "MERGED_ALL";

    // Manager instances
    private PaiViewManager paiViewManager;
    private RhrViewManager rhrViewManager;
    private ProgressManager progressManager;
    private CalculationManager calculationManager;
    private DataManager dataManager;
    private FileOperationsManager fileOperationsManager;
    private MenuManager menuManager;
    private PaiChartManager paiChartManager;
    private RhrChartManager rhrChartManager;
    private SettingsRepository settingsRepository;

    // Data fields
    private String currentDevice;
    private List<String> deviceNames;
    private SwipeRefreshLayout swipeRefreshLayout;
    private Spinner deviceSpinner;
    private ArrayAdapter<String> deviceAdapter;
    private boolean isCalculationInProgress = false;
    private boolean isSettingsDialogOpen = false;


    // Activity result launchers
    private final ActivityResultLauncher<Intent> filePickerLauncher = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(),
            result -> {
                if (result.getResultCode() == RESULT_OK && result.getData() != null) {
                    Uri fileUri = result.getData().getData();
                    if (fileUri != null) {
                        processSelectedDatabase(fileUri);
                    }
                }
            }
    );

    private final ActivityResultLauncher<Intent> settingsLauncher = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(),
            result -> {
                // Reset the flag when SettingsActivity returns
                isSettingsDialogOpen = false;

                if (result.getResultCode() == RESULT_OK) {
                    Toast.makeText(this, R.string.toast_settings_saved, Toast.LENGTH_SHORT).show();

                    // Start calculations when settings are saved
                    if (currentDevice != null && dataManager.hasDatabasePath()) {
                        AppLogger.d(TAG, "Settings saved - starting calculations");
                        startPaiCalculation();
                    } else if (dataManager.hasDatabasePath()) {
                        // We have a database but no device selected yet - reload to initialize
                        Uri databaseUri = dataManager.getDatabaseUri();
                        if (databaseUri != null) {
                            AppLogger.d(TAG, "Settings saved - reloading database initialization");
                            processSelectedDatabase(databaseUri);
                        }
                    }
                } else if (result.getResultCode() == RESULT_CANCELED) {
                    Toast.makeText(this, R.string.toast_changes_discarded, Toast.LENGTH_SHORT).show();

                    // Even if cancelled, if we have a database we might need to proceed
                    if (dataManager.hasDatabasePath() && hasSetHeartRates() && currentDevice == null) {
                        Uri databaseUri = dataManager.getDatabaseUri();
                        if (databaseUri != null) {
                            AppLogger.d(TAG, "Settings cancelled but we have heart rates - reloading initialization");
                            processSelectedDatabase(databaseUri);
                        }
                    }
                }
            }
    );

    private final ActivityResultLauncher<Intent> importActivityLauncher = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(),
            result -> {
                if (result.getResultCode() == RESULT_OK && result.getData() != null) {
                    Uri databaseUri = result.getData().getData();
                    if (databaseUri != null) {
                        processSelectedDatabase(databaseUri);
                    }
                } else {
                    // User cancelled import - check if we have a database
                    if (!dataManager.hasDatabasePath()) {
                        // No database available, stay on import page or show message
                        showImportPage();
                    }
                }
            }
    );

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        EdgeToEdge.enable(this);
        setContentView(R.layout.activity_main);

        AppLogger.d(TAG, "=== MainActivity onCreate ===");
        AppLogger.d(TAG, "savedInstanceState: " + (savedInstanceState != null));

        // Check if we were already in the settings initialization flow
        boolean settingsInitializationInProgress = savedInstanceState != null &&
                savedInstanceState.getBoolean("settings_initialization_in_progress", false);

        if (savedInstanceState != null) {
            AppLogger.d(TAG, "MainActivity was RECREATED by system");
            if (settingsInitializationInProgress) {
                AppLogger.d(TAG, "Settings initialization was in progress, skipping duplicate open");
                // Don't open settings again - wait for the existing SettingsActivity to return
                initializeManagers();
                setupUI();
                setupCalculationListener();
                return;
            }
        }

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.main), (v, insets) -> {
            androidx.core.graphics.Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom);
            return insets;
        });

        // Get DataManager from Application instead of creating new instance
        PaiesqueApplication app = (PaiesqueApplication) getApplication();
        dataManager = app.getDataManager();
        settingsRepository = new SettingsRepository(dataManager.getSettingsManager());

        initializeManagers();
        setupUI();
        setupCalculationListener();

        // SIMPLE FIX: ALWAYS load data, even on rotation
        loadInitialData();
    }


    @Override
    public void onExportBackup() {
        // Create a file name with timestamp - using .db extension
        String fileName = "Paiesque_Backup_" +
                java.time.LocalDateTime.now().format(java.time.format.DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss")) +
                ".db";

        Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.setType("application/x-sqlite3"); // SQLite database MIME type
        intent.putExtra(Intent.EXTRA_TITLE, fileName);
        exportBackupLauncher.launch(intent);
    }

    @Override
    public void onImportBackup() {
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);

        // Use more compatible MIME types
        intent.setType("*/*");

        // Optionally, specify multiple MIME types for better filtering
        String[] mimeTypes = {
                "application/x-sqlite3",
                "application/octet-stream",
                "application/vnd.sqlite3",
                "application/sqlite3"
        };
        intent.putExtra(Intent.EXTRA_MIME_TYPES, mimeTypes);

        importBackupLauncher.launch(intent);
    }

    private final ActivityResultLauncher<Intent> exportBackupLauncher = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(),
            result -> {
                if (result.getResultCode() == RESULT_OK && result.getData() != null) {
                    Uri exportUri = result.getData().getData();
                    if (exportUri != null) {
                        showProgressDialog(getString(R.string.toast_starting_backup_export));
                        new Thread(() -> {
                            BackupExportManager exportManager = new BackupExportManager(MainActivity.this);
                            exportManager.setProgressListener(MainActivity.this);
                            boolean success = exportManager.exportBackup(exportUri);
                            runOnUiThread(() -> {
                                dismissProgressDialog();
                                if (success) {
                                    Toast.makeText(MainActivity.this, R.string.toast_backup_export_success, Toast.LENGTH_LONG).show();
                                } else {
                                    Toast.makeText(MainActivity.this, R.string.toast_backup_export_failed, Toast.LENGTH_LONG).show();
                                }
                            });
                        }).start();
                    }
                }
            }
    );

    private final ActivityResultLauncher<Intent> importBackupLauncher = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(),
            result -> {
                AppLogger.d(TAG, "Import backup result: " + result.getResultCode() + ", data: " + result.getData());

                if (result.getResultCode() == RESULT_OK && result.getData() != null) {
                    Uri backupUri = result.getData().getData();
                    AppLogger.d(TAG, "Backup URI: " + backupUri);

                    if (backupUri != null) {
                        showProgressDialog(getString(R.string.toast_starting_backup_import));
                        new Thread(() -> {
                            try {
                                BackupImportManager importManager = new BackupImportManager(MainActivity.this);
                                importManager.setProgressListener(MainActivity.this);
                                boolean success = importManager.importBackup(backupUri);
                                runOnUiThread(() -> {
                                    dismissProgressDialog();
                                    if (success) {
                                        Toast.makeText(MainActivity.this, R.string.toast_backup_import_success, Toast.LENGTH_LONG).show();
                                        if (dataManager.hasDatabasePath()) {
                                            Uri databaseUri = dataManager.getDatabaseUri();
                                            if (databaseUri != null) {
                                                processSelectedDatabase(databaseUri);
                                            }
                                        }
                                    } else {
                                        Toast.makeText(MainActivity.this, R.string.toast_backup_import_failed, Toast.LENGTH_LONG).show();
                                    }
                                });
                            } catch (Exception e) {
                                AppLogger.e(TAG, "Import backup error", e);
                                runOnUiThread(() -> {
                                    dismissProgressDialog();
                                    Toast.makeText(MainActivity.this, getString(R.string.toast_import_error, e.getMessage()), Toast.LENGTH_LONG).show();
                                });
                            }
                        }).start();
                    } else {
                        AppLogger.e(TAG, "Backup URI is null");
                    }
                } else {
                    AppLogger.d(TAG, "Import cancelled or failed, result code: " + result.getResultCode());
                }
            }
    );

    // Helper methods for progress dialog
    private void showProgressDialog(String message) {
        runOnUiThread(() -> {
            progressManager.showProgress(true);
            progressManager.updateProgress(0, 100, message);
        });
    }

    private void dismissProgressDialog() {
        runOnUiThread(() -> {
            progressManager.showProgress(false);
            progressManager.updateProgress(0, 100, "Ready");
        });
    }

    private void initializeManagers() {

        paiViewManager = new PaiViewManager(
                findViewById(R.id.pai_summary_card),
                findViewById(R.id.value_today_pai),
                findViewById(R.id.value_7day_pai),
                findViewById(R.id.progress_pai),
                findViewById(R.id.text_pai_progress_status),
                this
        );

        paiViewManager.setZoneDetailsClickListener(v -> {
            SharedPreferences prefs = getSharedPreferences("app_prefs", MODE_PRIVATE);
            prefs.edit().putBoolean("swipe_hint_shown", true).apply();

            openZoneAnalysisActivity();
        });

        rhrViewManager = new RhrViewManager(
                findViewById(R.id.rhr_summary_card),
                findViewById(R.id.value_last_week_rhr),
                findViewById(R.id.value_last_month_rhr),
                findViewById(R.id.value_current_baseline_rhr),
                findViewById(R.id.text_rhr_trend),
                this  // Pass Activity instead of Consumer<Runnable>
        );

        // Initialize chart managers with UI thread runner
        paiChartManager = new PaiChartManager(
                this.getApplicationContext(),
                findViewById(R.id.svg_week_chart_pai),
                findViewById(R.id.svg_month_chart_pai),
                findViewById(R.id.svg_alltime_chart_pai),
                findViewById(R.id.subtitle_alltime_date_range),
                this::runOnUiThread
        );

        rhrChartManager = new RhrChartManager(
                findViewById(R.id.svg_week_chart_rhr),
                findViewById(R.id.svg_month_chart_rhr),
                findViewById(R.id.svg_alltime_chart_rhr),
                findViewById(R.id.subtitle_alltime_date_range_rhr),
                this::runOnUiThread
        );

        // Initialize progress manager with activity context
        progressManager = new ProgressManager(
                this,
                findViewById(R.id.progress_indicator),
                findViewById(R.id.progress_calculation),
                findViewById(R.id.text_progress_title),
                findViewById(R.id.text_progress_status)
        );

        // Initialize calculation manager
        calculationManager = new CalculationManager(dataManager);

        // Initialize file operations manager
        fileOperationsManager = new FileOperationsManager(this);

        // Initialize menu manager
        menuManager = new MenuManager(this);
        menuManager.setActionListener(this);

        paiViewManager.setZoneDetailsClickListener(v -> openZoneAnalysisActivity());

        paiViewManager.debugZoneData();

        // Initialize swipe refresh after other managers
        swipeRefreshLayout = findViewById(R.id.swipe_refresh_layout);
        setupSwipeRefresh();

        setupDeviceSpinner();
    }

    private void setupUI() {
        // Initial UI setup is handled by the view managers
    }

    private void loadInitialData() {
        // Skip if we're already calculating
        if (isCalculationInProgress) {
            AppLogger.d(TAG, "Calculation already in progress, skipping loadInitialData");
            return;
        }

        if (dataManager.hasDatabasePath()) {
            Uri databaseUri = dataManager.getDatabaseUri();
            if (databaseUri != null) {
                processSelectedDatabase(databaseUri);
            }
        } else {
            showImportPage();
        }
    }

    private void processSelectedDatabase(Uri databaseUri) {
        AppLogger.d(TAG, "=== processSelectedDatabase CALLED ===");
        AppLogger.d(TAG, "URI: " + databaseUri);

        try {
            getContentResolver().takePersistableUriPermission(databaseUri,
                    Intent.FLAG_GRANT_READ_URI_PERMISSION);

            dataManager.saveDatabasePath(databaseUri);

            // Show progress immediately
            progressManager.showProgress(true);
            progressManager.updateProgressStatus(getString(R.string.progress_starting_database_init));

            // Run initialization in background thread
            new Thread(() -> {
                try {
                    runOnUiThread(() -> progressManager.updateProgressStatus(getString(R.string.progress_copying_database)));
                    Thread.sleep(100);
                    runOnUiThread(() -> progressManager.updateProgressStatus(getString(R.string.progress_synchronizing_data)));

                    // DEBUG: Check timestamp BEFORE sync
                    long timestampBeforeSync = dataManager.getPaiesqueDBManager().getCheckpointManager().getGlobalLastSyncTimestamp();
                    AppLogger.d(TAG, "=== DEBUG: Timestamp BEFORE sync: " + timestampBeforeSync + " ===");

                    // Actually initialize the database WITH ERROR HANDLING
                    dataManager.initializeDatabase(databaseUri, new DataManager.DatabaseProgressListener() {
                        @Override
                        public void onProgress(String status) {
                            runOnUiThread(() -> progressManager.updateProgressStatus(status));
                        }

                        @Override
                        public void onComplete(boolean success) {
                            runOnUiThread(() -> {
                                progressManager.showProgress(false);
                                if (success) {
                                    deviceNames = getDeviceNamesAfterInitialization();
                                    if (deviceNames != null && !deviceNames.isEmpty()) {
                                        // Get device selection from database
                                        SettingsManager settingsManager = dataManager.getSettingsManager();

                                        if (deviceNames.size() > 1) {
                                            currentDevice = CURRENT_DEVICE_ALL_MERGED;
                                        } else {
                                            currentDevice = deviceNames.get(0);
                                        }

                                        String savedPreference = settingsManager.getDeviceSelection();
                                        if (savedPreference == null) {
                                            saveDeviceSelection(currentDevice);
                                        } else {
                                            currentDevice = savedPreference;
                                        }

                                        // Update the spinner with devices
                                        updateDeviceSpinner(deviceNames, currentDevice);
                                        boolean heartRatesSet = hasSetHeartRates();
                                        AppLogger.d(TAG, "After database init - heart rates set: " + heartRatesSet + ", currentDevice: " + currentDevice);

                                        // Only open settings if heart rates aren't set AND settings aren't already open
                                        if (!heartRatesSet && !isSettingsDialogOpen) {
                                            AppLogger.d(TAG, "Opening settings for initial setup");
                                            isSettingsDialogOpen = true;
                                            openSettings();
                                        } else if (heartRatesSet) {
                                            AppLogger.d(TAG, "Starting PAI calculation with settings available");
                                            startPaiCalculation(null);
                                        }
                                    } else {
                                        showFileCopyError("No devices found in the database");
                                    }
                                } else {
                                    // Generic failure message (specific errors handled in onError)
                                    Toast.makeText(MainActivity.this, "Database initialization failed", Toast.LENGTH_LONG).show();
                                }
                            });
                        }

                        // handle file copy errors
                        @Override
                        public void onError(String errorMessage) {
                            runOnUiThread(() -> {
                                progressManager.showProgress(false);
                                showFileCopyError(errorMessage);
                            });
                        }
                    });

                    // DEBUG: Check timestamp AFTER sync
                    long timestampAfterSync = dataManager.getPaiesqueDBManager().getCheckpointManager().getGlobalLastSyncTimestamp();
                    AppLogger.d(TAG, "=== DEBUG: Timestamp AFTER sync: " + timestampAfterSync + " ===");

                } catch (Exception e) {
                    runOnUiThread(() -> {
                        progressManager.showProgress(false);
                        Toast.makeText(MainActivity.this, getString(R.string.toast_error, e.getMessage()), Toast.LENGTH_LONG).show();
                    });
                }
            }).start();

        } catch (SecurityException e) {
            // it handles permission issues
            progressManager.showProgress(false);
            showFileCopyError("Failed to get file permission: " + e.getMessage());
        }
    }

    private void showFileCopyError(String errorMessage) {
        // Use existing ProgressManager to show error
        progressManager.showError("File Import Failed: " + errorMessage);

        // Also show toast for immediate attention
        Toast.makeText(this, "File import failed", Toast.LENGTH_LONG).show();

        // Log for debugging
        AppLogger.e(TAG, "Database file copy error: " + errorMessage);
    }

    /**
     * Refresh database with sync only (no re-initialization)
     * Used for swipe-to-refresh operations
     */
    private void refreshDatabaseSyncOnly() {
        AppLogger.d(TAG, "=== refreshDatabaseSyncOnly CALLED ===");

        if (!dataManager.hasDatabasePath()) {
            Toast.makeText(this, "No database selected", Toast.LENGTH_SHORT).show();
            swipeRefreshLayout.setRefreshing(false);
            return;
        }

        progressManager.showProgress(true);
        progressManager.updateProgressStatus("Refreshing data...");

        new Thread(() -> {
            try {
                // Re-initialize the source database connection before sync
                Uri databaseUri = dataManager.getDatabaseUri();
                if (databaseUri != null) {
                    AppLogger.d(TAG, "Re-initializing source database connection for sync");
                    String errorMessage = dataManager.reinitializeSourceConnection(databaseUri);

                    if (errorMessage != null) {
                        // PROPAGATE SPECIFIC ERROR TO UI
                        runOnUiThread(() -> {
                            progressManager.showProgress(false);
                            swipeRefreshLayout.setRefreshing(false);
                            showFileCopyError("Refresh failed: " + errorMessage);
                        });
                        return;
                    }

                    AppLogger.d(TAG, "Source database re-initialized successfully, proceeding with sync");
                } else {
                    AppLogger.e(TAG, "No database URI available for source re-initialization");
                    throw new IllegalStateException("No database URI available");
                }

                // Now sync with the re-initialized source connection
                AppLogger.d(TAG, "Starting database sync after source re-initialization");
                dataManager.getPaiesqueDBManager().syncDatabases();

                runOnUiThread(() -> {
                    progressManager.showProgress(false);
                    swipeRefreshLayout.setRefreshing(false);

                    // RESET THE FLAG BEFORE STARTING NEW CALCULATION
                    isCalculationInProgress = false;

                    // Start calculations with the newly synced data
                    if (currentDevice != null) {
                        AppLogger.d(TAG, "Starting PAI calculation after successful sync");
                        startPaiCalculation();
                    } else {
                        AppLogger.w(TAG, "No current device selected after sync");
                    }
                });
            } catch (Exception e) {
                AppLogger.e(TAG, "Error during swipe refresh sync", e);
                runOnUiThread(() -> {
                    progressManager.showProgress(false);
                    swipeRefreshLayout.setRefreshing(false);
                    showFileCopyError("Refresh failed: " + e.getMessage());
                });
            }
        }).start();
    }

    private List<String> getDeviceNamesAfterInitialization() {
        try {
            List<DeviceInfo> devices = dataManager.getPaiesqueDBManager().getDevicesFromPaiesqueDb();
            List<String> deviceDisplayNames = new ArrayList<>();

            for (DeviceInfo device : devices) {
                // Use just the name for display
                deviceDisplayNames.add(device.getName());
            }

            // Add "All Devices Merged" option if multiple devices
            if (deviceDisplayNames.size() > 1) {
                deviceDisplayNames.add(getString(R.string.all_devices_merged));
            }

            return deviceDisplayNames;
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting device names", e);
            return null;
        }
    }

    private synchronized void updateDeviceSpinner(List<String> devices, String currentDevice) {
        runOnUiThread(() -> {
            AppLogger.d(TAG, "updateDeviceSpinner - devices: " + devices + ", currentDevice: " + currentDevice);
            if (deviceAdapter != null && devices != null) {
                deviceAdapter.clear();

                if (devices.isEmpty()) {
                    deviceAdapter.add(getString(R.string.no_devices_available));
                    deviceSpinner.setEnabled(false);
                } else {
                    deviceAdapter.addAll(devices);
                    deviceSpinner.setEnabled(true);

                    // Set selection based on current device
                    int position = -1;
                    if (currentDevice != null) {
                        if (currentDevice.equals(CURRENT_DEVICE_ALL_MERGED)) {
                            position = devices.indexOf(getString(R.string.all_devices_merged));
                        } else {
                            position = devices.indexOf(currentDevice);
                        }
                    }

                    if (position >= 0 && position < devices.size()) {
                        AppLogger.d(TAG, "Setting spinner to position: " + position + ", device: " + devices.get(position));
                        // Simply set the selection - don't prevent user interactions
                        deviceSpinner.setSelection(position);
                    }
                }
            }
        });
    }

    private void startPaiCalculation() {
        startPaiCalculation(null); // Call the new method with null URI
    }

    private void startPaiCalculation(Uri databaseUri) {
        AppLogger.d(TAG, "=== startPaiCalculation CALLED ===");
        AppLogger.d(TAG, "currentDevice: " + currentDevice + ", databaseUri: " + databaseUri);

        // Prevent concurrent calculations
        if (isCalculationInProgress) {
            AppLogger.d(TAG, "Calculation already in progress, skipping");
            return;
        }

        if (currentDevice == null) {
            Toast.makeText(this, "No device selected", Toast.LENGTH_SHORT).show();
            return;
        }

        if (!hasSetHeartRates()) {
            // FIX: Only open settings if they're not already open
            if (!isSettingsDialogOpen) {
                AppLogger.d(TAG, "Heart rates not set, opening settings");
                isSettingsDialogOpen = true;
                openSettings();
            } else {
                AppLogger.d(TAG, "Settings already open, skipping duplicate call");
            }
            return;
        }

        AppLogger.d(TAG, "=== startPaiCalculation STARTING ===");
        isCalculationInProgress = true;
        progressManager.showProgress(true);

        // Run the calculation on a background thread to avoid blocking UI
        new Thread(() -> {
            try {
                int deviceId = getDeviceIdForCurrentDevice();
                calculationManager.calculateAll(MainActivity.this, deviceId, currentDevice, databaseUri);
            } catch (Exception e) {
                AppLogger.e(TAG, "Error starting calculation", e);
                runOnUiThread(() -> {
                    isCalculationInProgress = false; // IMPORTANT: Reset if fails to start
                    onError("Failed to start calculation: " + e.getMessage());
                });
            }
        }).start();
    }

    private int getDeviceIdForCurrentDevice() {
        int deviceId = -1; // all devices merged

        if (!currentDevice.equals(CURRENT_DEVICE_ALL_MERGED) && deviceNames != null) {
            // Find the device by name and get its ID
            List<DeviceInfo> devices = dataManager.getPaiesqueDBManager().getDevicesFromPaiesqueDb();
            for (DeviceInfo device : devices) {
                if (device.getName().equals(currentDevice)) {
                    deviceId = (int) device.getId();
                    break;
                }
            }
        }
        return deviceId;
    }

    private void showImportPage() {
        Intent intent = new Intent(this, ImportActivity.class);
        importActivityLauncher.launch(intent);
    }

    // CalculationProgressListener implementation
    @Override
    public void onProgressUpdate(int step, int totalSteps, String status) {
        progressManager.updateProgress(step, totalSteps, status);

        // Keep swipe refresh visible during the entire calculation process
        // This ensures users see the refresh animation until everything is complete
        if (step < totalSteps && !swipeRefreshLayout.isRefreshing()) {
            runOnUiThread(() -> swipeRefreshLayout.setRefreshing(true));
        }
    }

    @Override
    public void onHeartRatesLoaded(Map<LocalDate, DailyHeartRates> dailyHeartRates) {
        int totalReadings = dailyHeartRates.values().stream()
                .mapToInt(DailyHeartRates::getCount)
                .sum();

        progressManager.updateProgress(1, 5, "Heart rate data loaded: " +
                dailyHeartRates.size() + " days, " + totalReadings + " readings");

        // FIX: Don't return early - let the calculation continue but handle empty data
        if (dailyHeartRates.isEmpty() || totalReadings == 0) {
            AppLogger.w(TAG, "No heart rate data found for device: " + currentDevice);
            // Don't return here - let the calculation complete normally
            // The empty data will be handled in subsequent callbacks
        }

        // Update UI on main thread
        runOnUiThread(() -> updateActionBarTitle());
    }

    private void openZoneAnalysisActivity() {
        int deviceId = getDeviceIdForCurrentDevice();
        Intent intent = new Intent(this, ZoneAnalysisActivity.class);
        intent.putExtra("DEVICE_ID", deviceId);
        startActivity(intent);
    }

    @Override
    public void onPaiCalculated(List<Map<String, String>> rollingPAI) {
        AppLogger.d(TAG, "onPaiCalculated called with data: " +
                (rollingPAI != null ? rollingPAI.size() + " items" : "null"));

        // FIX: Check if we got any meaningful PAI data
        if (rollingPAI == null || rollingPAI.isEmpty()) {
            AppLogger.w(TAG, "No PAI data calculated - likely no heart rate data available");
        }

        progressManager.updateProgress(2, 5, "PAI calculation complete");
        paiViewManager.updatePaiSummary(rollingPAI);

        // FIX: Display PAI charts with the calculated data
        displayPaiCharts(rollingPAI);

        // FIX: Load and display today's zone data
        loadAndDisplayTodayZoneData();
        progressManager.updateProgressStatus("Preparing heart rates for RHR calculations...");
    }

    private void loadAndDisplayTodayZoneData() {
        new Thread(() -> {
            try {
                int deviceId = getDeviceIdForCurrentDevice();
                LocalDate today = LocalDate.now();

                // Get today's zone data from database
                Map<String, Double> todayZoneData = dataManager.getPaiesqueDBManager()
                        .getZoneDataForDate(deviceId, today);

                // Get today's total PAI
                double todayTotalPai = dataManager.getPaiesqueDBManager()
                        .getPaiForDate(deviceId, today);

                AppLogger.d(TAG, "Today's zone data: " + todayZoneData + ", total PAI: " + todayTotalPai);

                // Always update UI, even with empty data
                runOnUiThread(() -> {
                    if (!todayZoneData.isEmpty() && todayTotalPai > 0) {
                        paiViewManager.updateZoneBreakdown(todayZoneData, todayTotalPai);
                    } else {
                        // Create default zone data with zeros using constants
                        Map<String, Double> defaultZoneData = createDefaultZoneData();
                        paiViewManager.updateZoneBreakdown(defaultZoneData, 0.0);
                    }
                });
            } catch (Exception e) {
                AppLogger.e(TAG, "Error loading today's zone data", e);
                // Even on error, show default data
                runOnUiThread(() -> {
                    Map<String, Double> defaultZoneData = createDefaultZoneData();
                    paiViewManager.updateZoneBreakdown(defaultZoneData, 0.0);
                });
            }
        }).start();
    }

    private Map<String, Double> createDefaultZoneData() {
        Map<String, Double> defaultZones = new HashMap<>();
        defaultZones.put(AppConstants.ZONE_50_59, 0.0);
        defaultZones.put(AppConstants.ZONE_60_69, 0.0);
        defaultZones.put(AppConstants.ZONE_70_79, 0.0);
        defaultZones.put(AppConstants.ZONE_80_89, 0.0);
        defaultZones.put(AppConstants.ZONE_90_PLUS, 0.0);
        return defaultZones;
    }

    @Override
    public void onHrStored() {
        progressManager.updateProgress(3, 5, "Preparing heart rates for resting heart rates calculations complete");
        progressManager.updateProgressStatus("Calculating resting heart rates...");
    }

    @Override
    public void onRhrCalculated(RhrResult rhrResult) {
        progressManager.updateProgress(4, 5, "RHR calculation complete");

        // Calculate full summary and update view manager
        if (rhrResult != null) {
            int deviceId = getDeviceIdForCurrentDevice();

            // Calculate the full RHR summary using historical data
            RHRSummaryCalculator summaryCalculator = new RHRSummaryCalculator();
            TreeMap<LocalDate, Integer> historicalRHR = dataManager.getPaiesqueDBManager().loadHistoricalRHR(deviceId);
            RHRSummary rhrSummary = summaryCalculator.getSummary(historicalRHR, deviceId);

            // Update the view manager with the calculated summary data
            rhrViewManager.updateRHRSummaryData(
                    rhrSummary.lastWeekRHR,
                    rhrSummary.lastMonthRHR,
                    rhrSummary.currentBaselineRHR
            );

            // Also update charts with the historical data
            rhrChartManager.updateDailyRHRData(historicalRHR);
        } else {
            // Handle case where no RHR result was calculated
            rhrViewManager.updateRHRSummaryData(-1, -1, -1);
        }

        progressManager.updateProgressStatus("Saving data...");
    }

    @Override
    public void onStorageComplete() {
        progressManager.updateProgress(5, 5, "Saving data complete");
    }

    @Override
    public void onAllCalculationsComplete(Boolean hasHeartRateData) {
        AppLogger.d(TAG, "=== onAllCalculationsComplete CALLED ===");
        AppLogger.d(TAG, "hasHeartRateData: " + hasHeartRateData);

        isCalculationInProgress = false;

        runOnUiThread(() -> {
            progressManager.showProgress(false);
            swipeRefreshLayout.setRefreshing(false);

            if (hasHeartRateData != null && !hasHeartRateData) {
                // Only show the message when we actually have no data
                handleNoHeartRateData(currentDevice);
            } else {
                // Show success message and update UI normally
                Toast.makeText(this, R.string.toast_all_calculations_complete, Toast.LENGTH_SHORT).show();
                updateActionBarTitle();
            }
        });
    }

    @Override
    public void onConfigurationChanged(@NonNull android.content.res.Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        // Regenerate charts with new dimensions when orientation changes
        if (paiChartManager != null) {
            paiChartManager.onConfigurationChanged();
        }
        if (rhrChartManager != null) {
            rhrChartManager.onConfigurationChanged();
        }
    }

    @Override
    public void onError(String errorMessage) {
        AppLogger.e(TAG, "=== onError CALLED: " + errorMessage + " ===");

        isCalculationInProgress = false;

        progressManager.showError(errorMessage);
        swipeRefreshLayout.setRefreshing(false);

        runOnUiThread(() -> {
            Toast.makeText(this, getString(R.string.toast_error, errorMessage), Toast.LENGTH_LONG).show();
            updateActionBarTitle();
        });
    }

    // MenuManager.MenuActionListener implementation
    @Override
    public void onSelectDatabase() {
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.setType("*/*");
        filePickerLauncher.launch(intent);
    }

    @Override
    public void onReloadDatabase() {
        if (dataManager.hasDatabasePath()) {
            Uri databaseUri = dataManager.getDatabaseUri();
            if (databaseUri != null) {
                if (!swipeRefreshLayout.isRefreshing()) {
                    swipeRefreshLayout.setRefreshing(true);
                }

                // Run on background thread to avoid blocking UI
                new Thread(() -> {
                    processSelectedDatabase(databaseUri);
                }).start();

            }
        } else {
            swipeRefreshLayout.setRefreshing(false);
            Toast.makeText(this, R.string.toast_no_database_selected, Toast.LENGTH_LONG).show();
            onSelectDatabase();
        }
    }

    @Override
    public void onDeleteAllData() {
        fileOperationsManager.deleteAllData();

        // Also cleanup DataManager state
        if (dataManager != null) {
            dataManager.cleanupAfterDelete();
        }

        Toast.makeText(this, "All data deleted", Toast.LENGTH_SHORT).show();

        // Reset current device and device names
        currentDevice = null;
        deviceNames = null;

        // Update UI to show empty state
        updateActionBarTitle();

        // Show import page to allow selecting new database
        showImportPage();
    }

    @Override
    public void onDeviceSelected(String device) {
        currentDevice = device;
        saveDeviceSelection(device);

        AppLogger.i(TAG, "Selected device: " + currentDevice);

        // Update the device repository with current selection if needed
        // (This ensures device mapping works correctly for heart rate sync)
        startPaiCalculation();
    }


    @Override
    public void onBatchProgress(int currentBatch, int totalBatches, String batchType) {
        runOnUiThread(() -> {
            String progressText = String.format("%s (Batch %d of %d)", batchType, currentBatch, totalBatches);

            // Single call to update both progress bar and status
            if (totalBatches > 0) {
                int progressPercent = (int) ((currentBatch * 100.0) / totalBatches);
                progressManager.updateProgress(progressPercent, 100, progressText);
            } else {
                // If no total batches, just update status
                progressManager.updateProgressStatus(progressText);
            }

            AppLogger.d(TAG, "Batch progress: " + progressText);
        });
    }

    private void saveDeviceSelection(String device) {
        // Use DataManager instead of SharedPreferences
        dataManager.getSettingsManager().setDeviceSelection(device);
        Toast.makeText(this, "Device selected: " + device, Toast.LENGTH_SHORT).show();
    }

    private void updateActionBarTitle() {
        if (getSupportActionBar() != null) {
            // Just show app name, no device info
            getSupportActionBar().setTitle(R.string.app_name);
            getSupportActionBar().setSubtitle(null);
        }
    }

    private boolean hasSetHeartRates() {
        // Use database instead of SharedPreferences
        int restingHR = dataManager.getRestingHR();
        return restingHR != 0;
    }

    private void openSettings() {
        AppLogger.d(TAG, "openSettings() called - currentDevice: " + currentDevice + ", hasHeartRates: " + hasSetHeartRates());

        Intent settingsIntent = new Intent(this, SettingsActivity.class);
        int deviceId = getDeviceIdForCurrentDevice();
        settingsIntent.putExtra("DEVICE_ID", deviceId);
        settingsLauncher.launch(settingsIntent);
    }

    // Chart display methods
    private void displayPaiCharts(List<Map<String, String>> paiData) {
        paiChartManager.displayCharts(paiData);
    }

    private void setupSwipeRefresh() {
        swipeRefreshLayout.setOnRefreshListener(() -> {
            if (dataManager.hasDatabasePath()) {
                Toast.makeText(MainActivity.this, R.string.toast_refreshing_data, Toast.LENGTH_SHORT).show();
                refreshDatabaseSyncOnly(); // Use the new sync-only method
            } else {
                swipeRefreshLayout.setRefreshing(false);
                showImportPage();
            }
        });

        // Customize colors to match your app theme
        swipeRefreshLayout.setColorSchemeColors(
                ContextCompat.getColor(this, R.color.primaryColor),
                ContextCompat.getColor(this, android.R.color.holo_blue_light),
                ContextCompat.getColor(this, android.R.color.holo_green_light)
        );

        // Set background color to match your app
        swipeRefreshLayout.setProgressBackgroundColorSchemeColor(
                ContextCompat.getColor(this, android.R.color.black)
        );

        // FIX: Add sensitivity settings to prevent accidental triggers
        swipeRefreshLayout.setDistanceToTriggerSync(250); // Increase from default 120
        swipeRefreshLayout.setProgressViewOffset(true, 0, 100);
    }

    private void setupCalculationListener() {
        // Since MainActivity already implements CalculationProgressListener,
        // we just need to ensure the calculation manager uses this activity as the listener
        AppLogger.d(TAG, "Calculation listener setup complete");
    }

    private void setupDeviceSpinner() {
        deviceSpinner = findViewById(R.id.device_spinner);

        // Simple ArrayAdapter with standard layouts
        deviceAdapter = new ArrayAdapter<>(
                this,
                R.layout.spinner_item,
                new ArrayList<>()
        );

        deviceAdapter.setDropDownViewResource(R.layout.spinner_dropdown_item);
        deviceSpinner.setAdapter(deviceAdapter);

        // Set dropdown background color programmatically
        deviceSpinner.setPopupBackgroundResource(android.R.color.black);

        deviceSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                // Simple check: if calculation is already running, don't start another one
                if (isCalculationInProgress) {
                    AppLogger.d(TAG, "Calculation in progress, skipping device change");
                    return;
                }

                if (position != AdapterView.INVALID_POSITION && deviceNames != null) {
                    String selectedDevice = deviceAdapter.getItem(position);
                    if (selectedDevice != null && !selectedDevice.equals(currentDevice)) {
                        AppLogger.d(TAG, "User selected device: " + selectedDevice + " (was: " + currentDevice + ")");
                        // Handle device selection
                        if (selectedDevice.equals(getString(R.string.all_devices_merged))) {
                            currentDevice = CURRENT_DEVICE_ALL_MERGED;
                        } else {
                            currentDevice = selectedDevice;
                        }
                        saveDeviceSelection(currentDevice);
                        startPaiCalculation();
                    }
                }
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {
                // Do nothing
            }
        });
    }

    // Menu handling
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main_menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(@NonNull MenuItem item) {
        return menuManager.handleMenuAction(item);
    }

    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        AppLogger.d(TAG, "=== MainActivity onSaveInstanceState - activity being destroyed ===");

        // Save critical state
        if (currentDevice != null) {
            outState.putString("currentDevice", currentDevice);
        }

        // Save that we're in the settings initialization flow
        if (!hasSetHeartRates() && currentDevice != null) {
            outState.putBoolean("settings_initialization_in_progress", true);
            AppLogger.d(TAG, "Saved settings initialization state");
        }
    }

    @Override
    protected void onRestoreInstanceState(@NonNull Bundle savedInstanceState) {
        super.onRestoreInstanceState(savedInstanceState);

        // Restore current device from saved state
        if (currentDevice == null && savedInstanceState.containsKey("currentDevice")) {
            currentDevice = savedInstanceState.getString("currentDevice");
            AppLogger.d(TAG, "Restored currentDevice: " + currentDevice);
        }
    }

    @Override
    protected void onResume() {
        super.onResume();

        if (dataManager.hasSettingsChanged() || dataManager.hasSleepSettingsChanged()) {
            if (currentDevice != null) {
                startPaiCalculation(); // Use parameterless version for settings changes
                Toast.makeText(this, "Settings changed - recalculating", Toast.LENGTH_SHORT).show();
            } else if (dataManager.hasDatabasePath()) {
                // This is the initial setup case - we have a database but no device selected yet
                // Reload the database to trigger device selection and calculations
                Uri databaseUri = dataManager.getDatabaseUri();
                if (databaseUri != null) {
                    processSelectedDatabase(databaseUri);
                }
            } else {
                Toast.makeText(this, "Settings changed - please select a database first", Toast.LENGTH_SHORT).show();
            }
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        // Don't close DataManager here - it's application-scoped
        // Only shutdown calculation manager
        if (calculationManager != null) {
            calculationManager.shutdown();
        }
        AppLogger.d(TAG, "MainActivity onDestroy - DataManager remains alive");
    }

    private void handleNoHeartRateData(String deviceName) {
        runOnUiThread(() -> {
            progressManager.showProgress(false);
            swipeRefreshLayout.setRefreshing(false);
            isCalculationInProgress = false;

            Toast.makeText(this,
                    "No heart rate data available for " + deviceName + ". This device may not support heart rate monitoring.",
                    Toast.LENGTH_LONG).show();
        });
    }

}