package org.ojrandom.paiesque;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.MenuItem;
import android.view.View;
import android.widget.*;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import org.ojrandom.paiesque.logging.AppLogger;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;

public class LogViewerActivity extends AppCompatActivity implements AppLogger.LogUpdateListener {
    private static final String TAG = "LogViewerActivity";

    private LogAdapter adapter;
    private AppLogger.Level currentFilter = null;
    private String currentSearchQuery = "";
    private RecyclerView logRecyclerView;
    private ProgressBar progressBar;
    private TextView emptyStateText;

    // Spinner references for persistence
    private Spinner logLevelSpinner;
    private Spinner filterSpinner;
    private EditText searchEditText;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_log_viewer);

        if (getSupportActionBar() != null) {
            getSupportActionBar().setTitle(R.string.title_log_viewer);
            getSupportActionBar().setDisplayHomeAsUpEnabled(true);
        }

        setupViews();
        setupLogLevelControl();
        setupFilterSpinner();
        setupSearch();
        setupButtons();

        // Load saved filters
        loadSavedFilters();

        refreshLogs();
    }

    @Override
    protected void onStart() {
        super.onStart();
        AppLogger.addListener(this);
    }

    @Override
    protected void onStop() {
        super.onStop();
        AppLogger.removeListener(this);
    }

    @Override
    protected void onPause() {
        super.onPause();
        // Save filters when activity is paused
        saveFilters();
    }

    private void setupViews() {
        logRecyclerView = findViewById(R.id.log_recycler_view);
        progressBar = findViewById(R.id.progress_bar);
        emptyStateText = findViewById(R.id.empty_state_text);

        adapter = new LogAdapter();
        logRecyclerView.setLayoutManager(new LinearLayoutManager(this));
        logRecyclerView.setAdapter(adapter);
    }

    private void setupFilterSpinner() {
        filterSpinner = findViewById(R.id.filter_spinner);

        String[] levels = {
                getString(R.string.filter_all_levels),
                getString(R.string.filter_verbose),
                getString(R.string.filter_debug),
                getString(R.string.filter_info),
                getString(R.string.filter_warn),
                getString(R.string.filter_error)
        };

        ArrayAdapter<String> spinnerAdapter = new ArrayAdapter<>(
                this, android.R.layout.simple_spinner_item, levels);
        spinnerAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        filterSpinner.setAdapter(spinnerAdapter);

        filterSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                currentFilter = position == 0 ? null : AppLogger.Level.values()[position - 1];
                refreshLogs();
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {}
        });
    }

    private void setupSearch() {
        searchEditText = findViewById(R.id.search_edit_text);
        searchEditText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {}

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {}

            @Override
            public void afterTextChanged(Editable s) {
                currentSearchQuery = s.toString();
                refreshLogs();
            }
        });
    }

    private void setupButtons() {
        Button copyButton = findViewById(R.id.copy_button);
        Button shareButton = findViewById(R.id.share_button);
        Button saveButton = findViewById(R.id.save_button);
        Button clearButton = findViewById(R.id.clear_button);

        copyButton.setOnClickListener(v -> copyLogsToClipboard());
        shareButton.setOnClickListener(v -> shareLogs());
        saveButton.setOnClickListener(v -> saveLogsToFile());
        clearButton.setOnClickListener(v -> clearLogs());
    }

    private void loadSavedFilters() {
        // Load saved filter level
        AppLogger.Level savedFilterLevel = AppLogger.getSavedFilterLevel();
        if (savedFilterLevel != null) {
            int position = savedFilterLevel.ordinal() + 1; // +1 because position 0 is "All"
            if (position >= 0 && position < filterSpinner.getCount()) {
                filterSpinner.setSelection(position);
                currentFilter = savedFilterLevel;
            }
        } else {
            // If no saved filter, select "All"
            filterSpinner.setSelection(0);
        }

        // Load saved search query
        String savedSearchQuery = AppLogger.getSavedSearchQuery();
        if (savedSearchQuery != null && !savedSearchQuery.isEmpty()) {
            searchEditText.setText(savedSearchQuery);
            currentSearchQuery = savedSearchQuery;
        }
    }

    private void saveFilters() {
        // Save filter level
        AppLogger.saveFilterLevel(currentFilter);

        // Save search query
        AppLogger.saveSearchQuery(currentSearchQuery);
    }

    private void refreshLogs() {
        progressBar.setVisibility(View.VISIBLE);

        new Thread(() -> {
            List<AppLogger.LogEntry> logs = AppLogger.getLogs(currentFilter, currentSearchQuery);

            runOnUiThread(() -> {
                adapter.submitList(logs);
                progressBar.setVisibility(View.GONE);

                // Show empty state if no logs
                if (logs.isEmpty()) {
                    emptyStateText.setVisibility(View.VISIBLE);
                    logRecyclerView.setVisibility(View.GONE);
                } else {
                    emptyStateText.setVisibility(View.GONE);
                    logRecyclerView.setVisibility(View.VISIBLE);
                    logRecyclerView.scrollToPosition(0);
                }
            });
        }).start();
    }

    private void copyLogsToClipboard() {
        List<AppLogger.LogEntry> logs = AppLogger.getLogs(currentFilter, currentSearchQuery);
        String logText = AppLogger.formatLogsAsText(logs);

        ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
        ClipData clip = ClipData.newPlainText("App Logs", logText);
        clipboard.setPrimaryClip(clip);

        Toast.makeText(this, R.string.toast_logs_copied, Toast.LENGTH_SHORT).show();
    }

    private void shareLogs() {
        List<AppLogger.LogEntry> logs = AppLogger.getLogs(currentFilter, currentSearchQuery);
        String logText = AppLogger.formatLogsAsText(logs);

        Intent intent = new Intent(Intent.ACTION_SEND);
        intent.setType("text/plain");
        intent.putExtra(Intent.EXTRA_SUBJECT, getString(R.string.share_subject));
        intent.putExtra(Intent.EXTRA_TEXT, logText);

        startActivity(Intent.createChooser(intent, getString(R.string.share_chooser_title)));
    }

    private void saveLogsToFile() {
        new Thread(() -> {
            List<AppLogger.LogEntry> logs = AppLogger.getLogs(currentFilter, currentSearchQuery);
            String logText = AppLogger.formatLogsAsText(logs);

            try {
                String fileName = "paiesque_logs_" +
                        new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.getDefault()).format(new Date()) + ".txt";

                File file = new File(getExternalFilesDir(null), fileName);
                FileOutputStream fos = new FileOutputStream(file);
                fos.write(logText.getBytes());
                fos.close();

                runOnUiThread(() ->
                        Toast.makeText(this, getString(R.string.toast_logs_saved, file.getAbsolutePath()), Toast.LENGTH_LONG).show()
                );
            } catch (IOException e) {
                runOnUiThread(() ->
                        Toast.makeText(this, getString(R.string.toast_logs_save_failed, e.getMessage()), Toast.LENGTH_LONG).show()
                );
                AppLogger.e(TAG, "Failed to save logs to file", e);
            }
        }).start();
    }

    private void clearLogs() {
        AppLogger.clearLogs();
        refreshLogs();
        Toast.makeText(this, R.string.toast_logs_cleared, Toast.LENGTH_SHORT).show();
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == android.R.id.home) {
            finish();
            return true;
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    public void onLogUpdated(AppLogger.LogEntry entry) {
        runOnUiThread(() -> {
            // Check if entry level is at least as severe as currentFilter
            boolean levelMatch = currentFilter == null ||
                    entry.level.ordinal() >= currentFilter.ordinal();
            boolean searchMatch = currentSearchQuery.isEmpty() ||
                    entry.contains(currentSearchQuery);

            if (levelMatch && searchMatch) {
                adapter.addLogEntry(entry);
                emptyStateText.setVisibility(View.GONE);
                logRecyclerView.setVisibility(View.VISIBLE);
            }
        });
    }

    @Override
    public void onLogsCleared() {
        runOnUiThread(this::refreshLogs);
    }

    private void setupLogLevelControl() {
        logLevelSpinner = findViewById(R.id.log_level_spinner);

        // Create adapter with hardcoded levels
        ArrayAdapter<String> adapter = new ArrayAdapter<>(
                this, android.R.layout.simple_spinner_item,
                new String[]{"VERBOSE", "DEBUG", "INFO", "WARN", "ERROR"});
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        logLevelSpinner.setAdapter(adapter);

        // Set current selection
        AppLogger.Level currentLevel = AppLogger.getLogLevel();
        int position = getPositionFromLevel(currentLevel);
        logLevelSpinner.setSelection(position);

        logLevelSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                AppLogger.Level selectedLevel = getLevelFromPosition(position);

                AppLogger.setLogLevel(selectedLevel);

                Toast.makeText(LogViewerActivity.this,
                        "Log level set to: " + selectedLevel.name(), Toast.LENGTH_SHORT).show();
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {}
        });
    }

    private AppLogger.Level getLevelFromPosition(int position) {
        switch (position) {
            case 0: return AppLogger.Level.VERBOSE;
            case 1: return AppLogger.Level.DEBUG;
            case 2: return AppLogger.Level.INFO;
            case 3: return AppLogger.Level.WARN;
            case 4: return AppLogger.Level.ERROR;
            default: return AppLogger.Level.DEBUG;
        }
    }

    private int getPositionFromLevel(AppLogger.Level level) {
        switch (level) {
            case VERBOSE: return 0;
            case DEBUG: return 1;
            case INFO: return 2;
            case WARN: return 3;
            case ERROR: return 4;
            default: return 1;
        }
    }
}