package org.ojrandom.paiesque.ui;

import android.content.res.Configuration;
import android.util.Log;
import android.view.ViewTreeObserver;
import android.widget.TextView;
import com.caverock.androidsvg.SVG;
import com.caverock.androidsvg.SVGImageView;

import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.svg.ChartSvgGenerator;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class RhrChartManager {
    private static final String TAG = "RhrChartManager";

    private final SVGImageView weekChartView;
    private final SVGImageView monthChartView;
    private final SVGImageView allTimeChartView;
    private final TextView allTimeDateRangeView;
    private final ExecutorService executor;
    private final UiThreadRunner uiThreadRunner;

    private TreeMap<LocalDate, Integer> currentData;

    public RhrChartManager(SVGImageView weekChartView, SVGImageView monthChartView,
                           SVGImageView allTimeChartView, TextView allTimeDateRangeView,
                           UiThreadRunner uiThreadRunner) {
        this.weekChartView = weekChartView;
        this.monthChartView = monthChartView;
        this.allTimeChartView = allTimeChartView;
        this.allTimeDateRangeView = allTimeDateRangeView;
        this.uiThreadRunner = uiThreadRunner;
        this.executor = Executors.newSingleThreadExecutor();

        setupChartViewListeners();
        setupDynamicResizing();
    }

    private void setupChartViewListeners() {
        ViewTreeObserver.OnGlobalLayoutListener layoutListener = new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                // Charts are now measured, we can use their actual dimensions
                weekChartView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                monthChartView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                allTimeChartView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
            }
        };

        weekChartView.getViewTreeObserver().addOnGlobalLayoutListener(layoutListener);
        monthChartView.getViewTreeObserver().addOnGlobalLayoutListener(layoutListener);
        allTimeChartView.getViewTreeObserver().addOnGlobalLayoutListener(layoutListener);
    }

    public void updateDailyRHRData(TreeMap<LocalDate, Integer> historicalRHR) {
        this.currentData = historicalRHR; // Store data for regeneration
        regenerateChartsWithCurrentData();
    }

    public void onConfigurationChanged() {
        // Regenerate charts with new dimensions when orientation changes
        if (currentData != null) {
            regenerateChartsWithCurrentData();
        }
    }

    private void regenerateChartsWithCurrentData() {
        executor.execute(() -> {
            try {
                List<Map<String, String>> chartData = convertToChartData(currentData);

                // ALWAYS get fresh dimensions
                int weekWidth = getMeasuredWidth(weekChartView);
                int weekHeight = getMeasuredHeight(weekChartView);
                int monthWidth = getMeasuredWidth(monthChartView);
                int monthHeight = getMeasuredHeight(monthChartView);
                int allTimeWidth = getMeasuredWidth(allTimeChartView);
                int allTimeHeight = getMeasuredHeight(allTimeChartView);

                // Generate charts with current dimensions
                String weekSvg = ChartSvgGenerator.generate7DayRhrSvg(chartData, weekWidth, weekHeight);
                String monthSvg = ChartSvgGenerator.generate30DayRhrSvg(chartData, monthWidth, monthHeight);
                String allTimeSvg = ChartSvgGenerator.generateAllTimeRhrSvg(chartData, allTimeWidth, allTimeHeight);

                uiThreadRunner.runOnUiThread(() -> {
                    try {
                        setSvgFromString(weekChartView, weekSvg);
                        setSvgFromString(monthChartView, monthSvg);
                        setSvgFromString(allTimeChartView, allTimeSvg);

                        if (allTimeDateRangeView != null && !chartData.isEmpty()) {
                            String firstDate = chartData.get(0).get("date");
                            String lastDate = chartData.get(chartData.size() - 1).get("date");
                            allTimeDateRangeView.setText(firstDate + " to " + lastDate);
                        }
                    } catch (Exception e) {
                        AppLogger.e(TAG, "Error setting SVG: " + e.getMessage());
                    }
                });
            } catch (Exception e) {
                AppLogger.e(TAG, "Error generating charts: " + e.getMessage());
            }
        });
    }

    private int getMeasuredWidth(SVGImageView chartView) {
        int width = chartView.getWidth();
        if (width <= 0) {
            width = chartView.getContext().getResources().getDisplayMetrics().widthPixels - 32;
        }
        return Math.max(width, 300);
    }

    private int getMeasuredHeight(SVGImageView chartView) {
        int width = chartView.getWidth();
        int height = chartView.getHeight();

        if (width <= 0 || height <= 0) {
            // Calculate based on 2/3 aspect ratio (height = 2/3 of width)
            int screenWidth = chartView.getContext().getResources().getDisplayMetrics().widthPixels;
            int availableWidth = screenWidth - 32; // Account for padding

            // Height is 2/3 of width for portrait, 1/3 for landscape
            boolean isLandscape = chartView.getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE;

            if (isLandscape) {
                height = Math.min(availableWidth / 3, 300); // Shorter in landscape
            } else {
                height = (int) (availableWidth * 0.67); // 2/3 of width in portrait
            }
        } else {
            // Use the actual measured height
            // No modification needed since we want the container's natural height
        }

        return Math.max(height, 200);
    }

    private void setupDynamicResizing() {
        // Create listeners for each chart view
        setupChartResizing(weekChartView);
        setupChartResizing(monthChartView);
        setupChartResizing(allTimeChartView);
    }

    private void setupChartResizing(SVGImageView chartView) {
        ViewTreeObserver.OnGlobalLayoutListener resizeListener = new ViewTreeObserver.OnGlobalLayoutListener() {
            private int lastWidth = 0;
            private int lastHeight = 0;

            @Override
            public void onGlobalLayout() {
                int currentWidth = chartView.getWidth();
                int currentHeight = chartView.getHeight();

                // Regenerate if dimensions changed significantly
                boolean widthChanged = Math.abs(currentWidth - lastWidth) > 50;
                boolean heightChanged = Math.abs(currentHeight - lastHeight) > 50;

                if ((widthChanged || heightChanged) && currentWidth > 0 && currentHeight > 0) {
                    lastWidth = currentWidth;
                    lastHeight = currentHeight;

                    // Small delay to avoid rapid regeneration during layout
                    chartView.postDelayed(() -> {
                        if (currentData != null) {
                            regenerateChartsWithCurrentData();
                        }
                    }, 100);
                }
            }
        };

        chartView.getViewTreeObserver().addOnGlobalLayoutListener(resizeListener);
    }

    private void setSvgFromString(SVGImageView imageView, String svgString) {
        try {
            SVG svg = SVG.getFromString(svgString);
            imageView.setSVG(svg);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error parsing SVG: " + e.getMessage());
            try {
                int width = getMeasuredWidth(imageView);
                int height = getMeasuredHeight(imageView);
                String emptySvg = createEmptySvg(width, height, "Chart Error");
                SVG svg = SVG.getFromString(emptySvg);
                imageView.setSVG(svg);
            } catch (Exception e2) {
                AppLogger.e(TAG, "Error setting empty SVG: " + e2.getMessage());
            }
        }
    }

    private String createEmptySvg(int width, int height, String message) {
        return "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" +
                "<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"" + width +
                "\" height=\"" + height + "\" viewBox=\"0 0 " + width + " " + height + "\">" +
                "<rect x=\"0\" y=\"0\" width=\"" + width + "\" height=\"" + height +
                "\" fill=\"#000000\"/>" +
                "<text x=\"" + (width / 2) + "\" y=\"" + (height / 2) +
                "\" text-anchor=\"middle\" fill=\"#CCCCCC\" font-size=\"14\">" + message + "</text>" +
                "</svg>";
    }

    private List<Map<String, String>> convertToChartData(TreeMap<LocalDate, Integer> historicalRHR) {
        List<Map<String, String>> result = new ArrayList<>();

        for (Map.Entry<LocalDate, Integer> entry : historicalRHR.entrySet()) {
            Map<String, String> dataPoint = new java.util.HashMap<>();
            dataPoint.put("date", entry.getKey().toString());
            dataPoint.put("resting_heart_rate", String.valueOf(entry.getValue()));
            result.add(dataPoint);
        }

        return result;
    }

    public void cleanup() {
        if (executor != null && !executor.isShutdown()) {
            executor.shutdownNow();
        }
    }

    public interface UiThreadRunner {
        void runOnUiThread(Runnable action);
    }
}