package org.ojrandom.paiesque.rhr;

import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.time.LocalDate;
import java.util.TreeMap;

public class RHRSummaryCalculator {
    private static final String TAG = "RHRSummaryCalculator";

    public RHRSummary getSummary(TreeMap<LocalDate, Integer> historicalRHR, int deviceId) {
        AppLogger.d(TAG, "Calculating RHR summary for device: " + deviceId);

        if (historicalRHR.isEmpty()) {
            AppLogger.d(TAG, "No historical RHR data available");
            return RHRSummary.empty();
        }

        LocalDate today = LocalDate.now();

        // Calculate metrics using the provided historical data
        int lastWeekRHR = calculateWeightedAverage(historicalRHR, today.minusDays(7), today);
        int lastMonthRHR = calculateWeightedAverage(historicalRHR, today.minusDays(30), today);
        int currentBaselineRHR = calculateBestRHR(historicalRHR, today.minusDays(30), today);

        AppLogger.i(TAG, String.format("RHR Summary - Week: %d, Month: %d, Baseline: %d",
                lastWeekRHR, lastMonthRHR, currentBaselineRHR));

        return new RHRSummary(lastWeekRHR, lastMonthRHR, currentBaselineRHR);
    }

    /**
     * Calculate weighted average RHR in date range (more recent = higher weight)
     */
    private int calculateWeightedAverage(TreeMap<LocalDate, Integer> historicalRHR,
                                         LocalDate startDate, LocalDate endDate) {
        double sum = 0;
        double totalWeight = 0;
        int validDays = 0;

        for (java.util.Map.Entry<LocalDate, Integer> entry : historicalRHR.entrySet()) {
            LocalDate entryDate = entry.getKey();
            if (!entryDate.isBefore(startDate) && !entryDate.isAfter(endDate)) {
                int rhr = entry.getValue();
                // More recent days get higher weight
                long daysAgo = java.time.temporal.ChronoUnit.DAYS.between(entryDate, endDate);
                double weight = 1.0 / (daysAgo + 1); // Recent days have higher weight

                sum += rhr * weight;
                totalWeight += weight;
                validDays++;
            }
        }

        if (validDays == 0 || totalWeight == 0) {
            return -1;
        }

        return (int) Math.round(sum / totalWeight);
    }

    /**
     * Calculate best (lowest) RHR in date range
     */
    private int calculateBestRHR(TreeMap<LocalDate, Integer> historicalRHR,
                                 LocalDate startDate, LocalDate endDate) {
        int bestRHR = -1;

        for (java.util.Map.Entry<LocalDate, Integer> entry : historicalRHR.entrySet()) {
            LocalDate entryDate = entry.getKey();
            if (!entryDate.isBefore(startDate) && !entryDate.isAfter(endDate)) {
                int rhr = entry.getValue();
                if (bestRHR == -1 || rhr < bestRHR) {
                    bestRHR = rhr;
                }
            }
        }

        return bestRHR;
    }
}