package org.ojrandom.paiesque.rhr;

import android.content.Context;

import org.ojrandom.paiesque.data.DailyHeartRates;
import org.ojrandom.paiesque.data.SettingsManager;
import org.ojrandom.paiesque.logging.AppLogger;

import java.time.LocalDate;
import java.util.*;

public class RHRService {
    private static final String TAG = "RHRService";
    private final Map<LocalDate, Integer> historicalRHR;
    private final SimpleRHRCalculator calculator;
    private final SettingsManager settingsManager; // Use SettingsManager instead of SleepSettingsManager

    public RHRService(Map<LocalDate, Integer> historicalRHR, SettingsManager settingsManager) {
        this.historicalRHR = historicalRHR != null ? new TreeMap<>(historicalRHR) : new TreeMap<>();
        this.calculator = new SimpleRHRCalculator();
        this.settingsManager = settingsManager; // Use the passed SettingsManager
    }

    public Map<LocalDate, RhrResult> calculateEnhancedDailyRHR(
            Map<LocalDate, DailyHeartRates> dailyData) {

        Map<LocalDate, RhrResult> results = new TreeMap<>();

        // Get user's RHR window from SettingsManager
        int startHour = settingsManager.getSleepStartHour();
        int startMinute = settingsManager.getSleepStartMinute();
        int endHour = settingsManager.getSleepEndHour();
        int endMinute = settingsManager.getSleepEndMinute();

        int startMinutes = startHour * 60 + startMinute;
        int endMinutes = endHour * 60 + endMinute;

        AppLogger.d(TAG, "Calculating RHR with window: " +
                formatTime(startMinutes) + " - " + formatTime(endMinutes));

        // Calculate baseline from historical data
        HistoricalBaseline baseline = calculateHistoricalBaseline();
        AppLogger.d(TAG, "Using baseline: " + baseline);

        for (Map.Entry<LocalDate, DailyHeartRates> entry : dailyData.entrySet()) {
            LocalDate date = entry.getKey();
            List<Map<String, String>> dayData = convertToMapFormat(entry.getValue());

            RhrResult result = calculator.calculateDailyRHR(
                    dayData, date, startMinutes, endMinutes, baseline);

            if (result != null) {
                results.put(date, result);
                // Update historical data for next baseline calculation
                historicalRHR.put(date, result.getRhrValue());
            }
        }

        return results;
    }

    private HistoricalBaseline calculateHistoricalBaseline() {
        if (historicalRHR.isEmpty()) {
            return getDefaultBaseline();
        }

        List<Integer> recentRHR = getRecentRHR(30); // 30 days
        if (recentRHR.isEmpty()) {
            recentRHR = new ArrayList<>(historicalRHR.values());
        }

        Collections.sort(recentRHR);
        int median = recentRHR.get(recentRHR.size() / 2);
        int lowerQuartile = recentRHR.get(recentRHR.size() / 4);

        // Conservative personalized range
        int personalizedMin = Math.max(30, median - 15);
        int personalizedMax = Math.min(100, median + 20);

        return new HistoricalBaseline(median, 0, 30, 120, personalizedMin, personalizedMax);
    }

    private List<Integer> getRecentRHR(int days) {
        LocalDate cutoff = LocalDate.now().minusDays(days);
        List<Integer> recent = new ArrayList<>();
        for (Map.Entry<LocalDate, Integer> entry : historicalRHR.entrySet()) {
            if (!entry.getKey().isBefore(cutoff)) {
                recent.add(entry.getValue());
            }
        }
        return recent;
    }

    private HistoricalBaseline getDefaultBaseline() {
        return new HistoricalBaseline(65, 15, 30, 100, 30, 90);
    }

    private List<Map<String, String>> convertToMapFormat(DailyHeartRates dailyRates) {
        List<Map<String, String>> result = new ArrayList<>();
        for (DailyHeartRates.HeartRateRecord record : dailyRates.getRecords()) {
            Map<String, String> entry = new HashMap<>();
            entry.put("HEART_RATE", String.valueOf(record.getHeartRate()));
            entry.put("MINUTES_OF_DAY", String.valueOf(record.getMinutesOfDay()));
            entry.put("TIMESTAMP", String.valueOf(record.getTimestamp()));
            result.add(entry);
        }
        return result;
    }

    private String formatTime(int totalMinutes) {
        int hours = totalMinutes / 60;
        int minutes = totalMinutes % 60;
        return String.format("%02d:%02d", hours, minutes);
    }
}