package org.ojrandom.paiesque.pai;

import org.ojrandom.paiesque.AndroidLogger;
import org.ojrandom.paiesque.Logger;
import org.ojrandom.paiesque.data.HRPoint;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * Zone-based calculation of cardiovascular fitness score
 * <p>
 * IMPORTANT NOTE: This is NOT the PAI (Personal Activity Intelligence)
 * algorithm. It's a community implementation based on publicly available
 * information about how PAI might work. Results may differ from official PAI
 * calculations.
 * <p>
 * Key features:
 * - Exact linear interpolation between zone boundaries
 * - Zone-time accumulation pattern for performance
 * - Thread-safe for concurrent read operations (instance methods)
 * - Pre-calculated caches for O(1) lookups
 * - Batch processing for efficiency
 */
public class PAIesqueCalculator {
    private static final String TAG = "PAICalculator";

    // --- Constants ---
    // Note: All constants are static final for immutability and thread safety
    private static final int MIN_HR = 30;                    // Minimum valid heart rate
    private static final int MAX_HR = 250;                   // Maximum valid heart rate
    private static final long MAX_INTERPOLATION_SECONDS = 300L; // 5 minutes max gap
    private static final double EPSILON = 0.001;            // Tolerance for floating-point comparisons

    // Zone definitions (ordered from lowest to highest intensity)
    public static final String ZONE_BELOW_THRESHOLD = "zone_below_threshold";
    public static final String ZONE_50_59 = "zone_50_59";
    public static final String ZONE_60_69 = "zone_60_69";
    public static final String ZONE_70_79 = "zone_70_79";
    public static final String ZONE_80_89 = "zone_80_89";
    public static final String ZONE_90_PLUS = "zone_90_plus";

    // Zone definitions in order (0-5)
    private static final String[] ZONES = {
            ZONE_BELOW_THRESHOLD,  // Zone 0: < 50% intensity (0 PAI/min)
            ZONE_50_59,            // Zone 1: 50-59% intensity (0.2 PAI/min)
            ZONE_60_69,            // Zone 2: 60-69% intensity (0.5 PAI/min)
            ZONE_70_79,            // Zone 3: 70-79% intensity (1.0 PAI/min)
            ZONE_80_89,            // Zone 4: 80-89% intensity (1.5 PAI/min)
            ZONE_90_PLUS           // Zone 5: 90%+ intensity (2.0 PAI/min)
    };

    // PAI per second for each zone (pre-calculated from PAI per minute rates)
    // Using double[] instead of Map for better cache locality and performance
    private static final double[] ZONE_PAI_PER_SECOND = {
            0.0,                    // Zone 0: below threshold (0 PAI/min = 0 PAI/sec)
            0.2 / 60.0,            // Zone 1: 0.2 PAI/min = ~0.003333 PAI/sec
            0.5 / 60.0,            // Zone 2: 0.5 PAI/min = ~0.008333 PAI/sec
            1.0 / 60.0,            // Zone 3: 1.0 PAI/min = ~0.016667 PAI/sec
            1.5 / 60.0,            // Zone 4: 1.5 PAI/min = 0.025 PAI/sec
            2.0 / 60.0             // Zone 5: 2.0 PAI/min = ~0.033333 PAI/sec
    };

    // --- Instance variables (each instance is thread-safe for its own data) ---
    private final int restingHR;          // User's resting heart rate
    private final int maxHR;              // User's maximum heart rate
    private final double minEarningHR;    // Minimum HR to earn PAI (50% intensity)
    private final boolean useExperimentalTimeDecay;
    private final DateTimeFormatter formatter;
    private final Logger logger;

    // Pre-calculated zone boundaries in HR values (instance-specific)
    private final double[] zoneBoundariesHR;

    // Fast zone lookup cache: array index = HR - MIN_HR, value = zone (0-5)
    // Size: MAX_HR - MIN_HR + 1 = 250 - 30 + 1 = 221 elements
    private final int[] zoneCache;

    // Volatile lists for thread visibility (if shared across threads)
    private volatile List<Map<String, String>> paiPoints;

    // --- Constructors ---

    public PAIesqueCalculator(int restingHR, int maxHR, boolean useTimeDecay) {
        this(restingHR, maxHR, useTimeDecay, new AndroidLogger());
    }

    /**
     * Primary constructor - validates inputs and initializes caches
     *
     * @param restingHR User's resting heart rate (bpm)
     * @param maxHR User's maximum heart rate (bpm)
     * @param useTimeDecay If true, applies time decay to 7-day rolling PAI
     * @param logger Logger instance for debugging
     * @throws IllegalArgumentException if restingHR >= maxHR
     */
    public PAIesqueCalculator(int restingHR, int maxHR, boolean useTimeDecay, Logger logger) {
        if (restingHR >= maxHR) {
            throw new IllegalArgumentException(
                    String.format("restingHR (%d) must be less than maxHR (%d)", restingHR, maxHR)
            );
        }

        this.logger = logger;
        this.restingHR = Math.max(restingHR, MIN_HR);
        this.maxHR = Math.min(Math.max(maxHR, this.restingHR + 1), MAX_HR);
        this.useExperimentalTimeDecay = useTimeDecay;
        this.formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        this.paiPoints = Collections.synchronizedList(new ArrayList<>());

        // Pre-calculate minimum HR to earn PAI (50% intensity threshold)
        this.minEarningHR = restingHR + 0.5 * (this.maxHR - this.restingHR);

        // Pre-calculate zone boundaries in HR values (once per instance)
        this.zoneBoundariesHR = calculateZoneBoundariesHR();

        // Initialize zone cache for O(1) lookups (221-element array)
        this.zoneCache = new int[MAX_HR - MIN_HR + 1];
        initializeZoneCache();

        logger.debug(TAG, String.format(
                "PAICalculator initialized: restingHR=%d, maxHR=%d, minEarningHR=%.1f",
                this.restingHR, this.maxHR, this.minEarningHR
        ));
    }

    /**
     * Pre-calculates HR values for all zone boundaries.
     * Returns array of 6 boundaries: [50%, 60%, 70%, 80%, 90%, 100%]
     */
    private double[] calculateZoneBoundariesHR() {
        double hrRange = maxHR - restingHR;
        return new double[] {
                minEarningHR,                   // 50% threshold
                restingHR + 0.6 * hrRange,     // 60%
                restingHR + 0.7 * hrRange,     // 70%
                restingHR + 0.8 * hrRange,     // 80%
                restingHR + 0.9 * hrRange,     // 90%
                maxHR                          // 100% (max HR)
        };
    }

    /**
     * Initializes the zone cache array.
     * For each HR from MIN_HR to MAX_HR, pre-computes which zone it belongs to.
     * This eliminates repeated floating-point calculations during processing.
     */
    private void initializeZoneCache() {
        for (int hr = MIN_HR; hr <= MAX_HR; hr++) {
            zoneCache[hr - MIN_HR] = calculateZoneIndex(hr);
        }
    }

    /**
     * Calculates zone index (0-5) for a given heart rate without cache.
     * Used only during cache initialization.
     *
     * @param heartRate Heart rate in bpm
     * @return Zone index (0-5) or 0 for invalid HR
     */
    private int calculateZoneIndex(int heartRate) {
        if (heartRate < MIN_HR || heartRate > MAX_HR) {
            return 0; // Default to zone 0 for invalid HR
        }

        // Calculate intensity: (HR - resting) / (max - resting)
        double intensity = (heartRate - restingHR) / (double) (maxHR - restingHR);

        // Clamp to [0, 1] range to handle edge cases
        intensity = Math.max(0.0, Math.min(intensity, 1.0));

        // Determine zone based on intensity thresholds
        if (intensity < 0.5) return 0;
        if (intensity < 0.6) return 1;
        if (intensity < 0.7) return 2;
        if (intensity < 0.8) return 3;
        if (intensity < 0.9) return 4;
        return 5; // 90% and above
    }

    /**
     * Gets zone index with cache lookup - O(1) operation.
     * Much faster than recalculating intensity each time.
     *
     * @param heartRate Heart rate in bpm
     * @return Zone index (0-5) or 0 for invalid HR
     */
    private int getZoneIndex(int heartRate) {
        if (heartRate < MIN_HR || heartRate > MAX_HR) {
            return 0;
        }
        return zoneCache[heartRate - MIN_HR]; // Direct array access
    }

    // --- Public API (Thread-safe for read-only operations) ---

    /**
     * Enhanced result with zone breakdown for analysis.
     * Immutable for thread safety.
     */
    public static class DailyPAIResult {
        private final double totalPAI;
        private final Map<String, Double> paiByZone;
        private final Map<String, Double> secondsByZone;

        public DailyPAIResult(double totalPAI, Map<String, Double> paiByZone,
                              Map<String, Double> secondsByZone) {
            this.totalPAI = totalPAI;
            this.paiByZone = Collections.unmodifiableMap(new HashMap<>(paiByZone));
            this.secondsByZone = Collections.unmodifiableMap(new HashMap<>(secondsByZone));
        }

        public double getTotalPAI() { return totalPAI; }
        public Map<String, Double> getPaiByZone() { return paiByZone; }
        public Map<String, Double> getSecondsByZone() { return secondsByZone; }

        @Override
        public String toString() {
            return String.format("DailyPAIResult{total=%.2f, zones=%s}", totalPAI, paiByZone);
        }
    }

    // --- Core Calculation Methods ---

    /**
     * Calculates PAI with exact interpolation for a single day.
     * This is the core algorithm that replaces the original weight-based approach.
     * <p>
     * Key optimization: Uses zone-time accumulation pattern:
     * 1. Accumulate total time spent in each zone (using arrays for speed)
     * 2. Multiply once at the end (batch processing)
     *
     * @param dayPoints HR points for a single day, sorted by time
     * @return DailyPAIResult with total PAI and zone breakdown
     */
    public DailyPAIResult calculateDailyPAIWithZones(List<HRPoint> dayPoints) {
        if (dayPoints == null || dayPoints.isEmpty() || dayPoints.size() < 2) {
            logger.debug(TAG, "Insufficient data for PAI calculation");
            return new DailyPAIResult(0.0, initializeZoneMap(), initializeZoneMap());
        }

        // Use primitive array for maximum performance (one slot per zone)
        double[] zoneTimeSeconds = new double[ZONES.length];
        int intervalsProcessed = 0;
        int intervalsWithPAI = 0;

        // Process each consecutive pair of HR points
        for (int i = 1; i < dayPoints.size(); i++) {
            HRPoint prev = dayPoints.get(i - 1);
            HRPoint curr = dayPoints.get(i);

            long secondsBetween = Duration.between(prev.getTime(), curr.getTime()).getSeconds();

            // Skip invalid intervals (negative time or too large gap)
            if (secondsBetween <= 0 || secondsBetween > MAX_INTERPOLATION_SECONDS) {
                continue;
            }

            intervalsProcessed++;

            // Core: Accumulate time in each zone using exact interpolation
            accumulateZoneTimesExact(
                    prev.getHeartRate(),
                    curr.getHeartRate(),
                    secondsBetween,
                    zoneTimeSeconds
            );

            // Calculate PAI for this interval (for tracking/debugging)
            double intervalPAI = calculateIntervalPAIForTracking(prev, curr, secondsBetween);
            if (intervalPAI > 0.001) {
                intervalsWithPAI++;
                trackPAIPoint(curr, intervalPAI);
            }
        }

        // Batch calculation: Multiply accumulated times by PAI rates
        Map<String, Double> paiByZone = new HashMap<>();
        Map<String, Double> secondsByZone = new HashMap<>();
        double totalPAI = 0.0;

        for (int zone = 0; zone < ZONES.length; zone++) {
            double time = zoneTimeSeconds[zone];
            double pai = ZONE_PAI_PER_SECOND[zone] * time; // Single multiplication per zone

            secondsByZone.put(ZONES[zone], time);
            paiByZone.put(ZONES[zone], pai);
            totalPAI += pai;
        }

        logger.debug(TAG, String.format(
                "Day processed: %d intervals, %d with PAI, total PAI=%.2f",
                intervalsProcessed, intervalsWithPAI, totalPAI
        ));

        return new DailyPAIResult(totalPAI, paiByZone, secondsByZone);
    }

    /**
     * Core algorithm: Accumulates time in each zone with exact boundary crossing.
     * Uses linear interpolation to find exact times when HR crosses zone boundaries.
     * <p>
     * Example: HR 130→160 (resting=60, max=200) over 120s:
     * - Zone boundaries: [130, 144, 158, 172, 186, 200]
     * - 0-56s: HR 130→144 (zone 1)
     * - 56-112s: HR 144→158 (zone 2)
     * - 112-120s: HR 158→160 (zone 3)
     * <p>
     * Compare to old weight-based method: Would allocate 40s to zone1, 80s to zone3,
     * completely missing zone2.
     */
    private void accumulateZoneTimesExact(double startHR, double endHR,
                                          double totalSeconds, double[] zoneTimeSeconds) {
        // Fast path: No HR change (common case for resting periods)
        if (Math.abs(endHR - startHR) < EPSILON) {
            int zone = getZoneIndex((int)Math.round(startHR));
            zoneTimeSeconds[zone] += totalSeconds;
            return;
        }

        double hrChangePerSecond = (endHR - startHR) / totalSeconds;
        double currentTime = 0.0;
        double currentHR = startHR;
        int currentZone = getZoneIndex((int)Math.round(currentHR));

        // Process interval segment by segment (each segment is within one zone)
        while (currentTime < totalSeconds - EPSILON) {
            // Find the next zone boundary that will be crossed
            Double nextBoundaryHR = null;
            int nextZone = currentZone;
            boolean increasing = hrChangePerSecond > 0;

            if (increasing) {
                // For increasing HR: find the next higher boundary
                // Optimization: Start search from current zone (zones are ordered)
                for (int i = currentZone; i < zoneBoundariesHR.length; i++) {
                    if (zoneBoundariesHR[i] > currentHR &&
                            zoneBoundariesHR[i] <= endHR + EPSILON) {
                        nextBoundaryHR = zoneBoundariesHR[i];
                        nextZone = (i == 5) ? 5 : i + 1; // Special case for max zone
                        break;
                    }
                }
            } else {
                // For decreasing HR: find the next lower boundary
                for (int i = currentZone; i >= 0; i--) {
                    double boundary = (i == 0) ? minEarningHR : zoneBoundariesHR[i - 1];
                    if (boundary < currentHR - EPSILON &&
                            boundary >= endHR - EPSILON) {
                        nextBoundaryHR = boundary;
                        nextZone = Math.max(0, i - 1);
                        break;
                    }
                }
            }

            double segmentEndTime;
            if (nextBoundaryHR != null) {
                // Calculate time to reach this boundary using linear interpolation
                segmentEndTime = currentTime + (nextBoundaryHR - currentHR) / hrChangePerSecond;
                segmentEndTime = Math.min(segmentEndTime, totalSeconds);
            } else {
                // No more boundaries in this direction
                segmentEndTime = totalSeconds;
            }

            // Accumulate time spent in current zone during this segment
            double segmentDuration = segmentEndTime - currentTime;
            if (segmentDuration > EPSILON) {
                zoneTimeSeconds[currentZone] += segmentDuration;
            }

            // Move to next segment
            currentTime = segmentEndTime;

            if (nextBoundaryHR != null) {
                // We crossed a boundary - use pre-calculated next zone (optimization)
                currentZone = nextZone;
                currentHR = nextBoundaryHR; // We're exactly at the boundary HR
            } else {
                // No boundary crossed (end of interval)
                currentHR = startHR + hrChangePerSecond * currentTime;
                currentZone = getZoneIndex((int)Math.round(currentHR));

                // Exit loop if we've reached the end
                if (Math.abs(currentTime - totalSeconds) < EPSILON) {
                    break;
                }
            }
        }
    }

    /**
     * Calculates interval PAI for tracking purposes.
     * Uses the same exact interpolation as the main calculation for consistency.
     */
    private double calculateIntervalPAIForTracking(HRPoint start, HRPoint end, long totalSeconds) {
        // Use same exact interpolation as real calculation (for consistency)
        double[] zoneTimeSeconds = new double[ZONES.length];
        accumulateZoneTimesExact(start.getHeartRate(), end.getHeartRate(),
                totalSeconds, zoneTimeSeconds);

        // Calculate total PAI (same logic as in calculateDailyPAIWithZones)
        double totalPAI = 0.0;
        for (int zone = 0; zone < ZONES.length; zone++) {
            totalPAI += ZONE_PAI_PER_SECOND[zone] * zoneTimeSeconds[zone];
        }
        return totalPAI;
    }

    /**
     * Tracks detailed PAI points for debugging/analysis.
     * Points are stored in paiPoints list for later inspection.
     */
    private synchronized void trackPAIPoint(HRPoint point, double paiEarned) {
        Map<String, String> row = new HashMap<>();
        row.put("timestamp", point.getTime().format(DateTimeFormatter.ISO_LOCAL_DATE_TIME));
        row.put("heartrate", String.valueOf(point.getHeartRate()));
        row.put("pai", String.format("%.6f", paiEarned));
        paiPoints.add(row);
    }

    /**
     * Initializes a zone map with all zones set to 0.0.
     * Used for empty/default results.
     */
    private Map<String, Double> initializeZoneMap() {
        Map<String, Double> map = new HashMap<>();
        for (String zone : ZONES) {
            map.put(zone, 0.0);
        }
        return map;
    }

    /**
     * Calculates 7-day rolling PAI from daily PAI values.
     * Optionally applies experimental time decay (older days count less).
     */
    public List<Map<String, String>> calculate7DayRollingPAI(Map<LocalDate, Double> dailyPAI) {
        logger.debug(TAG, "Calculating 7-day rolling PAI");

        List<Map<String, String>> rollingPAI = new ArrayList<>();

        if (dailyPAI.isEmpty()) {
            return rollingPAI;
        }

        // Get date range from data
        LocalDate startDate = dailyPAI.keySet().stream()
                .min(LocalDate::compareTo)
                .orElse(LocalDate.now());
        LocalDate endDate = LocalDate.now();

        // Calculate rolling PAI for each day in range
        for (LocalDate currentDate = startDate;
             !currentDate.isAfter(endDate);
             currentDate = currentDate.plusDays(1)) {

            double rollingSum = 0.0;
            double todaysPai = dailyPAI.getOrDefault(currentDate, 0.0);

            // Sum last 7 days (including today)
            for (int j = 0; j < 7; j++) {
                LocalDate pastDate = currentDate.minusDays(j);
                double dailyPai = dailyPAI.getOrDefault(pastDate, 0.0);

                if (useExperimentalTimeDecay) {
                    // Apply linear decay: today=100%, 6 days ago=40%
                    double weight = Math.max(1.0 - (j * 0.1), 0.4);
                    rollingSum += dailyPai * weight;
                } else {
                    // Standard: all days count equally
                    rollingSum += dailyPai;
                }
            }

            // Format result
            Map<String, String> row = new HashMap<>();
            row.put("date", currentDate.format(formatter));
            row.put("pai_day", String.valueOf((int) Math.round(todaysPai)));
            row.put("pai_7_day", String.valueOf((int) Math.round(rollingSum)));
            rollingPAI.add(row);
        }

        return rollingPAI;
    }

    /**
     * Utility method for database filtering.
     * Calculates minimum HR to include in queries (25% intensity threshold).
     * Static method - thread-safe.
     */
    public static int getDatabaseFilterHR(int restingHR, int maxHR) {
        double filterIntensity = 0.25;
        int minHR = (int) Math.round(restingHR + (filterIntensity * (maxHR - restingHR)));
        return Math.max(40, minHR); // Never lower than 40 bpm
    }
}