package org.ojrandom.paiesque.logging;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

public class AppLogger {
    private static final int MAX_LOG_ENTRIES = 10000;
    private static final CopyOnWriteArrayList<LogEntry> logEntries = new CopyOnWriteArrayList<>();
    private static final CopyOnWriteArrayList<LogUpdateListener> listeners = new CopyOnWriteArrayList<>();

    // fields for log level persistence
    private static Level currentLogLevel = Level.DEBUG; // Default level
    private static SharedPreferences sharedPreferences;
    private static final String PREFS_NAME = "AppLoggerPrefs";
    private static final String KEY_LOG_LEVEL = "log_level";
    private static final String KEY_FILTER_LEVEL = "filter_level";
    private static final String KEY_SEARCH_QUERY = "search_query";

    public enum Level { VERBOSE, DEBUG, INFO, WARN, ERROR }

    public static class LogEntry {
        public final long timestamp;
        public final Level level;
        public final String tag;
        public final String message;
        public final Throwable throwable;

        public LogEntry(long timestamp, Level level, String tag, String message, Throwable throwable) {
            this.timestamp = timestamp;
            this.level = level;
            this.tag = tag;
            this.message = message;
            this.throwable = throwable;
        }

        public boolean contains(String query) {
            return tag.toLowerCase().contains(query.toLowerCase()) ||
                    message.toLowerCase().contains(query.toLowerCase());
        }

        public boolean containsAllTerms(String[] searchTerms) {
            if (searchTerms == null || searchTerms.length == 0) {
                return true;
            }

            String searchableText = (tag + " " + message).toLowerCase();

            for (String term : searchTerms) {
                if (term != null && !term.trim().isEmpty()) {
                    if (!searchableText.contains(term.toLowerCase().trim())) {
                        return false;
                    }
                }
            }
            return true;
        }
    }

    public interface LogUpdateListener {
        void onLogUpdated(LogEntry entry);
        void onLogsCleared();
    }

    /**
     * Initialize AppLogger with context (call this from Application class)
     */
    public static void initialize(Context context) {
        debugLogLevelState("AppLogger", "Before initialization");
        sharedPreferences = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE);
        loadStoredLogLevel();
        debugLogLevelState("AppLogger", "After initialization");
    }

    public static void debugLogLevelState(String tag, String operation) {
        Log.d("AppLoggerDebug", operation + " - Current level: " + currentLogLevel +
                ", SharedPrefs: " + (sharedPreferences != null ? "available" : "null") +
                ", Stored level: " + (sharedPreferences != null ? sharedPreferences.getString(KEY_LOG_LEVEL, "NOT_FOUND") : "N/A"));
    }

    /**
     * Load the stored log level from SharedPreferences
     */
    private static void loadStoredLogLevel() {
        if (sharedPreferences != null) {
            String levelName = sharedPreferences.getString(KEY_LOG_LEVEL, Level.DEBUG.name());
            try {
                Level previousLevel = currentLogLevel;
                currentLogLevel = Level.valueOf(levelName);
                Log.d("AppLogger", "Loaded log level from SharedPreferences: " + levelName + " (was: " + previousLevel + ")");
            } catch (IllegalArgumentException e) {
                currentLogLevel = Level.DEBUG; // Fallback
                Log.e("AppLogger", "Error loading log level, using default: " + e.getMessage());
            }
        } else {
            Log.e("AppLogger", "SharedPreferences is null - cannot load log level");
        }
    }

    /**
     * Set and persist log level
     */
    public static void setLogLevel(Level level) {
        debugLogLevelState("AppLogger", "Before setLogLevel: " + level);
        currentLogLevel = level;

        if (sharedPreferences != null) {
            sharedPreferences.edit()
                    .putString(KEY_LOG_LEVEL, level.name())
                    .apply();
            debugLogLevelState("AppLogger", "After setLogLevel and save");
        } else {
            Log.e("AppLoggerDebug", "SharedPreferences is null - cannot save");
        }
    }

    /**
     * Get current log level
     */
    public static Level getLogLevel() {
        return currentLogLevel;
    }

    /**
     * Save filter level to SharedPreferences
     */
    public static void saveFilterLevel(Level level) {
        if (sharedPreferences != null) {
            String levelName = level != null ? level.name() : "ALL";
            sharedPreferences.edit()
                    .putString(KEY_FILTER_LEVEL, levelName)
                    .apply();
            Log.d("AppLogger", "Saved filter level: " + levelName);
        }
    }

    /**
     * Get saved filter level from SharedPreferences
     */
    public static Level getSavedFilterLevel() {
        if (sharedPreferences != null) {
            String levelName = sharedPreferences.getString(KEY_FILTER_LEVEL, "ALL");
            if ("ALL".equals(levelName)) {
                return null;
            }
            try {
                return Level.valueOf(levelName);
            } catch (IllegalArgumentException e) {
                return null;
            }
        }
        return null;
    }

    /**
     * Save search query to SharedPreferences
     */
    public static void saveSearchQuery(String query) {
        if (sharedPreferences != null) {
            String sanitizedQuery = query != null ? query.trim() : "";
            sharedPreferences.edit()
                    .putString(KEY_SEARCH_QUERY, sanitizedQuery)
                    .apply();
            Log.d("AppLogger", "Saved search query: \"" + sanitizedQuery + "\"");
        }
    }

    /**
     * Get saved search query from SharedPreferences
     */
    public static String getSavedSearchQuery() {
        if (sharedPreferences != null) {
            return sharedPreferences.getString(KEY_SEARCH_QUERY, "");
        }
        return "";
    }

    /**
     * Check if a message should be logged based on current level
     */
    private static boolean shouldLog(Level messageLevel) {
        return messageLevel.ordinal() >= currentLogLevel.ordinal();
    }

    public static void v(String tag, String message) {
        if (shouldLog(Level.VERBOSE)) {
            v(tag, message, null);
        }
    }

    public static void v(String tag, String message, Throwable throwable) {
        if (shouldLog(Level.VERBOSE)) {
            logInternal(Level.VERBOSE, tag, message, throwable);
            Log.v(tag, message, throwable);
        }
    }

    public static void d(String tag, String message) {
        if (shouldLog(Level.DEBUG)) {
            d(tag, message, null);
        }
    }

    public static void d(String tag, String message, Throwable throwable) {
        if (shouldLog(Level.DEBUG)) {
            logInternal(Level.DEBUG, tag, message, throwable);
            Log.d(tag, message, throwable);
        }
    }

    public static void i(String tag, String message) {
        if (shouldLog(Level.INFO)) {
            i(tag, message, null);
        }
    }

    public static void i(String tag, String message, Throwable throwable) {
        if (shouldLog(Level.INFO)) {
            logInternal(Level.INFO, tag, message, throwable);
            Log.i(tag, message, throwable);
        }
    }

    public static void w(String tag, String message) {
        if (shouldLog(Level.WARN)) {
            w(tag, message, null);
        }
    }

    public static void w(String tag, String message, Throwable throwable) {
        if (shouldLog(Level.WARN)) {
            logInternal(Level.WARN, tag, message, throwable);
            if (throwable != null) {
                Log.w(tag, message, throwable);
            } else {
                Log.w(tag, message);
            }
        }
    }

    public static void e(String tag, String message) {
        if (shouldLog(Level.ERROR)) {
            e(tag, message, null);
        }
    }

    public static void e(String tag, String message, Throwable throwable) {
        if (shouldLog(Level.ERROR)) {
            logInternal(Level.ERROR, tag, message, throwable);
            Log.e(tag, message, throwable);
        }
    }

    private static void logInternal(Level level, String tag, String message, Throwable throwable) {
        LogEntry entry = new LogEntry(System.currentTimeMillis(), level, tag, message, throwable);

        synchronized (logEntries) {
            logEntries.add(entry);
            if (logEntries.size() > MAX_LOG_ENTRIES) {
                logEntries.remove(0);
            }
        }

        // Notify listeners
        for (LogUpdateListener listener : listeners) {
            listener.onLogUpdated(entry);
        }
    }

    // Query methods
    public static List<LogEntry> getLogs(Level minLevelFilter, String searchQuery) {
        synchronized (logEntries) {
            List<LogEntry> filteredLogs = new ArrayList<>();
            String[] searchTerms = searchQuery != null ? searchQuery.split("\\s+") : null;

            for (LogEntry entry : logEntries) {
                // Check if entry level is at least as severe as minLevelFilter
                boolean levelMatch = minLevelFilter == null ||
                        entry.level.ordinal() >= minLevelFilter.ordinal();
                boolean searchMatch = searchTerms == null || searchTerms.length == 0 ||
                        entry.containsAllTerms(searchTerms);

                if (levelMatch && searchMatch) {
                    filteredLogs.add(entry);
                }
            }
            Collections.reverse(filteredLogs);
            return filteredLogs;
        }
    }

    public static void clearLogs() {
        synchronized (logEntries) {
            logEntries.clear();
        }
        for (LogUpdateListener listener : listeners) {
            listener.onLogsCleared();
        }
    }

    public static String formatLogsAsText(List<LogEntry> logs) {
        StringBuilder sb = new StringBuilder();
        SimpleDateFormat dateFormat = new SimpleDateFormat("MM-dd HH:mm:ss.SSS", Locale.getDefault());

        for (LogEntry entry : logs) {
            String time = dateFormat.format(new Date(entry.timestamp));
            sb.append(entry.level.name())
                    .append("/[")
                    .append(entry.tag)
                    .append("] ")
                    .append(time)
                    .append(": ")
                    .append(entry.message);

            if (entry.throwable != null) {
                sb.append("\n").append(Log.getStackTraceString(entry.throwable));
            }
            sb.append("\n");
        }
        return sb.toString();
    }

    public static void addListener(LogUpdateListener listener) {
        listeners.add(listener);
    }

    public static void removeListener(LogUpdateListener listener) {
        listeners.remove(listener);
    }
}