package org.ojrandom.paiesque.data.sync;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import org.ojrandom.paiesque.logging.AppLogger;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class TimestampUnitDetector {
    private static final String TAG = "TimestampUnitDetector";

    // Threshold to distinguish between seconds and milliseconds
    // Any timestamp > this threshold is likely milliseconds
    private static final long MILLIS_THRESHOLD = 10000000000L; // 10 billion (year 2286)

    // Thread-safe cache: tableName -> usesSeconds
    private static final Map<String, Boolean> timestampUnitCache = new ConcurrentHashMap<>();

    /**
     * Detect if a table stores timestamps in seconds or milliseconds
     * Results are cached for performance
     */
    public static boolean usesSeconds(SQLiteDatabase database, String tableName) {
        if (database == null || !database.isOpen()) {
            AppLogger.w(TAG, "Database not available for timestamp detection: " + tableName);
            return false; // Default to milliseconds for safety
        }

        // Check cache first
        Boolean cachedResult = timestampUnitCache.get(tableName);
        if (cachedResult != null) {
            return cachedResult;
        }

        boolean result = detectTimestampUnits(database, tableName);

        // Cache the result
        timestampUnitCache.put(tableName, result);

        AppLogger.d(TAG, "Timestamp unit detection cached for " + tableName +
                ": " + (result ? "SECONDS" : "MILLISECONDS"));

        return result;
    }

    private static boolean detectTimestampUnits(SQLiteDatabase database, String tableName) {
        try {
            // Check a few recent records to determine the timestamp format
            String sampleQuery = "SELECT TIMESTAMP FROM " + tableName +
                    " WHERE TIMESTAMP IS NOT NULL " +
                    " ORDER BY TIMESTAMP DESC LIMIT 3";

            try (Cursor cursor = database.rawQuery(sampleQuery, null)) {
                if (cursor == null || cursor.getCount() == 0) {
                    AppLogger.d(TAG, "No data in table " + tableName + " for timestamp detection");
                    return false; // Default to milliseconds
                }

                int secondsCount = 0;
                int millisecondsCount = 0;

                while (cursor.moveToNext()) {
                    long timestamp = cursor.getLong(0);

                    // Simple heuristic: if timestamp > threshold, it's likely milliseconds
                    if (timestamp < MILLIS_THRESHOLD) {
                        secondsCount++;
                    } else {
                        millisecondsCount++;
                    }
                }

                boolean usesSeconds = secondsCount > millisecondsCount;

                // Log one sample for debugging
                cursor.moveToFirst();
                long sampleTimestamp = cursor.getLong(0);
                AppLogger.d(TAG, String.format(
                        "Timestamp detection for %s: sample=%d, verdict=%s",
                        tableName, sampleTimestamp,
                        usesSeconds ? "SECONDS" : "MILLISECONDS"
                ));

                return usesSeconds;
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error detecting timestamp units for table: " + tableName, e);
            return false; // Default to milliseconds on error
        }
    }

    /**
     * Normalize a timestamp for querying based on detected units
     */
    public static long normalizeForQuery(long timestamp, SQLiteDatabase database, String tableName) {
        boolean usesSeconds = usesSeconds(database, tableName);
        if (usesSeconds) {
            // Input is milliseconds, convert to seconds for this table
            long normalized = timestamp / 1000;
            AppLogger.v(TAG, "Normalized timestamp for " + tableName +
                    ": " + timestamp + "ms -> " + normalized + "s");
            return normalized;
        } else {
            // Input is already milliseconds, use as-is
            return timestamp;
        }
    }

    /**
     * Convert timestamp from table format to milliseconds for storage
     */
    public static long toMilliseconds(long timestamp, SQLiteDatabase database, String tableName) {
        boolean usesSeconds = usesSeconds(database, tableName);
        if (usesSeconds) {
            // Table stores seconds, convert to milliseconds
            return timestamp * 1000;
        } else {
            // Table already stores milliseconds
            return timestamp;
        }
    }

    /**
     * Clear cache (useful when database changes or for testing)
     */
    public static void clearCache() {
        timestampUnitCache.clear();
        AppLogger.d(TAG, "Timestamp unit cache cleared");
    }

    /**
     * Get cache statistics for debugging
     */
    public static String getCacheStats() {
        return "TimestampUnitDetector cache size: " + timestampUnitCache.size();
    }
}