// DatabaseInitializer.java
package org.ojrandom.paiesque.data.database;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;

import java.util.ArrayList;
import java.util.List;

public class DatabaseInitializer {
    private static final String TAG = "DatabaseInitializer";

    public void createTablesIfNotExist(SQLiteDatabase db) {
        try {
            // Device table
            String createDeviceTable = "CREATE TABLE IF NOT EXISTS " + AppConstants.Database.TABLE_DEVICE + " (" +
                    AppConstants.Columns.ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    AppConstants.Columns.NAME + " TEXT NOT NULL, " +
                    AppConstants.Columns.MANUFACTURER + " TEXT NOT NULL, " +
                    AppConstants.Columns.IDENTIFIER + " TEXT NOT NULL UNIQUE, " +
                    "SOURCE_DEVICE_ID TEXT, " +
                    "CREATED_AT INTEGER DEFAULT (strftime('%s','now')), " +
                    "UNIQUE(" + AppConstants.Columns.NAME + ", " + AppConstants.Columns.IDENTIFIER + ")" +
                    ")";

            // Heart rates table with composite primary key
            String createHeartRatesTable = "CREATE TABLE IF NOT EXISTS " + AppConstants.PaiesqueTables.HEART_RATES + " (" +
                    AppConstants.Columns.TIMESTAMP + " INTEGER NOT NULL, " +
                    AppConstants.Columns.DEVICE_ID + " INTEGER NOT NULL, " +
                    AppConstants.Columns.HEART_RATE + " INTEGER NOT NULL, " +
                    "LOCAL_DATE INTEGER NOT NULL, " +
                    "MINUTE_OF_DAY INTEGER NOT NULL, " +
                    "SYNCED_AT INTEGER DEFAULT (strftime('%s','now')), " +
                    "PRIMARY KEY (" + AppConstants.Columns.TIMESTAMP + ", " + AppConstants.Columns.DEVICE_ID + ")" +
                    ") WITHOUT ROWID";

            // PAI scores table
            String createPaiScoresTable = "CREATE TABLE IF NOT EXISTS " + AppConstants.PaiesqueTables.PAI_SCORES + " (" +
                    "LOCAL_DATE INTEGER NOT NULL, " +           // YYYYMMDD format
                    "DEVICE_ID INTEGER NOT NULL, " +            // -1 for "all devices", otherwise specific device ID
                    "PAI_SCORE REAL NOT NULL, " +               // Daily PAI points
                    "HEART_RATE_COUNT INTEGER, " +              // Number of HR readings used
                    "CALCULATION_TIME INTEGER, " +              // Time of last calculation
                    // Zone columns
                    "PAI_ZONE_BELOW_THRESHOLD REAL DEFAULT 0, " +
                    "PAI_ZONE_50_59 REAL DEFAULT 0, " +
                    "PAI_ZONE_60_69 REAL DEFAULT 0, " +
                    "PAI_ZONE_70_79 REAL DEFAULT 0, " +
                    "PAI_ZONE_80_89 REAL DEFAULT 0, " +
                    "PAI_ZONE_90_PLUS REAL DEFAULT 0, " +
                    "TIME_ZONE_BELOW_THRESHOLD REAL DEFAULT 0, " + // Time in seconds
                    "TIME_ZONE_50_59 REAL DEFAULT 0, " +
                    "TIME_ZONE_60_69 REAL DEFAULT 0, " +
                    "TIME_ZONE_70_79 REAL DEFAULT 0, " +
                    "TIME_ZONE_80_89 REAL DEFAULT 0, " +
                    "TIME_ZONE_90_PLUS REAL DEFAULT 0, " +
                    "LAST_UPDATED INTEGER DEFAULT (strftime('%s','now')), " +
                    "PRIMARY KEY (LOCAL_DATE, DEVICE_ID)" +     // Composite primary key
                    ")";

            // SIMPLIFIED RHR values table - removed unused metadata fields
            String createRhrTable = "CREATE TABLE IF NOT EXISTS " + AppConstants.PaiesqueTables.RHR_VALUES + " (" +
                    "LOCAL_DATE INTEGER NOT NULL, " +           // YYYYMMDD format
                    "DEVICE_ID INTEGER NOT NULL, " +            // -1 for "all devices", otherwise specific device ID
                    "RHR_VALUE INTEGER NOT NULL, " +            // Resting heart rate in BPM
                    "STRATEGY_USED TEXT, " +                    // Strategy used: "SleepAnalysis", "FallbackAnalysis"
                    "SAMPLE_SIZE INTEGER DEFAULT 0, " +         // Number of samples used in calculation
                    "CALCULATED_AT INTEGER DEFAULT (strftime('%s','now')), " +
                    "LAST_UPDATED INTEGER DEFAULT (strftime('%s','now')), " +
                    "PRIMARY KEY (LOCAL_DATE, DEVICE_ID)" +     // Composite primary key
                    ")";

            String createPaiTimestampsTable = "CREATE TABLE IF NOT EXISTS " + AppConstants.PaiesqueTables.PAI_TIMESTAMPS + " (" +
                    AppConstants.Columns.DEVICE_ID + " INTEGER PRIMARY KEY, " +
                    "LAST_PROCESSED_DATE INTEGER DEFAULT 0, " +  // Changed from LAST_PROCESSED_TIMESTAMP
                    "LAST_UPDATED INTEGER DEFAULT (strftime('%s','now'))" +
                    ")";

            String createRhrTimestampsTable = "CREATE TABLE IF NOT EXISTS " + AppConstants.PaiesqueTables.RHR_TIMESTAMPS + " (" +
                    AppConstants.Columns.DEVICE_ID + " INTEGER PRIMARY KEY, " +
                    "LAST_PROCESSED_DATE INTEGER DEFAULT 0, " +  // Changed from LAST_PROCESSED_TIMESTAMP
                    "LAST_UPDATED INTEGER DEFAULT (strftime('%s','now'))" +
                    ")";

            String createRollingPaiHistoryTable = "CREATE TABLE IF NOT EXISTS " +
                    AppConstants.PaiesqueTables.ROLLING_PAI_HISTORY + " (" +
                    AppConstants.Columns.ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    AppConstants.Columns.DEVICE_ID + " INTEGER NOT NULL, " +
                    AppConstants.Columns.LOCAL_DATE + " INTEGER NOT NULL, " + // YYYYMMDD format
                    AppConstants.Columns.ROLLING_PAI_SCORE + " REAL NOT NULL, " +
                    "FOREIGN KEY(" + AppConstants.Columns.DEVICE_ID + ") REFERENCES " +
                    AppConstants.Database.TABLE_DEVICE + "(" + AppConstants.Columns.ID + "), " +
                    "UNIQUE(" + AppConstants.Columns.DEVICE_ID + ", " + AppConstants.Columns.LOCAL_DATE + ")" +
                    ")";

            // Settings table using constants
            String createSettingsTable = "CREATE TABLE IF NOT EXISTS " +
                    AppConstants.SettingsTables.APP_SETTINGS + " (" +
                    AppConstants.SettingsColumns.ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    AppConstants.SettingsColumns.SETTING_KEY + " TEXT UNIQUE NOT NULL, " +
                    AppConstants.SettingsColumns.SETTING_VALUE + " TEXT, " +
                    AppConstants.SettingsColumns.SETTING_TYPE + " TEXT NOT NULL, " +
                    AppConstants.SettingsColumns.CATEGORY + " TEXT NOT NULL, " +
                    AppConstants.SettingsColumns.LAST_MODIFIED + " INTEGER DEFAULT (strftime('%s','now'))" +
                    ")";

            db.execSQL(createDeviceTable);
            db.execSQL(createHeartRatesTable);
            db.execSQL(createPaiScoresTable);
            db.execSQL(createRhrTable);
            db.execSQL(createPaiTimestampsTable);
            db.execSQL(createRhrTimestampsTable);
            db.execSQL(createRollingPaiHistoryTable);
            db.execSQL(createSettingsTable);

            // Run migration after table creation
            migrateToCompositePrimaryKey(db);

            AppLogger.d(TAG, "All tables created/verified successfully");
        } catch (Exception e) {
            AppLogger.e(TAG, "Error creating tables", e);
        }
    }

    public void createIndexes(SQLiteDatabase db) {
        try {
            // Device table indexes
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_device_identifier ON " +
                    AppConstants.Database.TABLE_DEVICE + " (" + AppConstants.Columns.IDENTIFIER + ")");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_device_name ON " +
                    AppConstants.Database.TABLE_DEVICE + " (" + AppConstants.Columns.NAME + ")");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_device_manufacturer ON " +
                    AppConstants.Database.TABLE_DEVICE + " (" + AppConstants.Columns.MANUFACTURER + ")");

            // Heart rates table indexes for composite primary key
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_heartrates_timestamp_device ON " +
                    AppConstants.PaiesqueTables.HEART_RATES + " (" +
                    AppConstants.Columns.TIMESTAMP + ", " + AppConstants.Columns.DEVICE_ID + ")");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_heartrates_local_date ON " +
                    AppConstants.PaiesqueTables.HEART_RATES + " (LOCAL_DATE)");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_heartrates_minute_of_day ON " +
                    AppConstants.PaiesqueTables.HEART_RATES + " (MINUTE_OF_DAY)");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_heartrates_device_date ON " +
                    AppConstants.PaiesqueTables.HEART_RATES + " (" +
                    AppConstants.Columns.DEVICE_ID + ", LOCAL_DATE)");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_heartrates_timestamp_hr ON " +
                    AppConstants.PaiesqueTables.HEART_RATES + " (" +
                    AppConstants.Columns.TIMESTAMP + ", " + AppConstants.Columns.HEART_RATE + ")");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_heartrates_date_hr ON " +
                    AppConstants.PaiesqueTables.HEART_RATES + " (LOCAL_DATE, " + AppConstants.Columns.HEART_RATE + ")");

            // PAI scores table indexes
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_pai_scores_date ON " +
                    AppConstants.PaiesqueTables.PAI_SCORES + " (LOCAL_DATE)");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_pai_scores_score ON " +
                    AppConstants.PaiesqueTables.PAI_SCORES + " (PAI_SCORE)");

            // SIMPLIFIED RHR values table indexes - removed unused indexes
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_rhr_values_date ON " +
                    AppConstants.PaiesqueTables.RHR_VALUES + " (LOCAL_DATE)");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_rhr_values_device ON " +
                    AppConstants.PaiesqueTables.RHR_VALUES + " (DEVICE_ID)");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_rhr_values_value ON " +
                    AppConstants.PaiesqueTables.RHR_VALUES + " (RHR_VALUE)");

            // Timestamp tables indexes
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_pai_timestamps_device ON " +
                    AppConstants.PaiesqueTables.PAI_TIMESTAMPS + " (" + AppConstants.Columns.DEVICE_ID + ")");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_rhr_timestamps_device ON " +
                    AppConstants.PaiesqueTables.RHR_TIMESTAMPS + " (" + AppConstants.Columns.DEVICE_ID + ")");

            // historical pai
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_rolling_pai_history_device_date " +
                    "ON " + AppConstants.PaiesqueTables.ROLLING_PAI_HISTORY + " (" +
                    AppConstants.Columns.DEVICE_ID + ", " +
                    AppConstants.Columns.LOCAL_DATE + ")");

            // Settings table indexes using constants
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_settings_key ON " +
                    AppConstants.SettingsTables.APP_SETTINGS + " (" +
                    AppConstants.SettingsColumns.SETTING_KEY + ")");
            db.execSQL("CREATE INDEX IF NOT EXISTS idx_settings_category ON " +
                    AppConstants.SettingsTables.APP_SETTINGS + " (" +
                    AppConstants.SettingsColumns.CATEGORY + ")");

            AppLogger.d(TAG, "All indexes created successfully");
        } catch (Exception e) {
            AppLogger.e(TAG, "Error creating indexes", e);
        }
    }

    public void removeStrategyUsedColumn(SQLiteDatabase db) {
        try {
            AppLogger.d(TAG, "Checking if STRATEGY_USED column exists in RHR_VALUES table");

            // Check if column exists
            Cursor cursor = db.rawQuery("PRAGMA table_info(" + AppConstants.PaiesqueTables.RHR_VALUES + ")", null);
            boolean columnExists = false;
            while (cursor.moveToNext()) {
                String columnName = cursor.getString(1);
                if ("STRATEGY_USED".equals(columnName)) {
                    columnExists = true;
                    break;
                }
            }
            cursor.close();

            if (columnExists) {
                AppLogger.d(TAG, "STRATEGY_USED column found - removing it");
                // Simple approach: SQLite doesn't support DROP COLUMN directly, so we use the table recreation method
                db.execSQL("CREATE TABLE RHR_VALUES_TEMP AS SELECT LOCAL_DATE, DEVICE_ID, RHR_VALUE, SAMPLE_SIZE, CALCULATED_AT, LAST_UPDATED FROM " + AppConstants.PaiesqueTables.RHR_VALUES);
                db.execSQL("DROP TABLE " + AppConstants.PaiesqueTables.RHR_VALUES);
                db.execSQL("ALTER TABLE RHR_VALUES_TEMP RENAME TO " + AppConstants.PaiesqueTables.RHR_VALUES);
                AppLogger.d(TAG, "STRATEGY_USED column removed successfully");
            } else {
                AppLogger.d(TAG, "STRATEGY_USED column does not exist - no action needed");
            }

        } catch (Exception e) {
            AppLogger.e(TAG, "Error removing STRATEGY_USED column", e);
        }
    }

    public void migrateToCompositePrimaryKey(SQLiteDatabase db) {
        try {
            // Check if we need to migrate by looking at the actual table schema
            Cursor cursor = db.rawQuery("PRAGMA table_info(" + AppConstants.PaiesqueTables.HEART_RATES + ")", null);
            List<String> primaryKeyColumns = new ArrayList<>();

            while (cursor.moveToNext()) {
                String columnName = cursor.getString(1);
                int pkPosition = cursor.getInt(5); // This is the position in PK (0 = not PK)

                if (pkPosition > 0) {
                    primaryKeyColumns.add(columnName);
                }
            }
            cursor.close();

            AppLogger.d(TAG, "Primary key columns found: " + primaryKeyColumns);

            // Old structure: Only TIMESTAMP is primary key
            // New structure: Both TIMESTAMP and DEVICE_ID are primary keys (composite)
            boolean hasOldStructure = (primaryKeyColumns.size() == 1 &&
                    primaryKeyColumns.contains("TIMESTAMP"));

            if (hasOldStructure) {
                AppLogger.d(TAG, "Migrating HEART_RATES table from single to composite primary key");

                // Create temp table with new structure
                db.execSQL("CREATE TABLE HEART_RATES_TEMP (" +
                        AppConstants.Columns.TIMESTAMP + " INTEGER NOT NULL, " +
                        AppConstants.Columns.DEVICE_ID + " INTEGER NOT NULL, " +
                        AppConstants.Columns.HEART_RATE + " INTEGER NOT NULL, " +
                        "LOCAL_DATE INTEGER NOT NULL, " +
                        "MINUTE_OF_DAY INTEGER NOT NULL, " +
                        "SYNCED_AT INTEGER, " +
                        "PRIMARY KEY (" + AppConstants.Columns.TIMESTAMP + ", " + AppConstants.Columns.DEVICE_ID + ")" +
                        ") WITHOUT ROWID");

                // Copy data - INSERT OR IGNORE will skip any records with duplicate (TIMESTAMP, DEVICE_ID) combinations
                // This preserves all heart rates from different devices while preventing primary key conflicts
                db.execSQL("INSERT OR IGNORE INTO HEART_RATES_TEMP " +
                        "SELECT TIMESTAMP, DEVICE_ID, HEART_RATE, LOCAL_DATE, MINUTE_OF_DAY, SYNCED_AT " +
                        "FROM " + AppConstants.PaiesqueTables.HEART_RATES);

                // Replace old table
                db.execSQL("DROP TABLE " + AppConstants.PaiesqueTables.HEART_RATES);
                db.execSQL("ALTER TABLE HEART_RATES_TEMP RENAME TO " + AppConstants.PaiesqueTables.HEART_RATES);

                AppLogger.d(TAG, "Heart rates table migration completed");
            } else {
                AppLogger.d(TAG, "Heart rates table already has composite primary key - no migration needed");
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error migrating heart rates table", e);
        }
    }
}