package org.ojrandom.paiesque.data.backup;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.util.Log;

import org.ojrandom.paiesque.calculation.CalculationProgressListener;
import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.OutputStream;

public class BackupExportManager {
    private static final String TAG = "BackupExportManager";

    private final Context context;
    private CalculationProgressListener progressListener;

    public BackupExportManager(Context context) {
        this.context = context;
    }

    public void setProgressListener(CalculationProgressListener listener) {
        this.progressListener = listener;
    }

    public boolean exportBackup(Uri destinationUri) {
        long startTime = System.currentTimeMillis();

        try {
            ContentResolver resolver = context.getContentResolver();

            // Get database file
            File dbFile = new File(context.getDatabasePath(AppConstants.KEY_PAIESQUE_DB_NAME).getAbsolutePath());
            if (!dbFile.exists()) {
                AppLogger.e(TAG, "Database file not found: " + dbFile.getAbsolutePath());
                return false;
            }

            updateProgress("Starting backup...", 0, 100);

            // Direct file copy - extremely fast!
            OutputStream outputStream = resolver.openOutputStream(destinationUri);
            if (outputStream == null) {
                AppLogger.e(TAG, "Failed to open output stream for URI: " + destinationUri);
                return false;
            }

            try (FileInputStream dbIn = new FileInputStream(dbFile);
                 BufferedInputStream bufferedIn = new BufferedInputStream(dbIn, AppConstants.Backup.BUFFER_SIZE);
                 BufferedOutputStream bufferedOut = new BufferedOutputStream(outputStream, AppConstants.Backup.BUFFER_SIZE)) {

                byte[] buffer = new byte[AppConstants.Backup.BUFFER_SIZE];
                int length;
                long totalBytes = dbFile.length();
                long bytesCopied = 0;

                while ((length = bufferedIn.read(buffer)) > 0) {
                    bufferedOut.write(buffer, 0, length);
                    bytesCopied += length;

                    // Update progress
                    if (totalBytes > 0) {
                        int progress = (int)((bytesCopied * 100) / totalBytes);
                        updateProgress("Backing up database...", progress, 100);
                    }
                }

                bufferedOut.flush();
            }

            long totalTime = System.currentTimeMillis() - startTime;
            updateProgress("Backup completed successfully!", 100, 100);

            AppLogger.i(TAG, String.format("Database backup completed in %.2f seconds", totalTime / 1000.0));
            return true;

        } catch (Exception e) {
            AppLogger.e(TAG, "Error exporting backup", e);
            updateProgress("Backup failed!", 0, 100);
            return false;
        }
    }

    private void updateProgress(String message, int current, int total) {
        if (progressListener != null) {
            progressListener.onBatchProgress(current, total, message);
        }
    }

}