package org.ojrandom.paiesque.data;

import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;

public class TimestampConverter {
    private static final String TAG = "TimestampConverter";
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    /**
     * Converts a date string (yyyy-MM-dd) to epoch seconds at 00:00:00 of that day
     */
    public static long convertDateToTimestamp(String dateString) {
        try {
            LocalDate date = LocalDate.parse(dateString, DATE_FORMATTER);
            ZonedDateTime startOfDay = date.atStartOfDay(ZoneId.systemDefault());
            long epochSeconds = startOfDay.toEpochSecond();

            AppLogger.d(TAG, "Converted date '" + dateString + "' to timestamp: " + epochSeconds);
            return epochSeconds;
        } catch (Exception e) {
            AppLogger.e(TAG, "Error converting date: " + dateString, e);
            return 0L;
        }
    }

    /**
     * Converts epoch seconds back to a date string for verification
     */
    public static String convertTimestampToDate(long epochSeconds) {
        try {
            ZonedDateTime dateTime = ZonedDateTime.ofInstant(
                    java.time.Instant.ofEpochSecond(epochSeconds),
                    ZoneId.systemDefault()
            );
            return dateTime.format(DATE_FORMATTER);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error converting timestamp: " + epochSeconds, e);
            return "ERROR";
        }
    }

    /**
     * Debug method to see what dates your heart rate data covers
     */
    public static void debugHeartRateDateRange(List<Map<String, String>> heartRateData, String timestampKey) {
        if (heartRateData == null || heartRateData.isEmpty()) {
            AppLogger.d(TAG, "No heart rate data available");
            return;
        }

        AppLogger.d(TAG, "=== HEART RATE DATA DATE RANGE ===");

        // Get first timestamp
        String firstTimestampStr = heartRateData.get(0).get(timestampKey);
        if (firstTimestampStr != null) {
            try {
                long firstTs = Long.parseLong(firstTimestampStr);
                String firstDate = convertTimestampToDate(firstTs);
                AppLogger.d(TAG, "First entry: " + firstTimestampStr + " -> " + firstDate);
            } catch (NumberFormatException e) {
                AppLogger.e(TAG, "Invalid first timestamp: " + firstTimestampStr);
            }
        }

        // Get last timestamp
        String lastTimestampStr = heartRateData.get(heartRateData.size() - 1).get(timestampKey);
        if (lastTimestampStr != null) {
            try {
                long lastTs = Long.parseLong(lastTimestampStr);
                String lastDate = convertTimestampToDate(lastTs);
                AppLogger.d(TAG, "Last entry: " + lastTimestampStr + " -> " + lastDate);
            } catch (NumberFormatException e) {
                AppLogger.e(TAG, "Invalid last timestamp: " + lastTimestampStr);
            }
        }

        AppLogger.d(TAG, "=== END DATE RANGE ===");
    }
}