package org.ojrandom.paiesque.data;

import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.pai.HealthMetricsAnalyzer;
import org.ojrandom.paiesque.rhr.RHRSummary;
import org.ojrandom.paiesque.rhr.RHRSummaryCalculator;

import java.time.LocalDate;
import java.util.TreeMap;

public class SettingsRepository {
    private static final String TAG = "SettingsRepository";
    private final SettingsManager settingsManager;
    private final DataManager dataManager;

    public SettingsRepository(SettingsManager settingsManager) {
        this.settingsManager = settingsManager;
        this.dataManager = null;
    }

    public SettingsRepository(DataManager dataManager) {
        this.settingsManager = dataManager.getSettingsManager();
        this.dataManager = dataManager;
    }

    public PaiSettings getPaiSettings() {
        return new PaiSettings(
                settingsManager.getRHR(),
                settingsManager.getMaxHR(),
                settingsManager.getMaxHrMode(),
                settingsManager.isManualRHR(),
                settingsManager.getAge(),
                settingsManager.getGender(),
                settingsManager.getMaxHRFormula()
        );
    }

    public SleepSettings getSleepSettings() {
        return new SleepSettings(
                settingsManager.getSleepStartHour(),
                settingsManager.getSleepStartMinute(),
                settingsManager.getSleepEndHour(),
                settingsManager.getSleepEndMinute()
        );
    }

    public void savePaiSettings(PaiSettings settings) {
        settingsManager.setRHR(settings.getRestingHR());
        settingsManager.setMaxHR(settings.getMaxHR());
        settingsManager.setMaxHrMode(settings.getMaxHrMode()); // NEW: Save max HR mode
        settingsManager.setManualRHR(settings.isManualRHR());
        settingsManager.setAge(settings.getAge());
        settingsManager.setGender(settings.getGender());
        settingsManager.setMaxHRFormula(settings.getMaxHRFormula());
        settingsManager.markSettingsAsChanged();
        AppLogger.d(TAG, "PAI settings saved");
    }

    public void saveSleepSettings(SleepSettings settings) {
        settingsManager.setSleepTimes(
                settings.getStartHour(),
                settings.getStartMinute(),
                settings.getEndHour(),
                settings.getEndMinute()
        );
        AppLogger.d(TAG, "Sleep settings saved");
    }

    public HealthData getCalculatedHealthData(int deviceId) {
        try {
            if (dataManager == null || dataManager.getPaiesqueDBManager() == null) {
                AppLogger.e(TAG, "DataManager not available for health data calculation");
                return new HealthData(0, 0);
            }

            // Get historical RHR data for calculated RHR
            TreeMap<LocalDate, Integer> historicalRHR =
                    dataManager.getPaiesqueDBManager().loadHistoricalRHR(deviceId);

            // Use RHRSummaryCalculator for average RHR
            RHRSummaryCalculator calculator = new RHRSummaryCalculator();
            RHRSummary summary = calculator.getSummary(historicalRHR, deviceId);

            // Get measured max HR from HealthMetricsAnalyzer
            HealthMetricsAnalyzer healthAnalyzer = new HealthMetricsAnalyzer();
            int measuredMaxHR = healthAnalyzer.calculateMeasuredMaxHR(dataManager.getPaiesqueDBManager().getWritableDatabase());

            int calculatedRHR = summary.lastWeekRHR > 0 ? summary.lastWeekRHR :
                    (summary.lastMonthRHR > 0 ? summary.lastMonthRHR : 0);

            return new HealthData(calculatedRHR, measuredMaxHR);

        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting calculated health data", e);
            return new HealthData(0, 0);
        }
    }

    public static class PaiSettings {
        private final int restingHR;
        private final int maxHR;
        private final String maxHrMode;
        private final boolean manualRHR;
        private final int age;
        private final String gender;
        private final String maxHRFormula;

        // Updated constructor with max HR mode
        public PaiSettings(int restingHR, int maxHR, String maxHrMode, boolean manualRHR, int age, String gender, String maxHRFormula) {
            this.restingHR = restingHR;
            this.maxHR = maxHR;
            this.maxHrMode = maxHrMode;
            this.manualRHR = manualRHR;
            this.age = age;
            this.gender = gender;
            this.maxHRFormula = maxHRFormula;
        }

        // Overloaded constructor for backward compatibility
        public PaiSettings(int restingHR, int maxHR, boolean manualMaxHR, boolean manualRHR, int age, String gender, String maxHRFormula) {
            this(restingHR, maxHR, manualMaxHR ? "manual" : "formula", manualRHR, age, gender, maxHRFormula);
        }

        // Getters
        public int getRestingHR() { return restingHR; }
        public int getMaxHR() { return maxHR; }
        public String getMaxHrMode() { return maxHrMode; }
        public boolean isManualRHR() { return manualRHR; }
        public int getAge() { return age; }
        public String getGender() { return gender; }
        public String getMaxHRFormula() { return maxHRFormula; }

        // Helper method for backward compatibility
        public boolean isManualMaxHR() {
            return "manual".equals(maxHrMode);
        }
    }

    public static class SleepSettings {
        private final int startHour;
        private final int startMinute;
        private final int endHour;
        private final int endMinute;

        public SleepSettings(int startHour, int startMinute, int endHour, int endMinute) {
            this.startHour = startHour;
            this.startMinute = startMinute;
            this.endHour = endHour;
            this.endMinute = endMinute;
        }

        // Getters
        public int getStartHour() { return startHour; }
        public int getStartMinute() { return startMinute; }
        public int getEndHour() { return endHour; }
        public int getEndMinute() { return endMinute; }

        public String getStartTimeString() {
            return String.format("%02d:%02d", startHour, startMinute);
        }

        public String getEndTimeString() {
            return String.format("%02d:%02d", endHour, endMinute);
        }
    }

    public static class HealthData {
        public final int restingHR;
        public final int maxHR;
        public final boolean hasRHRData;
        public final boolean hasMaxHRData;

        public HealthData(int restingHR, int maxHR) {
            this.restingHR = restingHR;
            this.maxHR = maxHR;
            this.hasRHRData = restingHR > 0;
            this.hasMaxHRData = maxHR > 0;
        }
    }
}