package org.ojrandom.paiesque.data;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Manages the Gadgetbridge source database for synchronization.
 * Read-only access to the copied gadgetbridge.db file.
 */
public class GadgetbridgeSyncSource {
    private static final String TAG = "GadgetbridgeSyncSource";
    private SQLiteDatabase sourceDb;
    private volatile boolean isInitialized = false;

    /**
     * Initializes the Gadgetbridge source database for synchronization
     */
    public boolean initializeSource(String dbPath) {
        try {
            sourceDb = SQLiteDatabase.openDatabase(dbPath, null, SQLiteDatabase.OPEN_READONLY);
            isInitialized = true;
            AppLogger.d(TAG, "Gadgetbridge source database initialized: " + dbPath);
            return true;
        } catch (Exception e) {
            AppLogger.e(TAG, "Error initializing Gadgetbridge source database", e);
            return false;
        }
    }

    public boolean isSourceInitialized() {
        return isInitialized && sourceDb != null && sourceDb.isOpen();
    }

    public void closeSource() {
        if (sourceDb != null && sourceDb.isOpen()) {
            sourceDb.close();
        }
        isInitialized = false;
        AppLogger.d(TAG, "Gadgetbridge source database closed");
    }

    /**
     * Get the SQLiteDatabase instance for raw query operations
     * Used for performance-optimized streaming operations
     */
    public SQLiteDatabase getDatabase() {
        return sourceDb;
    }

    // Method to read data dynamically based on table name and columns
    public List<Map<String, String>> readDataFromTable(String tableName, String[] columns,
                                                       String[] whereClauses, String[] whereArgs,
                                                       String orderBy) {
        List<Map<String, String>> result = new ArrayList<>();

        if (sourceDb == null || !sourceDb.isOpen()) {
            AppLogger.e(TAG, "Gadgetbridge source database is not initialized or not open.");
            return result;
        }

        // Step 1: Analyze table schema to determine column types
        Map<String, String> columnTypes = getColumnTypes(tableName);

        // Step 2: Build query with automatic type casting for parameters
        StringBuilder query = new StringBuilder("SELECT ");

        for (int i = 0; i < columns.length; i++) {
            query.append(columns[i]);
            if (i < columns.length - 1) {
                query.append(", ");
            }
        }

        query.append(" FROM ").append(tableName);

        // Step 3: Apply automatic type casting to WHERE clauses
        if (whereClauses != null && whereClauses.length > 0) {
            query.append(" WHERE ");
            for (int i = 0; i < whereClauses.length; i++) {
                String processedClause = applyAutoTypeCasting(whereClauses[i], columnTypes);
                query.append(processedClause);

                if (i < whereClauses.length - 1) {
                    query.append(" AND ");
                }
            }
        }

        // NEW: Add ORDER BY clause if specified
        if (orderBy != null && !orderBy.trim().isEmpty()) {
            query.append(" ORDER BY ").append(orderBy);
        }

        AppLogger.d(TAG, "Generated SQL Query: " + query);
        AppLogger.d(TAG, "Where args: " + Arrays.toString(whereArgs));

        // Validate parameter count
        if (whereArgs != null && whereClauses != null && whereArgs.length != whereClauses.length) {
            AppLogger.e(TAG, "Parameter count mismatch. Expected: " + whereClauses.length + ", Got: " + whereArgs.length);
            return result;
        }

        // Execute query
        try (Cursor cursor = sourceDb.rawQuery(query.toString(), whereArgs)) {
            AppLogger.d(TAG, "Cursor count: " + (cursor != null ? cursor.getCount() : "null"));
            result = processCursorWithSmartTypeHandling(cursor, columns);
            AppLogger.d(TAG, "Processed results: " + result.size());
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading from Gadgetbridge source database", e);
        }

        return result;
    }

    // Overloaded method for backward compatibility
    public List<Map<String, String>> readDataFromTable(String tableName, String[] columns,
                                                       String[] whereClauses, String[] whereArgs) {
        return readDataFromTable(tableName, columns, whereClauses, whereArgs, null);
    }

    private Map<String, String> getColumnTypes(String tableName) {
        Map<String, String> columnTypes = new HashMap<>();

        try (Cursor cursor = sourceDb.rawQuery("PRAGMA table_info(" + tableName + ")", null)) {
            if (cursor != null && cursor.moveToFirst()) {
                do {
                    String name = cursor.getString(1);
                    String type = cursor.getString(2).toUpperCase();
                    columnTypes.put(name, type);
                    AppLogger.d(TAG, "Column: " + name + " -> Type: " + type);
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error reading table schema from Gadgetbridge source", e);
        }

        return columnTypes;
    }

    private String applyAutoTypeCasting(String whereClause, Map<String, String> columnTypes) {
        // Extract column name from where clause
        String columnName = extractColumnNameFromWhereClause(whereClause);

        if (columnName != null && columnTypes.containsKey(columnName)) {
            String columnType = columnTypes.get(columnName);

            // Apply appropriate CAST based on column type
            if (whereClause.contains("?")) {
                if (isNumericType(columnType)) {
                    return whereClause.replace("?", "CAST(? AS REAL)");
                } else if (isIntegerType(columnType)) {
                    return whereClause.replace("?", "CAST(? AS INTEGER)");
                } else if (isTextType(columnType)) {
                    return whereClause; // No casting needed for text
                }
            }
        }

        // Fallback: apply smart casting based on common patterns
        return applySmartFallbackCasting(whereClause);
    }

    private String extractColumnNameFromWhereClause(String whereClause) {
        // Simple pattern matching to extract column names
        Pattern pattern = Pattern.compile("([a-zA-Z_][a-zA-Z0-9_]*)\\s*[=<>!]+");
        Matcher matcher = pattern.matcher(whereClause);

        if (matcher.find()) {
            return matcher.group(1);
        }

        // Handle function calls like ABS(column)
        pattern = Pattern.compile("([a-zA-Z_][a-zA-Z0-9_]*)\\s*\\(\\s*([a-zA-Z_][a-zA-Z0-9_]*)\\s*\\)");
        matcher = pattern.matcher(whereClause);

        if (matcher.find()) {
            return matcher.group(2); // Return the column name inside the function
        }

        return null;
    }

    private boolean isNumericType(String type) {
        return type.contains("REAL") || type.contains("FLOAT") || type.contains("DOUBLE") ||
                type.contains("DECIMAL") || type.contains("NUMERIC");
    }

    private boolean isIntegerType(String type) {
        return type.contains("INT") || type.contains("LONG") || type.contains("BOOLEAN") ||
                type.contains("TINYINT") || type.contains("SMALLINT") || type.contains("BIGINT");
    }

    private boolean isTextType(String type) {
        return type.contains("TEXT") || type.contains("CHAR") || type.contains("VARCHAR") ||
                type.contains("CLOB") || type.equals("") || type.equals("NULL");
    }

    private String applySmartFallbackCasting(String whereClause) {
        // Fallback logic for when column type is unknown
        if (whereClause.contains("?")) {
            // Common patterns for numeric comparison
            if (whereClause.toLowerCase().contains("abs(") ||
                    whereClause.toLowerCase().contains("timestamp") ||
                    whereClause.toLowerCase().contains("time") ||
                    whereClause.toLowerCase().contains("date") ||
                    whereClause.toLowerCase().contains("count") ||
                    whereClause.toLowerCase().contains("sum(") ||
                    whereClause.toLowerCase().contains("avg(")) {
                return whereClause.replace("?", "CAST(? AS REAL)");
            }

            // Common patterns for integer comparison
            if (whereClause.toLowerCase().contains("id") ||
                    whereClause.toLowerCase().contains("_id") ||
                    whereClause.toLowerCase().contains("flag") ||
                    whereClause.toLowerCase().contains("status")) {
                return whereClause.replace("?", "CAST(? AS INTEGER)");
            }
        }

        return whereClause;
    }

    private List<Map<String, String>> processCursorWithSmartTypeHandling(Cursor cursor, String[] columns) {
        List<Map<String, String>> result = new ArrayList<>();

        if (cursor == null) {
            AppLogger.d(TAG, "Cursor is null");
            return result;
        }

        if (cursor.moveToFirst()) {
            // Pre-cache column indices and types
            int[] columnIndices = new int[columns.length];
            int[] columnSqlTypes = new int[columns.length];

            for (int i = 0; i < columns.length; i++) {
                columnIndices[i] = cursor.getColumnIndex(columns[i]);
                if (columnIndices[i] != -1) {
                    columnSqlTypes[i] = cursor.getType(columnIndices[i]);
                } else {
                    AppLogger.w(TAG, "Column not found in cursor: " + columns[i]);
                }
            }

            do {
                Map<String, String> row = new HashMap<>();
                for (int i = 0; i < columns.length; i++) {
                    if (columnIndices[i] != -1) {
                        String value = convertCursorValueToString(cursor, columnIndices[i], columnSqlTypes[i]);
                        row.put(columns[i], value);
                    } else {
                        row.put(columns[i], null);
                    }
                }
                result.add(row);
            } while (cursor.moveToNext());
        }

        return result;
    }

    private String convertCursorValueToString(Cursor cursor, int columnIndex, int sqlType) {
        try {
            switch (sqlType) {
                case Cursor.FIELD_TYPE_NULL:
                    return null;
                case Cursor.FIELD_TYPE_INTEGER:
                    return String.valueOf(cursor.getLong(columnIndex));
                case Cursor.FIELD_TYPE_FLOAT:
                    return String.valueOf(cursor.getDouble(columnIndex));
                case Cursor.FIELD_TYPE_STRING:
                    return cursor.getString(columnIndex);
                case Cursor.FIELD_TYPE_BLOB:
                    byte[] blobData = cursor.getBlob(columnIndex);
                    return "[BLOB:" + blobData.length + " bytes]";
                default:
                    return cursor.getString(columnIndex); // Fallback
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error converting value at column index " + columnIndex, e);
            return "[CONVERSION_ERROR]";
        }
    }

}