package org.ojrandom.paiesque.calculation;

import android.net.Uri;
import android.util.Log;

import org.ojrandom.paiesque.data.DailyHeartRates;
import org.ojrandom.paiesque.data.DataManager;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.rhr.RhrResult;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;

public class CalculationManager {
    private static final String TAG = "CalculationManager"; // ADD THIS LINE
    private final DataManager dataManager;
    private final ExecutorService executor;
    private final AtomicBoolean isCalculationRunning;
    private Future<?> currentCalculationFuture;

    public CalculationManager(DataManager dataManager) {
        this.dataManager = dataManager;
        this.executor = Executors.newSingleThreadExecutor();
        this.isCalculationRunning = new AtomicBoolean(false);
    }

    public void calculateAll(CalculationProgressListener listener,
                             int deviceId, String deviceName,
                             Uri databaseUri) {
        synchronized (isCalculationRunning) {
            if (isCalculationRunning.get()) {
                listener.onError("Calculation already in progress");
                return;
            }
            isCalculationRunning.set(true);
        }

        stopCalculation();

        currentCalculationFuture = executor.submit(() -> {
            Boolean hasHeartRateData = null;
            try {
                if (databaseUri != null) {
                    // New database needs to be initialized
                    hasHeartRateData = performCalculationsWithDatabaseInit(listener, deviceId, deviceName, databaseUri);
                } else {
                    // Database already initialized, just do calculations
                    hasHeartRateData = performCalculationsWithProgress(listener, deviceId, deviceName);
                }

                if (!Thread.currentThread().isInterrupted()) {
                    listener.onAllCalculationsComplete(hasHeartRateData);
                }
            } catch (Exception e) {
                if (!Thread.currentThread().isInterrupted()) {
                    listener.onError("Calculation failed: " + e.getMessage());
                }
            } finally {
                isCalculationRunning.set(false);
            }
        });
    }

    private Boolean performCalculationsWithDatabaseInit(CalculationProgressListener listener,
                                                        int deviceId, String deviceName,
                                                        Uri databaseUri) {
        // Step 0: Initialize database (copy and sync)
        listener.onProgressUpdate(0, 6, "Initializing database...");
        boolean dbInitialized = dataManager.initializeDatabase(databaseUri, new DataManager.DatabaseProgressListener() {
            @Override
            public void onProgress(String status) {
                listener.onProgressUpdate(0, 6, status);
            }

            @Override
            public void onComplete(boolean success) {
                if (!success) {
                    throw new RuntimeException("Database initialization failed");
                }
            }

            @Override
            public void onError(String errorMessage) {
                // Forward the error to the main calculation listener
                listener.onError("Database error: " + errorMessage);
            }
        });

        if (!dbInitialized || Thread.currentThread().isInterrupted()) return false;

        // Continue with normal calculations
        return performCalculationsWithProgress(listener, deviceId, deviceName);
    }

    private Boolean performCalculationsWithProgress(CalculationProgressListener listener,
                                                    int deviceId, String deviceName) {
        boolean hasHeartRateData = false;

        AppLogger.i(TAG, "Starting PAI and RHR calculations for " + deviceName);

        // Step 1: Load heart rates as pre-grouped daily data
        listener.onProgressUpdate(1, 5, "Loading heart rate data...");
        Map<LocalDate, DailyHeartRates> dailyHeartRatesForPAI = dataManager.loadDailyHeartRatesForPAI(deviceId);

        // Check if we have any heart rate data
        if (dailyHeartRatesForPAI != null && !dailyHeartRatesForPAI.isEmpty()) {
            hasHeartRateData = true;
            AppLogger.i(TAG, "Heart rate data loaded: " + dailyHeartRatesForPAI.size() + " days with data");
            listener.onHeartRatesLoaded(dailyHeartRatesForPAI);
        } else {
            // No heart rate data found
            AppLogger.w(TAG, "No heart rate data found for device: " + deviceName);
        }


        if (Thread.currentThread().isInterrupted()) return hasHeartRateData;

        // Step 2: Calculate and store PAI - only if we have data
        if (hasHeartRateData) {
            listener.onProgressUpdate(2, 5, "Calculating and storing PAI scores...");
            List<Map<String, String>> rollingPAI = dataManager.calculatePAI(dailyHeartRatesForPAI, deviceId);
            AppLogger.i(TAG, "PAI calculation completed");
            listener.onPaiCalculated(rollingPAI);
        } else {
            listener.onProgressUpdate(2, 5, "Skipping PAI calculation - no heart rate data");
        }

        if (Thread.currentThread().isInterrupted()) return hasHeartRateData;

        // Step 3: Prepare for RHR calculation
        listener.onProgressUpdate(3, 5, "Preparing data for RHR calculation...");
        listener.onHrStored();

        if (Thread.currentThread().isInterrupted()) return hasHeartRateData;

        // Step 4: Calculate and store RHR - only if we have data
        if (hasHeartRateData) {
            listener.onProgressUpdate(4, 5, "Calculating and storing resting heart rates...");
            Map<LocalDate, DailyHeartRates> dailyHeartRatesForRHR = dataManager.loadDailyHeartRatesForRHR(deviceId);
            RhrResult rhrResult = dataManager.calculateRHR(dailyHeartRatesForRHR, deviceId);
            AppLogger.i(TAG, "RHR calculation completed");
            listener.onRhrCalculated(rhrResult);
        } else {
            listener.onProgressUpdate(4, 5, "Skipping RHR calculation - no heart rate data");
        }

        if (Thread.currentThread().isInterrupted()) return hasHeartRateData;

        listener.onStorageComplete();

        AppLogger.i(TAG, "All calculations finished successfully");

        return hasHeartRateData;
    }

    public void stopCalculation() {
        if (currentCalculationFuture != null && !currentCalculationFuture.isDone()) {
            currentCalculationFuture.cancel(true);
            isCalculationRunning.set(false);
        }
    }

    public void shutdown() {
        stopCalculation();
        if (executor != null) {
            executor.shutdownNow();
        }
    }
}