package org.ojrandom.paiesque.svg;

import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

public class ChartSvgGenerator {
    private static final String TAG = "ChartSvgGenerator";

    // Use fixed dimensions - SAME FOR BOTH CHART TYPES
    private static final int CHART_WIDTH = 800;
    private static final int CHART_HEIGHT = 400;
    private static final int PADDING = 70;
    private static final int BAR_SPACING = 3;
    private static final int LABEL_FONT_SIZE_DEFAULT = 36;
    private static final int MIN_BAR_WIDTH = 2;


    private static final String BACKGROUND_COLOR = "transparent";

    // Use Material3 theme colors that will work in both light and dark modes
    private static final String TEXT_COLOR_LIGHT = "#1C1B1F"; // onSurface for light theme
    private static final String TEXT_COLOR_DARK = "#E6E1E5"; // onSurface for dark theme
    private static final String AXIS_COLOR_LIGHT = "#79747E"; // outline for light theme
    private static final String AXIS_COLOR_DARK = "#938F99"; // outline for dark theme
    private static final String GRID_LINE_COLOR_LIGHT = "#79747E"; // outline for light theme
    private static final String GRID_LINE_COLOR_DARK = "#938F99"; // outline for dark theme
    private static boolean isDarkMode = false;

    public static void setDarkMode(boolean darkMode) {
        isDarkMode = darkMode;
    }

    private static String getTextColor() {
        return isDarkMode ? TEXT_COLOR_DARK : TEXT_COLOR_LIGHT;
    }

    private static String getAxisColor() {
        return isDarkMode ? AXIS_COLOR_DARK : AXIS_COLOR_LIGHT;
    }

    private static String getGridLineColor() {
        return isDarkMode ? GRID_LINE_COLOR_DARK : GRID_LINE_COLOR_LIGHT;
    }

    // KEPT: PAI Colors - brand colors for bars (unchanged)
    private static final String NO_PAI_COLOR = "#666666";
    private static final String LOW_PAI_COLOR = "#FF5252";
    private static final String MEDIUM_PAI_COLOR = "#FFA726";
    private static final String HIGH_PAI_COLOR = "#4CAF50";
    private static final String TARGET_LINE_COLOR = "#FF1744";

    // KEPT: RHR Colors - health indicator colors for bars (unchanged)
    private static final String ATHLETE_RHR_COLOR = "#4FC3F7";
    private static final String EXCELLENT_RHR_COLOR = "#4DB6AC";
    private static final String IDEAL_RHR_COLOR = "#4CAF50";
    private static final String AVERAGE_RHR_COLOR = "#FFC107";
    private static final String ELEVATED_RHR_COLOR = "#FF9800";
    private static final String HIGH_RISK_RHR_COLOR = "#FF5252";

    // RHR zones
    private static final int ATHLETE_MAX = 50;       // <50
    private static final int EXCELLENT_MAX = 60;     // 50-59
    private static final int IDEAL_MAX = 70;         // 60-69
    private static final int AVERAGE_MAX = 80;       // 70-79
    private static final int ELEVATED_MAX = 90;      // 80-89
    // ≥90: HIGH_RISK_RHR_COLOR

    public enum ChartType {
        PAI,
        RESTING_HEART_RATE
    }

    // Add a method to detect dark mode (you'll need to pass this from your Activity/Fragment)

    // Public API - UPDATED WITH DIMENSION PARAMETERS
    public static String generate7DayPaiSvg(List<Map<String, String>> data, int containerWidth, int containerHeight) {
        return generateSvgChart(data, 7, "7-Day PAI Trend", true, false, ChartType.PAI, containerWidth, containerHeight);
    }

    public static String generate30DayPaiSvg(List<Map<String, String>> data, int containerWidth, int containerHeight) {
        return generateSvgChart(data, 30, "30-Day PAI Trend", false, false, ChartType.PAI, containerWidth, containerHeight);
    }

    public static String generateAllTimePaiSvg(List<Map<String, String>> data, int containerWidth, int containerHeight) {
        if (data == null || data.isEmpty()) {
            return generateSvgChart(Collections.emptyList(), -1, "All-Time PAI Trend", false, true, ChartType.PAI, containerWidth, containerHeight);
        }
        List<Map<String, String>> aggregatedData = aggregateAllTimeData(data, ChartType.PAI);
        debugChartData(aggregatedData, "PAI");
        debugDateRange(aggregatedData, "PAI");
        return generateSvgChart(aggregatedData, -1, "All-Time PAI Trend", false, true, ChartType.PAI, containerWidth, containerHeight);
    }

    public static String generate7DayRhrSvg(List<Map<String, String>> data, int containerWidth, int containerHeight) {
        return generateSvgChart(data, 7, "7-Day Resting Heart Rate", true, false, ChartType.RESTING_HEART_RATE, containerWidth, containerHeight);
    }

    public static String generate30DayRhrSvg(List<Map<String, String>> data, int containerWidth, int containerHeight) {
        return generateSvgChart(data, 30, "30-Day Resting Heart Rate", false, false, ChartType.RESTING_HEART_RATE, containerWidth, containerHeight);
    }

    public static String generateAllTimeRhrSvg(List<Map<String, String>> data, int containerWidth, int containerHeight) {
        if (data == null || data.isEmpty()) {
            return generateSvgChart(Collections.emptyList(), -1, "All-Time Resting Heart Rate", false, true, ChartType.RESTING_HEART_RATE, containerWidth, containerHeight);
        }
        List<Map<String, String>> aggregatedData = aggregateAllTimeData(data, ChartType.RESTING_HEART_RATE);
        return generateSvgChart(aggregatedData, -1, "All-Time Resting Heart Rate", false, true, ChartType.RESTING_HEART_RATE, containerWidth, containerHeight);
    }

    // [Keep all the aggregation methods the same as before...]
    // Unified aggregation logic - SAME FOR BOTH CHART TYPES
    private static List<Map<String, String>> aggregateAllTimeData(List<Map<String, String>> data, ChartType chartType) {
        if (data.isEmpty()) return data;

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
        List<Map<String, String>> sortedData = new ArrayList<>(data);
        Collections.sort(sortedData, new Comparator<Map<String, String>>() {
            @Override
            public int compare(Map<String, String> o1, Map<String, String> o2) {
                try {
                    Date date1 = dateFormat.parse(o1.get("date"));
                    Date date2 = dateFormat.parse(o2.get("date"));
                    return date1.compareTo(date2);
                } catch (ParseException e) {
                    return 0;
                }
            }
        });

        String earliestDate = sortedData.get(0).get("date");
        String today = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault()).format(new Date());

        try {
            Date startDate = dateFormat.parse(earliestDate);
            Date endDate = dateFormat.parse(today);
            long daysBetween = (endDate.getTime() - startDate.getTime()) / (24 * 60 * 60 * 1000);

            AppLogger.d(TAG, "All-time chart aggregation: " + earliestDate + " to " + today +
                    " = " + daysBetween + " days, " + data.size() + " data points");

            // UNIFIED AGGREGATION THRESHOLDS - SAME FOR BOTH
            List<Map<String, String>> result;
            if (daysBetween <= 90) {
                result = fillGapsInAllTimeData(data, today, chartType);
            } else if (daysBetween <= 730) {  // 2 years = 730 days
                result = aggregateByWeeks(data, startDate, endDate, chartType);
            } else {
                result = aggregateByMonths(data, startDate, endDate, chartType);
            }

            AppLogger.d(TAG, "Aggregation result: " + result.size() + " entries");
            return result;

        } catch (ParseException e) {
            AppLogger.e(TAG, "Date parsing error in aggregation: " + e.getMessage());
            return fillGapsInAllTimeData(data, today, chartType);
        }
    }

    private static List<Map<String, String>> aggregateByWeeks(List<Map<String, String>> data, Date startDate, Date endDate, ChartType chartType) {
        // [Keep the same implementation as before...]
        List<Map<String, String>> weeklyData = new ArrayList<>();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
        SimpleDateFormat weekFormat = new SimpleDateFormat("yyyy-'W'ww", Locale.getDefault());

        // Group data by week
        Map<String, List<Integer>> weeklyValues = new HashMap<>();
        Map<String, String> weekStartDates = new HashMap<>();

        for (Map<String, String> entry : data) {
            try {
                Date entryDate = dateFormat.parse(entry.get("date"));
                String weekKey = weekFormat.format(entryDate);

                if (chartType == ChartType.PAI) {
                    int paiValue = Integer.parseInt(entry.get("pai_7_day"));
                    weeklyValues.computeIfAbsent(weekKey, k -> new ArrayList<>()).add(paiValue);
                } else {
                    String valueStr = entry.get("resting_heart_rate");
                    int value = Integer.parseInt(valueStr);
                    if (value > 0) {
                        weeklyValues.computeIfAbsent(weekKey, k -> new ArrayList<>()).add(value);
                    }
                }

                // Store the earliest date in the week as representative date
                if (!weekStartDates.containsKey(weekKey)) {
                    weekStartDates.put(weekKey, entry.get("date"));
                } else {
                    String currentStart = weekStartDates.get(weekKey);
                    Date currentStartDate = dateFormat.parse(currentStart);
                    if (entryDate.before(currentStartDate)) {
                        weekStartDates.put(weekKey, entry.get("date"));
                    }
                }
            } catch (Exception e) {
                AppLogger.e(TAG, "Error processing entry for weekly aggregation: " + e.getMessage());
            }
        }

        // FIX: Generate ALL weeks from start date to end date (inclusive)
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDate);
        // Set to first day of the week
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getFirstDayOfWeek());

        Calendar endCal = Calendar.getInstance();
        endCal.setTime(endDate);
        // Ensure we include the week containing the end date
        endCal.set(Calendar.DAY_OF_WEEK, endCal.getFirstDayOfWeek());
        endCal.add(Calendar.WEEK_OF_YEAR, 1); // Move to start of next week
        endCal.add(Calendar.MILLISECOND, -1); // Go back 1ms to stay in current week

        // Get current week key for comparison
        String currentWeekKey = weekFormat.format(new Date());

        while (!calendar.after(endCal)) {
            String weekKey = weekFormat.format(calendar.getTime());
            Map<String, String> weekEntry = new HashMap<>();

            // FIX: Use better representative dates, especially for current week
            String weekStartDate = weekStartDates.get(weekKey);

            if (weekKey.equals(currentWeekKey)) {
                // For current week, use today's date as representative
                weekEntry.put("date", dateFormat.format(new Date()));
            } else if (weekStartDate != null) {
                // For past weeks with data, use the earliest date in that week
                weekEntry.put("date", weekStartDate);
            } else {
                // For weeks without data, use the start of the week
                weekEntry.put("date", dateFormat.format(calendar.getTime()));
            }

            List<Integer> weekValues = weeklyValues.get(weekKey);
            if (weekValues != null && !weekValues.isEmpty()) {
                if (chartType == ChartType.PAI) {
                    int avgValue = (int) weekValues.stream().mapToInt(Integer::intValue).average().orElse(0);
                    weekEntry.put("pai_7_day", String.valueOf(avgValue));
                    weekEntry.put("pai_day", "0");
                } else {
                    // For RHR, use minimum value (best RHR of the week)
                    int minValue = weekValues.stream().mapToInt(Integer::intValue).min().orElse(0);
                    weekEntry.put("resting_heart_rate", String.valueOf(minValue));
                }
                weekEntry.put("aggregated", "week");
                weekEntry.put("data_points", String.valueOf(weekValues.size()));
            } else {
                // Create zero entries for weeks without data
                if (chartType == ChartType.PAI) {
                    weekEntry.put("pai_7_day", "0");
                    weekEntry.put("pai_day", "0");
                } else {
                    weekEntry.put("resting_heart_rate", "0");
                }
                weekEntry.put("aggregated", "week");
                weekEntry.put("data_points", "0");
            }

            weeklyData.add(weekEntry);
            calendar.add(Calendar.WEEK_OF_YEAR, 1);
        }

        // FIX: Ensure we have the current week if it's missing
        boolean hasCurrentWeek = weeklyData.stream()
                .anyMatch(entry -> {
                    try {
                        Date entryDate = dateFormat.parse(entry.get("date"));
                        return weekFormat.format(entryDate).equals(currentWeekKey);
                    } catch (ParseException e) {
                        return false;
                    }
                });

        if (!hasCurrentWeek) {
            Map<String, String> currentWeekEntry = new HashMap<>();
            currentWeekEntry.put("date", dateFormat.format(new Date())); // Use today's date
            if (chartType == ChartType.PAI) {
                currentWeekEntry.put("pai_7_day", "0");
                currentWeekEntry.put("pai_day", "0");
            } else {
                currentWeekEntry.put("resting_heart_rate", "0");
            }
            currentWeekEntry.put("aggregated", "week");
            currentWeekEntry.put("data_points", "0");
            weeklyData.add(currentWeekEntry);
        }

        AppLogger.d(TAG, "Weekly aggregation: " + weeklyData.size() + " weeks generated from " +
                dateFormat.format(startDate) + " to " + dateFormat.format(endDate) +
                " (includes current week: " + hasCurrentWeek + ")");

        return weeklyData;
    }

    private static List<Map<String, String>> aggregateByMonths(List<Map<String, String>> data, Date startDate, Date endDate, ChartType chartType) {
        // [Keep the same implementation as before...]
        List<Map<String, String>> monthlyData = new ArrayList<>();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
        SimpleDateFormat monthFormat = new SimpleDateFormat("yyyy-MM", Locale.getDefault());

        // Group data by month
        Map<String, List<Integer>> monthlyValues = new HashMap<>();
        Map<String, String> monthStartDates = new HashMap<>();

        for (Map<String, String> entry : data) {
            try {
                Date entryDate = dateFormat.parse(entry.get("date"));
                String monthKey = monthFormat.format(entryDate);

                if (chartType == ChartType.PAI) {
                    int paiValue = Integer.parseInt(entry.get("pai_7_day"));
                    monthlyValues.computeIfAbsent(monthKey, k -> new ArrayList<>()).add(paiValue);
                } else {
                    String valueStr = entry.get("resting_heart_rate");
                    int value = Integer.parseInt(valueStr);
                    if (value > 0) {
                        monthlyValues.computeIfAbsent(monthKey, k -> new ArrayList<>()).add(value);
                    }
                }

                // Store the earliest date in the month as representative date
                if (!monthStartDates.containsKey(monthKey)) {
                    monthStartDates.put(monthKey, entry.get("date"));
                } else {
                    String currentStart = monthStartDates.get(monthKey);
                    Date currentStartDate = dateFormat.parse(currentStart);
                    if (entryDate.before(currentStartDate)) {
                        monthStartDates.put(monthKey, entry.get("date"));
                    }
                }
            } catch (Exception e) {
                AppLogger.e(TAG, "Error processing entry for monthly aggregation: " + e.getMessage());
            }
        }

        // FIX: Generate ALL months in the date range INCLUDING current month
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDate);
        calendar.set(Calendar.DAY_OF_MONTH, 1);

        Calendar endCal = Calendar.getInstance();
        endCal.setTime(endDate);
        endCal.set(Calendar.DAY_OF_MONTH, 1);
        endCal.add(Calendar.MONTH, 1); // Move to first day of next month
        endCal.add(Calendar.MILLISECOND, -1); // Go back 1ms to stay in current month

        while (!calendar.after(endCal)) {
            String monthKey = monthFormat.format(calendar.getTime());
            Map<String, String> monthEntry = new HashMap<>();

            // FIX: For current month, use today's date or a recent date as representative
            String monthStartDate = monthStartDates.get(monthKey);
            if (monthStartDate != null) {
                monthEntry.put("date", monthStartDate);
            } else {
                // For months without data, use the 15th as a reasonable midpoint
                Calendar midMonth = Calendar.getInstance();
                midMonth.setTime(calendar.getTime());
                midMonth.set(Calendar.DAY_OF_MONTH, 15);
                monthEntry.put("date", dateFormat.format(midMonth.getTime()));
            }

            List<Integer> monthValues = monthlyValues.get(monthKey);
            if (monthValues != null && !monthValues.isEmpty()) {
                if (chartType == ChartType.PAI) {
                    int avgValue = (int) monthValues.stream().mapToInt(Integer::intValue).average().orElse(0);
                    monthEntry.put("pai_7_day", String.valueOf(avgValue));
                    monthEntry.put("pai_day", "0");
                } else {
                    // For RHR, use minimum value (best RHR of the month)
                    int minValue = monthValues.stream().mapToInt(Integer::intValue).min().orElse(0);
                    monthEntry.put("resting_heart_rate", String.valueOf(minValue));
                }
                monthEntry.put("aggregated", "month");
                monthEntry.put("data_points", String.valueOf(monthValues.size()));
            } else {
                // Create zero entries for months without data
                if (chartType == ChartType.PAI) {
                    monthEntry.put("pai_7_day", "0");
                    monthEntry.put("pai_day", "0");
                } else {
                    monthEntry.put("resting_heart_rate", "0");
                }
                monthEntry.put("aggregated", "month");
                monthEntry.put("data_points", "0");
            }

            monthlyData.add(monthEntry);
            calendar.add(Calendar.MONTH, 1);
        }

        // FIX: Ensure we have the current month if it's missing
        String currentMonthKey = monthFormat.format(new Date());
        boolean hasCurrentMonth = monthlyData.stream()
                .anyMatch(entry -> {
                    try {
                        Date entryDate = dateFormat.parse(entry.get("date"));
                        return monthFormat.format(entryDate).equals(currentMonthKey);
                    } catch (ParseException e) {
                        return false;
                    }
                });

        if (!hasCurrentMonth) {
            Map<String, String> currentMonthEntry = new HashMap<>();
            currentMonthEntry.put("date", dateFormat.format(new Date())); // Use today's date for current month
            if (chartType == ChartType.PAI) {
                currentMonthEntry.put("pai_7_day", "0");
                currentMonthEntry.put("pai_day", "0");
            } else {
                currentMonthEntry.put("resting_heart_rate", "0");
            }
            currentMonthEntry.put("aggregated", "month");
            currentMonthEntry.put("data_points", "0");
            monthlyData.add(currentMonthEntry);
        }

        AppLogger.d(TAG, "Monthly aggregation: " + monthlyData.size() + " months generated from " +
                dateFormat.format(startDate) + " to " + dateFormat.format(endDate) +
                " (includes current month: " + hasCurrentMonth + ")");

        return monthlyData;
    }

    // Unified gap filling - SAME LOGIC, DIFFERENT VALUE HANDLING
    private static List<Map<String, String>> fillGapsInAllTimeData(List<Map<String, String>> data, String today, ChartType chartType) {
        // [Keep the same implementation as before...]
        List<Map<String, String>> result = new ArrayList<>();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());

        if (data.isEmpty()) {
            // If no data at all, create entries for the last 30 days
            Calendar cal = Calendar.getInstance();
            cal.add(Calendar.DAY_OF_YEAR, -29); // Start from 30 days ago
            for (int i = 0; i < 30; i++) {
                String dateStr = dateFormat.format(cal.getTime());
                result.add(createEmptyEntry(dateStr, chartType));
                cal.add(Calendar.DAY_OF_YEAR, 1);
            }
            return result;
        }

        // Create lookup map
        Map<String, Map<String, String>> dataMap = new HashMap<>();
        for (Map<String, String> entry : data) {
            dataMap.put(entry.get("date"), entry);
        }

        // Find date range from actual data
        String earliestDate = data.get(0).get("date");
        String latestDate = data.get(0).get("date");

        for (Map<String, String> entry : data) {
            String entryDate = entry.get("date");
            if (entryDate.compareTo(earliestDate) < 0) {
                earliestDate = entryDate;
            }
            if (entryDate.compareTo(latestDate) > 0) {
                latestDate = entryDate;
            }
        }

        // Use today as end date if it's later than the latest data point
        String endDateStr = today.compareTo(latestDate) > 0 ? today : latestDate;

        // Generate ALL days from earliest to end date
        try {
            Date startDate = dateFormat.parse(earliestDate);
            Date endDate = dateFormat.parse(endDateStr);

            Calendar calendar = Calendar.getInstance();
            calendar.setTime(startDate);

            Calendar endCal = Calendar.getInstance();
            endCal.setTime(endDate);

            while (!calendar.after(endCal)) {
                String currentDateStr = dateFormat.format(calendar.getTime());

                if (dataMap.containsKey(currentDateStr)) {
                    result.add(dataMap.get(currentDateStr));
                } else {
                    result.add(createEmptyEntry(currentDateStr, chartType));
                }

                calendar.add(Calendar.DAY_OF_YEAR, 1);
            }

        } catch (ParseException e) {
            // Fallback: just return the original data
            AppLogger.e(TAG, "Error parsing dates in fillGapsInAllTimeData: " + e.getMessage());
            result = new ArrayList<>(data);
        }

        return result;
    }

    private static Map<String, String> createEmptyEntry(String date, ChartType chartType) {
        Map<String, String> entry = new HashMap<>();
        entry.put("date", date);

        if (chartType == ChartType.PAI) {
            entry.put("pai_7_day", "0");
            entry.put("pai_day", "0");
        } else {
            entry.put("resting_heart_rate", "0");
        }

        return entry;
    }

    // Core SVG generation - UPDATED WITH PROPER MATERIAL3 COLORS
    private static String generateSvgChart(List<Map<String, String>> data, int maxDays, String title,
                                           boolean is7DayChart, boolean isAllTimeChart, ChartType chartType,
                                           int containerWidth, int containerHeight) {
        if (data == null) data = new ArrayList<>();

        // Use container dimensions or fall back to defaults
        int chartWidth = containerWidth > 0 ? containerWidth : CHART_WIDTH;
        int chartHeight = containerHeight > 0 ? containerHeight : CHART_HEIGHT;

        List<Map<String, String>> chartData = getDataForTimeWindow(data, maxDays, isAllTimeChart, chartType);

        if (chartData.isEmpty()) {
            return generateEmptyChart(title, chartType, chartWidth, chartHeight);
        }

        // Calculate values based on chart type - ALWAYS START FROM 0 FOR BOTH
        int maxValue;
        if (chartType == ChartType.PAI) {
            maxValue = calculateMaxValuePai(chartData);
        } else {
            maxValue = calculateMaxValueRhr(chartData);
        }

        int chartAreaHeight = chartHeight - 2 * PADDING - 30;
        int chartAreaWidth = chartWidth - 2 * PADDING;

        StringBuilder svgBuilder = new StringBuilder();

        // SVG header with dynamic dimensions
        svgBuilder.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>")
                .append("<svg xmlns=\"http://www.w3.org/2000/svg\" ")
                .append("width=\"").append(chartWidth).append("\" ")
                .append("height=\"").append(chartHeight).append("\" ")
                .append("viewBox=\"0 0 ").append(chartWidth).append(" ").append(chartHeight).append("\">");

        // Background - transparent to blend with app background
        svgBuilder.append("<rect x=\"0\" y=\"0\" width=\"").append(chartWidth)
                .append("\" height=\"").append(chartHeight)
                .append("\" fill=\"").append(BACKGROUND_COLOR).append("\"/>");

        // Draw chart elements based on type
        if (chartType == ChartType.PAI) {
            drawAxes(svgBuilder, chartAreaWidth, chartAreaHeight, 0, maxValue, chartHeight, chartType);
            drawBars(svgBuilder, chartData, chartAreaWidth, chartAreaHeight, maxValue, is7DayChart, chartType, chartHeight);
            drawGridLines(svgBuilder, chartAreaWidth, chartAreaHeight, 0, maxValue, chartHeight, chartType);
            drawTargetLine(svgBuilder, chartAreaWidth, chartAreaHeight, maxValue, chartHeight);
        } else {
            drawAxes(svgBuilder, chartAreaWidth, chartAreaHeight, 0, maxValue, chartHeight, chartType);
            drawBars(svgBuilder, chartData, chartAreaWidth, chartAreaHeight, maxValue, is7DayChart, chartType, chartHeight);
            drawGridLines(svgBuilder, chartAreaWidth, chartAreaHeight, 0, maxValue, chartHeight, chartType);
        }

        // Labels
        if (is7DayChart) {
            draw7DayDataLabels(svgBuilder, chartData, chartAreaWidth, chartHeight);
        } else {
            drawDateLabels(svgBuilder, chartData, chartAreaWidth, isAllTimeChart, chartHeight);
        }

        svgBuilder.append("</svg>");
        return svgBuilder.toString();
    }

    private static List<Map<String, String>> getDataForTimeWindow(List<Map<String, String>> data, int maxDays, boolean isAllTimeChart, ChartType chartType) {
        // [Keep the same implementation as before...]
        List<Map<String, String>> result = new ArrayList<>();
        String today = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault()).format(new Date());

        if (isAllTimeChart) {
            // For all-time charts, use the aggregated data as-is
            result = new ArrayList<>(data);

            // Only add today for daily charts, not for weekly/monthly aggregated charts
            boolean isAggregated = !result.isEmpty() && result.get(0).containsKey("aggregated");
            if (!isAggregated) {
                // Only check for today if we're dealing with daily data
                boolean hasToday = false;
                for (Map<String, String> entry : result) {
                    if (today.equals(entry.get("date"))) {
                        hasToday = true;
                        break;
                    }
                }

                if (!hasToday) {
                    result.add(createEmptyEntry(today, chartType));
                }
            }
        } else {
            // For time-limited charts (7-day, 30-day), generate complete date range
            result = generateCompleteDateRange(maxDays, data, chartType);
        }

        return result;
    }

    private static List<Map<String, String>> generateCompleteDateRange(int maxDays, List<Map<String, String>> existingData, ChartType chartType) {
        // [Keep the same implementation as before...]
        List<Map<String, String>> result = new ArrayList<>();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());

        // Create a map for quick lookup of existing data
        Map<String, Map<String, String>> dataMap = new HashMap<>();
        for (Map<String, String> entry : existingData) {
            dataMap.put(entry.get("date"), entry);
        }

        // Generate dates from oldest to newest (today on the right)
        for (int i = maxDays - 1; i >= 0; i--) {
            Date date = new Date(System.currentTimeMillis() - (i * 24 * 60 * 60 * 1000L));
            String dateStr = dateFormat.format(date);

            if (dataMap.containsKey(dateStr)) {
                result.add(dataMap.get(dateStr));
            } else {
                result.add(createEmptyEntry(dateStr, chartType));
            }
        }

        return result;
    }

    // UNIFIED bar drawing with consistent positioning
    private static void drawBars(StringBuilder svgBuilder, List<Map<String, String>> data,
                                 int width, int height, int maxValue,
                                 boolean is7DayChart, ChartType chartType,
                                 int chartHeight) {
        int x = PADDING;
        int y = chartHeight - PADDING;
        int totalBars = data.size();

        // Calculate optimal bar width to use available space
        int barWidth;
        int actualBarSpacing = BAR_SPACING;
        int availableWidth = width;
        int maxPossibleBars = availableWidth / (MIN_BAR_WIDTH + actualBarSpacing);

        if (totalBars <= maxPossibleBars) {
            // We have enough space - use reasonable bar width
            barWidth = Math.max(MIN_BAR_WIDTH, (availableWidth - ((totalBars - 1) * actualBarSpacing)) / totalBars);
        } else {
            // Too many bars - use minimum width and reduce spacing
            barWidth = MIN_BAR_WIDTH;
            actualBarSpacing = Math.max(0, (availableWidth - (totalBars * barWidth)) / Math.max(1, totalBars - 1));
        }

        // FIXED: Calculate total width and center it
        int totalBarsWidth = (totalBars * barWidth) + (Math.max(0, totalBars - 1) * actualBarSpacing);
        int startOffset = Math.max(0, (availableWidth - totalBarsWidth) / 2);

        int fontSize = calculateFontSize(barWidth);

        for (int i = 0; i < totalBars; i++) {
            Map<String, String> entry = data.get(i);

            try {
                int value, barHeight, barX, barY;
                String barColor;

                if (chartType == ChartType.PAI) {
                    value = Integer.parseInt(entry.get("pai_7_day"));
                    barHeight = (int) ((double) value / maxValue * height);
                    barColor = getPaiBarColor(value);
                } else {
                    value = Integer.parseInt(entry.get("resting_heart_rate"));
                    // ALWAYS calculate from 0 for RHR
                    barHeight = (int) (((double) value / maxValue) * height);
                    barColor = getRhrBarColor(value);
                }

                // FIX: Use the same barWidth and actualBarSpacing that we calculated above
                barX = x + startOffset + i * (barWidth + actualBarSpacing);
                barY = y - barHeight;

                // Only draw if the bar fits within the chart area
                if (barX >= x && barX + barWidth <= x + width) {
                    if (value > 0 || chartType == ChartType.PAI) {
                        svgBuilder.append("<rect x=\"").append(barX).append("\" y=\"").append(barY)
                                .append("\" width=\"").append(barWidth).append("\" height=\"").append(barHeight)
                                .append("\" fill=\"").append(barColor).append("\"")
                                .append(chartType == ChartType.PAI ? " rx=\"3\"" : "").append("/>");

                        // Show value on top of bar for 7-day charts when there's enough space
                        if (is7DayChart && barHeight > 1 && barWidth > 20) {
                            int textY = barY - 5;
                            if (chartType == ChartType.PAI) {
                                value = replacePAIvalue(entry);
                            }
                            String displayValue = String.valueOf(value);

                            svgBuilder.append("<text x=\"").append(barX + barWidth / 2).append("\" y=\"").append(textY)
                                    .append("\" text-anchor=\"middle\" fill=\"").append(getTextColor()) // UPDATED HERE
                                    .append("\" font-size=\"").append(fontSize).append("\" font-weight=\"bold\">")
                                    .append(displayValue).append("</text>");
                        }
                    }
                }

            } catch (NumberFormatException e) {
                AppLogger.e(TAG, "Error parsing value: " + e.getMessage());
            }
        }
    }

    private static int replacePAIvalue(Map<String, String> entry){
        int value = Integer.parseInt(entry.get("pai_day"));
        return value;
    }

    private static int calculateFontSize(int width) {
        int fontSize = (int) (width * 0.5);
        return Math.max(fontSize, 10);
    }

    // UNIFIED label positioning - SAME FOR BOTH
    private static void drawDateLabels(StringBuilder svgBuilder, List<Map<String, String>> data,
                                       int width, boolean isAllTimeChart,
                                       int chartHeight) {
        int x = PADDING;
        int y = chartHeight - PADDING + LABEL_FONT_SIZE_DEFAULT;
        int totalBars = data.size();

        // FIXED: Use the EXACT SAME calculation as drawBars
        int barWidth;
        int actualBarSpacing = BAR_SPACING;
        int availableWidth = width;
        int maxPossibleBars = availableWidth / (MIN_BAR_WIDTH + actualBarSpacing);

        if (totalBars <= maxPossibleBars) {
            barWidth = Math.max(MIN_BAR_WIDTH, (availableWidth - ((totalBars - 1) * actualBarSpacing)) / totalBars);
        } else {
            barWidth = MIN_BAR_WIDTH;
            actualBarSpacing = Math.max(0, (availableWidth - (totalBars * barWidth)) / Math.max(1, totalBars - 1));
        }

        int totalBarsWidth = (totalBars * barWidth) + (Math.max(0, totalBars - 1) * actualBarSpacing);
        int startOffset = Math.max(0, (availableWidth - totalBarsWidth) / 2);

        int labelInterval = isAllTimeChart ? Math.max(1, totalBars / 12) : Math.max(1, totalBars / 7);

        for (int i = 0; i < totalBars; i += labelInterval) {
            Map<String, String> entry = data.get(i);
            String dateStr = entry.get("date");
            String displayDate = formatDateForDisplay(dateStr, entry);

            int labelX = x + startOffset + i * (barWidth + actualBarSpacing) + barWidth / 2;

            if (labelX >= x && labelX <= x + width) {
                svgBuilder.append("<text x=\"").append(labelX).append("\" y=\"").append(y)
                        .append("\" text-anchor=\"middle\" fill=\"").append(getTextColor())
                        .append("\" font-size=\"").append(LABEL_FONT_SIZE_DEFAULT * 0.7).append("\" transform=\"rotate(45, ")
                        .append(labelX).append(", ").append(y).append(")\">")
                        .append(displayDate).append("</text>");
            }
        }
    }

    private static void debugChartData(List<Map<String, String>> data, String chartType) {
        AppLogger.d(TAG, "=== " + chartType + " Chart Data Debug ===");
        AppLogger.d(TAG, "Total entries: " + data.size());

        if (!data.isEmpty()) {
            AppLogger.d(TAG, "First date: " + data.get(0).get("date"));
            AppLogger.d(TAG, "Last date: " + data.get(data.size() - 1).get("date"));

            // Count non-zero entries
            int nonZeroCount = 0;
            for (Map<String, String> entry : data) {
                String value = chartType.equals("PAI") ? entry.get("pai_7_day") : entry.get("resting_heart_rate");
                if (!value.equals("0")) {
                    nonZeroCount++;
                }
            }
            AppLogger.d(TAG, "Non-zero entries: " + nonZeroCount + "/" + data.size());
        }
    }

    private static void debugDateRange(List<Map<String, String>> data, String aggregationType) {
        if (!data.isEmpty()) {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
            String firstDate = data.get(0).get("date");
            String lastDate = data.get(data.size() - 1).get("date");
            AppLogger.d(TAG, aggregationType + " aggregation date range: " + firstDate + " to " + lastDate);

            // Check if we're missing recent data
            String today = dateFormat.format(new Date());
            if (!lastDate.equals(today)) {
                AppLogger.d(TAG, "WARNING: Aggregation ends at " + lastDate + " but today is " + today);
            }
        }
    }

    private static void draw7DayDataLabels(StringBuilder svgBuilder, List<Map<String, String>> data,
                                           int width, int chartHeight) {
        int x = PADDING;
        int y = chartHeight - PADDING + LABEL_FONT_SIZE_DEFAULT;
        int totalBars = data.size();
        int barWidth = (width - ((totalBars - 1) * BAR_SPACING)) / totalBars;

        for (int i = 0; i < totalBars; i++) {
            Map<String, String> entry = data.get(i);
            String dateStr = entry.get("date");
            String dayAbbreviation = getDayAbbreviationFromDate(dateStr);

            int labelX = x + i * (barWidth + BAR_SPACING) + barWidth / 2;

            svgBuilder.append("<text x=\"").append(labelX).append("\" y=\"").append(y)
                    .append("\" text-anchor=\"middle\" fill=\"").append(getTextColor())
                    .append("\" font-size=\"").append(LABEL_FONT_SIZE_DEFAULT).append("\">")
                    .append(dayAbbreviation).append("</text>");
        }
    }

    private static String getDayAbbreviationFromDate(String dateStr) {
        try {
            SimpleDateFormat inputFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
            SimpleDateFormat outputFormat = new SimpleDateFormat("E", Locale.getDefault());
            Date date = inputFormat.parse(dateStr);
            return outputFormat.format(date).substring(0, 2);
        } catch (Exception e) {
            return "??";
        }
    }

    private static String formatDateForDisplay(String dateStr, Map<String, String> entry) {
        try {
            SimpleDateFormat inputFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
            Date date = inputFormat.parse(dateStr);

            String aggregationLevel = entry.get("aggregated");

            if (aggregationLevel == null) {
                // Daily data - show day and month
                SimpleDateFormat outputFormat = new SimpleDateFormat("dd MM", Locale.getDefault());
                return outputFormat.format(date);
            } else if ("week".equals(aggregationLevel)) {
                // Weekly data - show start date of the week in dd.mm.yyyy format
                SimpleDateFormat outputFormat = new SimpleDateFormat("dd MM", Locale.getDefault());
                return outputFormat.format(date);
            } else if ("month".equals(aggregationLevel)) {
                // Monthly data - show first day of month in dd.mm.yyyy format
                SimpleDateFormat outputFormat = new SimpleDateFormat("dd MM", Locale.getDefault());
                return outputFormat.format(date);
            } else {
                // Fallback
                SimpleDateFormat outputFormat = new SimpleDateFormat("dd MM", Locale.getDefault());
                return outputFormat.format(date);
            }
        } catch (Exception e) {
            return dateStr.length() > 7 ? dateStr.substring(5) : dateStr;
        }
    }

    private static void drawAxes(StringBuilder svgBuilder, int width, int height, int minValue, int maxValue, int chartHeight, ChartType chartType) {
        int x = PADDING;
        int y = chartHeight - PADDING;

        // X-axis and Y-axis lines with proper theme colors
        svgBuilder.append("<line x1=\"").append(x).append("\" y1=\"").append(y)
                .append("\" x2=\"").append(x + width).append("\" y2=\"").append(y)
                .append("\" stroke=\"").append(getAxisColor()).append("\" stroke-width=\"2\"/>");

        svgBuilder.append("<line x1=\"").append(x).append("\" y1=\"").append(y)
                .append("\" x2=\"").append(x).append("\" y2=\"").append(y - height)
                .append("\" stroke=\"").append(getAxisColor()).append("\" stroke-width=\"2\"/>");

        // Use the unified step calculation with actual chart type
        int step = calculateOptimalStep(height, maxValue - minValue, chartType);

        // Y-axis labels - ALWAYS use the calculated step with proper text color
        for (int value = minValue; value <= maxValue; value += step) {
            int labelY = y - (int) (((double) (value - minValue) / (maxValue - minValue)) * height);

            if (labelY >= (y - height) && labelY <= y) {
                svgBuilder.append("<text x=\"").append(x - 8).append("\" y=\"").append(labelY + 4)
                        .append("\" text-anchor=\"end\" fill=\"").append(getTextColor())
                        .append("\" font-size=\"").append(LABEL_FONT_SIZE_DEFAULT).append("\">")
                        .append(value).append("</text>");
            }
        }
    }

    private static void drawGridLines(StringBuilder svgBuilder, int width, int height, int minValue, int maxValue, int chartHeight, ChartType chartType) {
        int x = PADDING;
        int y = chartHeight - PADDING;

        int step = calculateOptimalStep(height, maxValue - minValue, chartType);

        // Grid lines with proper theme colors
        for (int value = minValue + step; value <= maxValue; value += step) {
            int gridY = y - (int) (((double) (value - minValue) / (maxValue - minValue)) * height);

            if (gridY >= (y - height) && gridY <= y) {
                svgBuilder.append("<line x1=\"").append(x).append("\" y1=\"").append(gridY)
                        .append("\" x2=\"").append(x + width).append("\" y2=\"").append(gridY)
                        .append("\" stroke=\"").append(getGridLineColor()).append("\" ")
                        .append("stroke-width=\"2\" ")
                        .append("stroke-dasharray=\"4,3\" ")
                        .append("opacity=\"0.8\"/>");
            }
        }
    }

    private static void drawTargetLine(StringBuilder svgBuilder, int width, int height, int maxValue, int chartHeight) {
        int x = PADDING;
        int y = chartHeight - PADDING;
        int targetY = y - (int) ((100.0 / maxValue) * height);

        // Target line at value 100 for PAI
        svgBuilder.append("<line x1=\"").append(x).append("\" y1=\"").append(targetY)
                .append("\" x2=\"").append(x + width).append("\" y2=\"").append(targetY)
                .append("\" stroke=\"").append(TARGET_LINE_COLOR).append("\" stroke-width=\"4\" opacity=\"0.9\"/>");
    }

    private static int calculateMaxValuePai(List<Map<String, String>> data) {
        int max = 0;
        boolean hasValidData = false;

        for (Map<String, String> entry : data) {
            try {
                int value = Integer.parseInt(entry.get("pai_7_day"));
                if (value > max) max = value;
                hasValidData = true;
            } catch (NumberFormatException e) {
                AppLogger.e(TAG, "Error parsing PAI value: " + e.getMessage());
            }
        }

        if (!hasValidData) {
            return 100; // Default range when no data
        }

        // CONSISTENT: Add 20% padding and round to nearest clean interval
        int paddedMax = (int) Math.ceil(max * 1.2);
        return roundToNiceInterval(paddedMax);
    }

    private static int calculateMaxValueRhr(List<Map<String, String>> data) {
        int max = 0;
        boolean hasValidData = false;

        for (Map<String, String> entry : data) {
            try {
                int value = Integer.parseInt(entry.get("resting_heart_rate"));
                if (value > 0 && value <= 250) {
                    if (value > max) max = value;
                    hasValidData = true;
                }
            } catch (NumberFormatException e) {
                AppLogger.e(TAG, "Error parsing RHR value: " + e.getMessage());
            }
        }

        if (!hasValidData) {
            return 80;
        }

        // SIMPLER FIX: Add padding but cap at reasonable RHR values
        int paddedMax = (int) Math.ceil(max * 1.2);

        // Don't let the chart go much higher than needed for RHR
        if (max <= 60) {
            // For typical RHR values, cap at 70-80 range
            return Math.min(paddedMax, 80);
        } else if (max <= 70) {
            return Math.min(paddedMax, 90);
        } else {
            return Math.min(paddedMax, 120);
        }
    }

    // Unified step calculation for both chart types
    private static int roundToNiceInterval(int value) {
        // Round to visually pleasing intervals
        if (value <= 20) return 20;
        else if (value <= 50) return 50;
        else if (value <= 100) return 100;
        else if (value <= 150) return 150;
        else if (value <= 200) return 200;
        else return ((value + 49) / 50) * 50; // Round to nearest 50
    }

    private static int calculateOptimalStep(int height, int maxValue, ChartType chartType) {
        if (maxValue == 0) return 10;

        int maxLabels = Math.max(3, height / 40);
        int rawStep = (int) Math.ceil((double) maxValue / maxLabels);

        // Chart-type specific step preferences
        if (chartType == ChartType.RESTING_HEART_RATE) {
            // For RHR, prefer smaller steps since values are typically 40-100
            if (rawStep <= 5) return 5;
            else if (rawStep <= 10) return 10;
            else if (rawStep <= 20) return 20;
            else if (rawStep <= 30) return 30;
            else return 50;
        } else {
            // For PAI, use larger steps since values can be 0-200+
            if (rawStep <= 10) return 10;
            else if (rawStep <= 25) return 20;
            else if (rawStep <= 60) return 50;
            else if (rawStep <= 120) return 100;
            else return 200;
        }
    }

    // Color getters
    private static String getPaiBarColor(int paiValue) {
        if (paiValue == 0) return NO_PAI_COLOR;
        else if (paiValue < 50) return LOW_PAI_COLOR;
        else if (paiValue < 100) return MEDIUM_PAI_COLOR;
        else return HIGH_PAI_COLOR;
    }

    private static String getRhrBarColor(int restingHeartRate) {
        if (restingHeartRate <= ATHLETE_MAX) return ATHLETE_RHR_COLOR;
        else if (restingHeartRate <= EXCELLENT_MAX) return EXCELLENT_RHR_COLOR;
        else if (restingHeartRate <= IDEAL_MAX) return IDEAL_RHR_COLOR;
        else if (restingHeartRate <= AVERAGE_MAX) return AVERAGE_RHR_COLOR;
        else if (restingHeartRate <= ELEVATED_MAX) return ELEVATED_RHR_COLOR;
        else return HIGH_RISK_RHR_COLOR;
    }

    private static String generateEmptyChart(String title, ChartType chartType, int chartWidth, int chartHeight) {
        String message = chartType == ChartType.PAI ? "No PAI data available" : "No resting heart rate data available";

        return "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" +
                "<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"" + chartWidth +
                "\" height=\"" + chartHeight + "\" viewBox=\"0 0 " + chartWidth + " " + chartHeight + "\">" +
                "<rect x=\"0\" y=\"0\" width=\"" + chartWidth + "\" height=\"" + chartHeight +
                "\" fill=\"" + BACKGROUND_COLOR + "\"/>" +
                "<text x=\"" + (chartWidth / 2) + "\" y=\"" + (chartHeight / 2) +
                "\" text-anchor=\"middle\" fill=\"" + getTextColor() +
                "\" font-size=\"14\">" + message + "</text>" +
                "</svg>";
    }
}