package org.ojrandom.paiesque.data.sync;

import android.database.Cursor;
import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.util.Map;

/**
 * Base class for all query handlers providing common heart rate processing logic.
 * Elegant solution that avoids code duplication while allowing manufacturer-specific customizations.
 */
public abstract class BaseQueryHandler implements QueryHandler {
    protected static final String TAG = "BaseQueryHandler";

    /**
     * Default implementation for single-table queries.
     * Derived classes can override this for custom behavior.
     */
    @Override
    public String buildQuery(String tableName, long lastSyncTimestamp) {
        return "SELECT TIMESTAMP, DEVICE_ID, HEART_RATE " +
                "FROM " + tableName + " " +
                "WHERE TIMESTAMP > " + lastSyncTimestamp + " " +
                "ORDER BY TIMESTAMP";
    }

    /**
     * Default implementation for multi-table queries.
     * By default, uses the first table for single-table compatibility.
     * Derived classes should override this for JOIN queries.
     */
    @Override
    public String buildQuery(String[] tableNames, long lastSyncTimestamp) {
        if (tableNames.length == 1) {
            return buildQuery(tableNames[0], lastSyncTimestamp);
        }
        throw new UnsupportedOperationException(
                "This query handler doesn't support multiple tables. " +
                        "Override buildQuery(String[] tableNames, ...) to implement JOIN logic.");
    }

    /**
     * Creates a heart rate record with proper validation and device mapping.
     * Reusable across different query handlers.
     */
    protected HeartRateRecord createHeartRateRecord(long timestamp, String sourceDeviceId,
                                                    int heartRate, Map<String, Long> deviceMap,
                                                    String context) {
        Long targetDeviceId = findTargetDeviceId(sourceDeviceId, deviceMap, context);
        if (targetDeviceId == null) {
            return HeartRateRecord.invalid();
        }

        int correctedHr = correctHeartRate(heartRate);
        if (!isValidHeartRate(correctedHr)) {
            return HeartRateRecord.invalid();
        }

        return HeartRateRecord.valid(timestamp, targetDeviceId, correctedHr);
    }

    /**
     * Corrects heart rate values from device-specific formats to standard range.
     * Handles common issues like negative values in certain device protocols.
     *
     * Example
     *
     *  - Device sends: 0x96 (150 in decimal, normal heart rate)
     *  - Java interprets: 0x96 = -106 (because Java bytes are signed!)
     *  - Correction: -106 + 256 = 150
     *
     *  Byte Representation Problem
     *
     *  - Java bytes: Signed 8-bit integers (-128 to 127)
     *  - Device bytes: Often unsigned 8-bit integers (0 to 255)
     *  - Heart rates: Typically range from 30-220 bpm (fits in 0-255 range)
     */
    protected int correctHeartRate(int heartRate) {
        // See also https://codeberg.org/Freeyourgadget/Gadgetbridge/src/branch/master/app/src/main/java/nodomain/freeyourgadget/gadgetbridge/model/ActivitySample.java#L39
        // This interface defines hr = -1 for timestamps without hr measurement
        if (heartRate < -1) {
            int corrected = heartRate + 256;
            // Only return if it's a plausible heart rate after correction
            if (corrected >= 30 && corrected <= 220) {
                AppLogger.v(TAG, "Corrected heart rate from " + heartRate + " to " + corrected);
                return corrected;
            }
        }
        return heartRate;
    }

    /**
     * Validates heart rate values against physiological limits.
     * Filters out implausible values that could skew calculations.
     */
    protected boolean isValidHeartRate(int heartRate) {
        // Reasonable heart rate range for humans (30-220 BPM)
        boolean valid = heartRate >= 30 && heartRate <= 220;
        if (!valid) {
            //AppLogger.v(TAG, "Filtered out invalid heart rate: " + heartRate);
        }
        return valid;
    }

    /**
     * Finds target device ID from source device ID using the device mapping.
     * Provides consistent logging for device mapping issues.
     */
    protected Long findTargetDeviceId(String sourceDeviceId, Map<String, Long> deviceMap, String context) {
        Long targetDeviceId = deviceMap.get(sourceDeviceId);
        if (targetDeviceId == null) {
            AppLogger.d(TAG, "No device mapping for source ID: " + sourceDeviceId + " in context: " + context);
        }
        return targetDeviceId;
    }
}