package org.ojrandom.paiesque.data.repositories;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.util.Log;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;

public class TimestampRepository {
    private static final String TAG = "TimestampRepository";
    private final SQLiteDatabase database;

    public TimestampRepository(SQLiteDatabase database) {
        this.database = database;
    }

    // RHR methods using DATE approach
    public int getLastProcessedDateForRHR(int deviceId) {
        return getLastProcessedDate(AppConstants.PaiesqueTables.RHR_TIMESTAMPS, deviceId);
    }

    public void updateLastProcessedDateForRHR(int deviceId, int localDate) {
        updateLastProcessedDate(AppConstants.PaiesqueTables.RHR_TIMESTAMPS, deviceId, localDate);
    }

    // PAI methods using DATE approach
    public int getLastProcessedDateForPAI(int deviceId) {
        return getLastProcessedDate(AppConstants.PaiesqueTables.PAI_TIMESTAMPS, deviceId);
    }

    public void updateLastProcessedDateForPAI(int deviceId, int localDate) {
        updateLastProcessedDate(AppConstants.PaiesqueTables.PAI_TIMESTAMPS, deviceId, localDate);
    }

    private int getLastProcessedDate(String tableName, int deviceId) {
        String sql = "SELECT LAST_PROCESSED_DATE FROM " + tableName +
                " WHERE " + AppConstants.Columns.DEVICE_ID + " = ?";

        try (Cursor cursor = database.rawQuery(sql, new String[]{String.valueOf(deviceId)})) {
            if (cursor.moveToFirst()) {
                return cursor.getInt(0); // Returns YYYYMMDD format
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting last processed date from " + tableName, e);
        }

        return 0; // Return 0 if no record found
    }

    private void updateLastProcessedDate(String tableName, int deviceId, int localDate) {
        database.beginTransaction();
        try {
            String sql = "INSERT OR REPLACE INTO " + tableName +
                    " (" + AppConstants.Columns.DEVICE_ID + ", LAST_PROCESSED_DATE, LAST_UPDATED) VALUES (?, ?, strftime('%s','now'))";

            SQLiteStatement statement = database.compileStatement(sql);
            statement.bindLong(1, deviceId);
            statement.bindLong(2, localDate);
            statement.execute();

            database.setTransactionSuccessful();
            AppLogger.d(TAG, "Updated last processed date in " + tableName + " for device: " + deviceId + ": " + localDate);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error updating last processed date in " + tableName, e);
        } finally {
            database.endTransaction();
        }
    }
}