package org.ojrandom.paiesque.data.repositories;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.util.Log;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.rhr.RhrResult;

import java.time.LocalDate;
import java.util.Map;
import java.util.TreeMap;

public class RhrRepository {
    private static final String TAG = "RhrRepository";
    private final SQLiteDatabase database;

    public RhrRepository(SQLiteDatabase database) {
        this.database = database;
    }

    public synchronized TreeMap<LocalDate, Integer> loadHistoricalRHR(int deviceId) {
        TreeMap<LocalDate, Integer> historicalRHR = new TreeMap<>();

        String deviceFilter = deviceId != -1 ? " AND " + AppConstants.Columns.DEVICE_ID + " = " + deviceId : "";

        String sql = "SELECT LOCAL_DATE, RHR_VALUE FROM " +
                AppConstants.PaiesqueTables.RHR_VALUES +
                " WHERE 1=1" + deviceFilter +
                " ORDER BY LOCAL_DATE";

        Cursor cursor = null; // DECLARE HERE
        try {
            cursor = database.rawQuery(sql, null);
            while (cursor.moveToNext()) {
                int dateInt = cursor.getInt(0);
                int rhrValue = cursor.getInt(1);
                LocalDate date = convertIntToLocalDate(dateInt);
                historicalRHR.put(date, rhrValue);
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error loading historical RHR for device: " + deviceId, e);
        } finally {
            // ADD cursor cleanup:
            if (cursor != null) {
                try {
                    cursor.close();
                } catch (Exception e) {
                    AppLogger.e(TAG, "Error closing cursor", e);
                }
            }
        }
        return historicalRHR;
    }

    private int convertLocalDateToInt(LocalDate date) {
        return date.getYear() * 10000 + date.getMonthValue() * 100 + date.getDayOfMonth();
    }

    private LocalDate convertIntToLocalDate(int dateInt) {
        int year = dateInt / 10000;
        int month = (dateInt % 10000) / 100;
        int day = dateInt % 100;
        return LocalDate.of(year, month, day);
    }
    public synchronized void storeEnhancedDailyRHRValues(Map<LocalDate, RhrResult> dailyEnhancedRHR, int deviceId) {
        if (database == null) {
            AppLogger.e(TAG, "Database not initialized");
            return;
        }

        database.beginTransaction();
        try {
            // UPDATED SQL - REMOVED STRATEGY_USED COLUMN
            String insertSql = "INSERT OR REPLACE INTO " + AppConstants.PaiesqueTables.RHR_VALUES + " (" +
                    "DEVICE_ID, LOCAL_DATE, RHR_VALUE, SAMPLE_SIZE" +  // Removed STRATEGY_USED
                    ") VALUES (?, ?, ?, ?)";  // Now only 4 parameters instead of 5

            SQLiteStatement statement = database.compileStatement(insertSql);

            for (Map.Entry<LocalDate, RhrResult> entry : dailyEnhancedRHR.entrySet()) {
                LocalDate date = entry.getKey();
                RhrResult result = entry.getValue();

                int dateInt = convertLocalDateToInt(date);

                statement.bindLong(1, deviceId);
                statement.bindLong(2, dateInt);
                statement.bindLong(3, result.getRhrValue());
                statement.bindLong(4, result.getSampleSize());
                // REMOVED: statement.bindString(4, result.getStrategyUsed());

                statement.execute();
                statement.clearBindings();
            }

            database.setTransactionSuccessful();
            AppLogger.d(TAG, "Stored " + dailyEnhancedRHR.size() + " enhanced RHR values for device: " + deviceId);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error storing enhanced RHR values", e);
        } finally {
            database.endTransaction();
        }
    }

}