// PaiRepository.java
package org.ojrandom.paiesque.data.repositories;

import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;

import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.pai.PAIesqueCalculator;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;

public class PaiRepository {
    private static final String TAG = "PaiRepository";
    private final SQLiteDatabase database;

    public PaiRepository(SQLiteDatabase database) {
        this.database = database;
    }

    /**
     * Store daily PAI scores with zone breakdown in PAI_SCORES table
     */
    public synchronized void storeDailyPAIScoresWithZones(Map<LocalDate, PAIesqueCalculator.DailyPAIResult> dailyPAIResults, int deviceId) {
        if (database == null) {
            AppLogger.e(TAG, "Database not available");
            return;
        }

        database.beginTransaction();
        try {
            String sql = "INSERT OR REPLACE INTO " + AppConstants.PaiesqueTables.PAI_SCORES + " (" +
                    "LOCAL_DATE, DEVICE_ID, PAI_SCORE, " +
                    "PAI_ZONE_BELOW_THRESHOLD, PAI_ZONE_50_59, PAI_ZONE_60_69, PAI_ZONE_70_79, PAI_ZONE_80_89, PAI_ZONE_90_PLUS, " +
                    "TIME_ZONE_BELOW_THRESHOLD, TIME_ZONE_50_59, TIME_ZONE_60_69, TIME_ZONE_70_79, TIME_ZONE_80_89, TIME_ZONE_90_PLUS, " +
                    "LAST_UPDATED" +
                    ") VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

            SQLiteStatement statement = database.compileStatement(sql);

            for (Map.Entry<LocalDate, PAIesqueCalculator.DailyPAIResult> entry : dailyPAIResults.entrySet()) {
                LocalDate date = entry.getKey();
                PAIesqueCalculator.DailyPAIResult result = entry.getValue();

                int localDateInt = date.getYear() * 10000 + date.getMonthValue() * 100 + date.getDayOfMonth();

                statement.bindLong(1, localDateInt);
                statement.bindLong(2, deviceId);
                statement.bindDouble(3, result.getTotalPAI());

                // Bind zone PAI values
                statement.bindDouble(4, result.getPaiByZone().getOrDefault(PAIesqueCalculator.ZONE_BELOW_THRESHOLD, 0.0));
                statement.bindDouble(5, result.getPaiByZone().getOrDefault(PAIesqueCalculator.ZONE_50_59, 0.0));
                statement.bindDouble(6, result.getPaiByZone().getOrDefault(PAIesqueCalculator.ZONE_60_69, 0.0));
                statement.bindDouble(7, result.getPaiByZone().getOrDefault(PAIesqueCalculator.ZONE_70_79, 0.0));
                statement.bindDouble(8, result.getPaiByZone().getOrDefault(PAIesqueCalculator.ZONE_80_89, 0.0));
                statement.bindDouble(9, result.getPaiByZone().getOrDefault(PAIesqueCalculator.ZONE_90_PLUS, 0.0));

                // Bind zone time values
                statement.bindDouble(10, result.getSecondsByZone().getOrDefault(PAIesqueCalculator.ZONE_BELOW_THRESHOLD, 0.0));
                statement.bindDouble(11, result.getSecondsByZone().getOrDefault(PAIesqueCalculator.ZONE_50_59, 0.0));
                statement.bindDouble(12, result.getSecondsByZone().getOrDefault(PAIesqueCalculator.ZONE_60_69, 0.0));
                statement.bindDouble(13, result.getSecondsByZone().getOrDefault(PAIesqueCalculator.ZONE_70_79, 0.0));
                statement.bindDouble(14, result.getSecondsByZone().getOrDefault(PAIesqueCalculator.ZONE_80_89, 0.0));
                statement.bindDouble(15, result.getSecondsByZone().getOrDefault(PAIesqueCalculator.ZONE_90_PLUS, 0.0));

                statement.bindLong(16, System.currentTimeMillis() / 1000);

                statement.execute();
                statement.clearBindings();
            }

            database.setTransactionSuccessful();
            AppLogger.d(TAG, "Stored " + dailyPAIResults.size() + " PAI scores with zone breakdown");
        } catch (Exception e) {
            AppLogger.e(TAG, "Error storing PAI scores with zones", e);
        } finally {
            database.endTransaction();
        }
    }


    public synchronized void storeHistoricalRollingPAI(List<Map<String, String>> rollingPAI, int deviceId) {
        database.beginTransaction();
        try {
            String sql = "INSERT OR REPLACE INTO " + AppConstants.PaiesqueTables.ROLLING_PAI_HISTORY + " (" +
                    AppConstants.Columns.DEVICE_ID + ", " +
                    AppConstants.Columns.LOCAL_DATE + ", " +
                    AppConstants.Columns.ROLLING_PAI_SCORE +
                    ") VALUES (?, ?, ?)";

            SQLiteStatement statement = database.compileStatement(sql);

            for (Map<String, String> record : rollingPAI) {
                try {
                    LocalDate date = LocalDate.parse(record.get("date"));
                    double rollingPai = Double.parseDouble(record.get("pai_7_day"));

                    int calculationDateInt = date.getYear() * 10000 +
                            date.getMonthValue() * 100 +
                            date.getDayOfMonth();

                    statement.bindLong(1, deviceId);
                    statement.bindLong(2, calculationDateInt);
                    statement.bindDouble(3, rollingPai);

                    statement.execute();
                    statement.clearBindings();
                } catch (Exception e) {
                    AppLogger.w(TAG, "Error storing rolling PAI record: " + record, e);
                }
            }

            database.setTransactionSuccessful();
            AppLogger.d(TAG, "Stored " + rollingPAI.size() + " historical rolling PAI records for device: " + deviceId);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error storing historical rolling PAI for device: " + deviceId, e);
        } finally {
            database.endTransaction();
        }
    }
}