package org.ojrandom.paiesque.data.backup;

import android.content.ContentResolver;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.net.Uri;
import android.util.Log;

import org.ojrandom.paiesque.calculation.CalculationProgressListener;
import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.data.DataManager;
import org.ojrandom.paiesque.logging.AppLogger;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

public class BackupImportManager {
    private static final String TAG = "BackupImportManager";

    private final Context context;
    private final DataManager dataManager; // Use DataManager instead of PaiesqueDBManager
    private CalculationProgressListener progressListener;

    public BackupImportManager(Context context) {
        this.context = context;
        this.dataManager = DataManager.getInstance(context);
    }

    public void setProgressListener(CalculationProgressListener listener) {
        this.progressListener = listener;
    }

    public boolean importBackup(Uri backupUri) {
        File tempDir = context.getCacheDir();
        File backupDbFile = new File(tempDir, "paiesque_backup_import.db");

        try {
            ContentResolver resolver = context.getContentResolver();

            updateProgress("Copying backup file...", 0, 100);

            // Direct file copy from backup URI to temporary file
            if (!copyBackupToTempFile(backupUri, backupDbFile, resolver)) {
                AppLogger.e(TAG, "Failed to copy backup file");
                return false;
            }

            updateProgress("Analyzing backup data...", 30, 100);

            // Smart merge (same robust logic as before)
            return smartMergeBackupData(backupDbFile);

        } catch (Exception e) {
            AppLogger.e(TAG, "Error importing backup", e);
            updateProgress("Import failed!", 0, 100);
            return false;
        } finally {
            // Clean up temporary file
            if (backupDbFile.exists()) {
                backupDbFile.delete();
            }
        }
    }

    private boolean copyBackupToTempFile(Uri backupUri, File tempFile, ContentResolver resolver) {
        try (InputStream inputStream = resolver.openInputStream(backupUri);
             BufferedInputStream bufferedIn = new BufferedInputStream(inputStream, AppConstants.Backup.BUFFER_SIZE);
             FileOutputStream out = new FileOutputStream(tempFile);
             BufferedOutputStream bufferedOut = new BufferedOutputStream(out, AppConstants.Backup.BUFFER_SIZE)) {

            byte[] buffer = new byte[AppConstants.Backup.BUFFER_SIZE];
            int length;
            long bytesCopied = 0;

            while ((length = bufferedIn.read(buffer)) > 0) {
                bufferedOut.write(buffer, 0, length);
                bytesCopied += length;

                // Simple progress - assume most files are 10-50MB
                int progress = Math.min(30, (int)(bytesCopied / 100000)); // Rough estimate
                updateProgress("Copying backup file...", progress, 100);
            }

            bufferedOut.flush();
            return true;

        } catch (Exception e) {
            AppLogger.e(TAG, "Error copying backup file", e);
            return false;
        }
    }

    private boolean smartMergeBackupData(File backupDbFile) {
        // Use DataManager to get the database instance
        SQLiteDatabase currentDb = dataManager.getPaiesqueDBManager().getWritableDatabase();
        if (currentDb == null) {
            AppLogger.e(TAG, "Current database not available");
            return false;
        }

        SQLiteDatabase backupDb = null;

        try {
            // Open backup database as read-only
            backupDb = SQLiteDatabase.openDatabase(backupDbFile.getAbsolutePath(),
                    null, SQLiteDatabase.OPEN_READONLY);

            // Reset progress for device merge stage
            updateProgress("Merging devices...", 0, 100);

            // Step 1: Merge devices using IDENTIFIER matching
            Map<Integer, Integer> deviceIdMap = mergeDevices(currentDb, backupDb);

            // Reset progress for heart rate merge stage
            updateProgress("Merging heart rate data...", 0, 100);

            // Step 2: Merge heart rates with device ID translation
            int heartRatesImported = mergeHeartRates(currentDb, backupDb, deviceIdMap);

            updateProgress("Import completed successfully!", 100, 100);

            AppLogger.i(TAG, "Smart import completed: " + deviceIdMap.size() + " devices matched, " +
                    heartRatesImported + " heart rates imported");
            return true;

        } catch (Exception e) {
            AppLogger.e(TAG, "Error during smart merge", e);
            return false;
        } finally {
            if (backupDb != null && backupDb.isOpen()) {
                backupDb.close();
            }
        }
    }

    private Map<Integer, Integer> mergeDevices(SQLiteDatabase currentDb, SQLiteDatabase backupDb) {
        Map<Integer, Integer> deviceIdMap = new HashMap<>();
        Map<String, Integer> currentDevicesByIdentifier = new HashMap<>();

        // Step 1: Get all current devices by IDENTIFIER (same as Gadgetbridge sync)
        String currentDevicesSql = "SELECT " + AppConstants.Columns.ID + ", " +
                AppConstants.Columns.IDENTIFIER + " FROM " + AppConstants.Database.TABLE_DEVICE;

        try (Cursor cursor = currentDb.rawQuery(currentDevicesSql, null)) {
            while (cursor.moveToNext()) {
                int deviceId = cursor.getInt(0);
                String identifier = cursor.getString(1);
                currentDevicesByIdentifier.put(identifier, deviceId);
            }
        }

        // Step 2: Process backup devices using IDENTIFIER matching
        String backupDevicesSql = "SELECT " + AppConstants.Columns.ID + ", " +
                AppConstants.Columns.NAME + ", " + AppConstants.Columns.MANUFACTURER + ", " +
                AppConstants.Columns.IDENTIFIER + ", SOURCE_DEVICE_ID, CREATED_AT " +
                "FROM " + AppConstants.Database.TABLE_DEVICE;

        String insertSql = "INSERT OR IGNORE INTO " + AppConstants.Database.TABLE_DEVICE + " (" +
                AppConstants.Columns.NAME + ", " + AppConstants.Columns.MANUFACTURER + ", " +
                AppConstants.Columns.IDENTIFIER + ", SOURCE_DEVICE_ID, CREATED_AT) VALUES (?, ?, ?, ?, ?)";

        currentDb.beginTransaction();
        try {
            SQLiteStatement insertStmt = currentDb.compileStatement(insertSql);

            try (Cursor cursor = backupDb.rawQuery(backupDevicesSql, null)) {
                while (cursor.moveToNext()) {
                    int backupDeviceId = cursor.getInt(0);
                    String name = cursor.getString(1);
                    String manufacturer = cursor.getString(2);
                    String identifier = cursor.getString(3); // ← CRITICAL: Use IDENTIFIER
                    String sourceDeviceId = cursor.getString(4);
                    long createdAt = cursor.getLong(5);

                    // ✅ FIXED: Use IDENTIFIER matching (same as Gadgetbridge sync)
                    Integer currentDeviceId = currentDevicesByIdentifier.get(identifier);

                    if (currentDeviceId != null) {
                        // Device exists - map backup ID to current ID
                        deviceIdMap.put(backupDeviceId, currentDeviceId);
                        AppLogger.d(TAG, "✓ Matched device by IDENTIFIER: " + identifier +
                                " -> Backup ID " + backupDeviceId + " maps to Current ID " + currentDeviceId);
                    } else {
                        // New device - insert with original data
                        insertStmt.bindString(1, name);
                        insertStmt.bindString(2, manufacturer);
                        insertStmt.bindString(3, identifier);
                        insertStmt.bindString(4, sourceDeviceId);
                        insertStmt.bindLong(5, createdAt);

                        long newDeviceId = insertStmt.executeInsert();
                        if (newDeviceId != -1) {
                            deviceIdMap.put(backupDeviceId, (int) newDeviceId);
                            currentDevicesByIdentifier.put(identifier, (int) newDeviceId);
                            AppLogger.d(TAG, "✓ Created historic device: " + name + " (" + identifier +
                                    ") -> New ID: " + newDeviceId);
                        }

                        insertStmt.clearBindings();
                    }
                }
            }

            currentDb.setTransactionSuccessful();

        } finally {
            currentDb.endTransaction();
        }

        return deviceIdMap;
    }

    private int mergeHeartRates(SQLiteDatabase currentDb, SQLiteDatabase backupDb,
                                Map<Integer, Integer> deviceIdMap) {
        int importedCount = 0;

        // First, get total count for proper progress calculation
        int totalHeartRates = 0;
        String countSql = "SELECT COUNT(*) FROM " + AppConstants.PaiesqueTables.HEART_RATES;
        try (Cursor countCursor = backupDb.rawQuery(countSql, null)) {
            if (countCursor != null && countCursor.moveToFirst()) {
                totalHeartRates = countCursor.getInt(0);
            }
        }

        AppLogger.d(TAG, "Total heart rates to import: " + totalHeartRates);

        String insertSql = "INSERT OR IGNORE INTO " + AppConstants.PaiesqueTables.HEART_RATES + " (" +
                AppConstants.Columns.TIMESTAMP + ", " + AppConstants.Columns.DEVICE_ID + ", " +
                AppConstants.Columns.HEART_RATE + ", LOCAL_DATE, MINUTE_OF_DAY, SYNCED_AT) " +
                "VALUES (?, ?, ?, ?, ?, ?)";

        currentDb.beginTransaction();
        try {
            SQLiteStatement insertStmt = currentDb.compileStatement(insertSql);

            // Process in batches to avoid memory issues
            int offset = 0;
            boolean hasMore = true;
            int batchesProcessed = 0;
            int totalBatches = (int) Math.ceil((double) totalHeartRates / AppConstants.Backup.BATCH_SIZE);

            AppLogger.d(TAG, "Processing " + totalBatches + " batches of heart rate data");

            while (hasMore) {
                String backupHeartRatesSql = "SELECT " + AppConstants.Columns.TIMESTAMP + ", " +
                        AppConstants.Columns.DEVICE_ID + ", " + AppConstants.Columns.HEART_RATE + ", " +
                        "LOCAL_DATE, MINUTE_OF_DAY, SYNCED_AT " +
                        "FROM " + AppConstants.PaiesqueTables.HEART_RATES + " " +
                        "ORDER BY " + AppConstants.Columns.TIMESTAMP + " ASC " +
                        "LIMIT " + AppConstants.Backup.BATCH_SIZE + " OFFSET " + offset;

                try (Cursor cursor = backupDb.rawQuery(backupHeartRatesSql, null)) {
                    if (!cursor.moveToFirst()) {
                        hasMore = false;
                        continue;
                    }

                    batchesProcessed++;

                    // Update progress based on batches processed
                    int progress = (int) ((batchesProcessed * 100.0) / totalBatches);
                    updateProgress(String.format("Importing heart rates... (Batch %d/%d)",
                            batchesProcessed, totalBatches), progress, 100);

                    do {
                        long timestamp = cursor.getLong(0);
                        int backupDeviceId = cursor.getInt(1);
                        int heartRate = cursor.getInt(2);
                        int localDate = cursor.getInt(3);
                        int minuteOfDay = cursor.getInt(4);
                        long syncedAt = cursor.getLong(5);

                        // Translate device ID using our mapping
                        Integer currentDeviceId = deviceIdMap.get(backupDeviceId);
                        if (currentDeviceId == null) {
                            AppLogger.w(TAG, "Skipping heart rate with unmapped device ID: " + backupDeviceId);
                            continue;
                        }

                        // Insert with translated device ID
                        insertStmt.bindLong(1, timestamp);
                        insertStmt.bindLong(2, currentDeviceId);
                        insertStmt.bindLong(3, heartRate);
                        insertStmt.bindLong(4, localDate);
                        insertStmt.bindLong(5, minuteOfDay);
                        insertStmt.bindLong(6, syncedAt);

                        long rowId = insertStmt.executeInsert();
                        if (rowId != -1) {
                            importedCount++;
                        }

                        insertStmt.clearBindings();

                    } while (cursor.moveToNext());

                    offset += AppConstants.Backup.BATCH_SIZE;

                    AppLogger.d(TAG, String.format("Processed batch %d/%d, imported %d heart rates so far",
                            batchesProcessed, totalBatches, importedCount));
                }
            }

            currentDb.setTransactionSuccessful();

        } finally {
            currentDb.endTransaction();
        }

        return importedCount;
    }

    private void updateProgress(String message, int current, int total) {
        if (progressListener != null) {
            progressListener.onBatchProgress(current, total, message);
        }
    }
}