package org.ojrandom.paiesque.data;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;

import org.ojrandom.paiesque.logging.AppLogger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DeviceDiscoveryService {
    private static final String TAG = "DeviceDiscoveryService";

    public static class DiscoveryResult {
        public final Map<String, Long> deviceMap;
        public final Map<String, String> deviceNames; // Add this field
        public final List<String> discoveryLog;

        // Update constructor
        public DiscoveryResult(Map<String, Long> deviceMap,
                               Map<String, String> deviceNames, // Add this parameter
                               Map<String, Integer> manufacturerStats,
                               List<String> discoveryLog) {
            this.deviceMap = deviceMap;
            this.deviceNames = deviceNames; // Initialize the field
            this.discoveryLog = discoveryLog;
        }
    }

    public static DiscoveryResult discoverAndMapDevices(GadgetbridgeSyncSource sourceDbHelper, SQLiteDatabase targetDb) {
        Map<String, Long> deviceMap = new HashMap<>();
        Map<String, String> deviceNames = new HashMap<>();
        Map<String, Integer> manufacturerStats = new HashMap<>();
        List<String> discoveryLog = new ArrayList<>();

        try {
            String[] sourceColumns = {
                    AppConstants.Columns.ID,
                    AppConstants.Columns.NAME,
                    AppConstants.Columns.MANUFACTURER,
                    AppConstants.Columns.IDENTIFIER
            };

            List<Map<String, String>> sourceDevices = sourceDbHelper.readDataFromTable(
                    AppConstants.Database.TABLE_DEVICE, sourceColumns, null, null);

            discoveryLog.add("Found " + sourceDevices.size() + " devices in source database");

            for (Map<String, String> sourceDevice : sourceDevices) {
                String sourceId = sourceDevice.get(AppConstants.Columns.ID);
                String sourceName = sourceDevice.get(AppConstants.Columns.NAME);
                String sourceManufacturer = sourceDevice.get(AppConstants.Columns.MANUFACTURER);
                String sourceIdentifier = sourceDevice.get(AppConstants.Columns.IDENTIFIER);

                // Store device name
                if (sourceName != null && sourceId != null) {
                    deviceNames.put(sourceId, sourceName);
                }

                String discoveredManufacturer = AppConstants.Manufacturers.discoverManufacturer(
                        sourceName, sourceManufacturer);

                manufacturerStats.merge(discoveredManufacturer, 1, Integer::sum);

                String logEntry = String.format("Device: '%s' → Original: '%s' → Discovered: '%s'",
                        sourceName, sourceManufacturer, discoveredManufacturer);
                discoveryLog.add(logEntry);

                Long targetDeviceId = findOrCreateTargetDevice(
                        targetDb, sourceName, discoveredManufacturer, sourceIdentifier, sourceId);

                if (targetDeviceId != null) {
                    deviceMap.put(sourceId, targetDeviceId);
                    discoveryLog.add("  Mapped: source " + sourceId + " → target " + targetDeviceId);
                } else {
                    discoveryLog.add("  FAILED to map device: " + sourceName);
                }
            }

        } catch (Exception e) {
            String errorMsg = "Error in device discovery: " + e.getMessage();
            discoveryLog.add(errorMsg);
            AppLogger.e(TAG, errorMsg, e);
        }

        logDiscoverySummary(deviceMap, manufacturerStats, discoveryLog);

        return new DiscoveryResult(deviceMap, deviceNames, manufacturerStats, discoveryLog);
    }

    private static Long findOrCreateTargetDevice(SQLiteDatabase targetDb, String name,
                                                 String manufacturer, String identifier, String sourceId) {
        // Generate stable identifier if missing
        if (identifier == null || identifier.trim().isEmpty()) {
            identifier = generateStableIdentifier(name, manufacturer, sourceId);
        }

        // Try to find existing device
        Long existingId = findDeviceByIdentifier(targetDb, identifier);
        if (existingId != null) {
            return existingId;
        }

        // Create new device
        return createDevice(targetDb, name, manufacturer, identifier, sourceId);
    }

    private static String generateStableIdentifier(String name, String manufacturer, String sourceId) {
        String base = (name != null ? name : "Unknown") + "_" +
                (manufacturer != null ? manufacturer : "Generic");
        return base + "_" + sourceId; // Include source ID for uniqueness
    }

    private static Long findDeviceByIdentifier(SQLiteDatabase targetDb, String identifier) {
        String sql = "SELECT " + AppConstants.Columns.ID +
                " FROM " + AppConstants.Database.TABLE_DEVICE +
                " WHERE " + AppConstants.Columns.IDENTIFIER + " = ?";

        try (Cursor cursor = targetDb.rawQuery(sql, new String[]{identifier})) {
            if (cursor != null && cursor.moveToFirst()) {
                return cursor.getLong(0);
            }
        } catch (Exception e) {
            AppLogger.e(TAG, "Error finding device by identifier: " + identifier, e);
        }
        return null;
    }

    private static Long createDevice(SQLiteDatabase targetDb, String name, String manufacturer,
                                     String identifier, String sourceId) {
        String sql = "INSERT INTO " + AppConstants.Database.TABLE_DEVICE + " (" +
                AppConstants.Columns.NAME + ", " +
                AppConstants.Columns.MANUFACTURER + ", " +
                AppConstants.Columns.IDENTIFIER + ", " +
                "SOURCE_DEVICE_ID" +
                ") VALUES (?, ?, ?, ?)";

        try {
            android.database.sqlite.SQLiteStatement stmt = targetDb.compileStatement(sql);
            stmt.bindString(1, name != null ? name : "Unknown Device");
            stmt.bindString(2, manufacturer != null ? manufacturer : AppConstants.Manufacturers.GENERIC);
            stmt.bindString(3, identifier);
            stmt.bindString(4, sourceId);

            long newId = stmt.executeInsert();
            stmt.close();

            AppLogger.i(TAG, "Created new device: " + name + " (ID: " + newId + ")");
            return newId;

        } catch (Exception e) {
            AppLogger.e(TAG, "Error creating device: " + name, e);
            return null;
        }
    }

    private static void logDiscoverySummary(Map<String, Long> deviceMap,
                                            Map<String, Integer> manufacturerStats,
                                            List<String> discoveryLog) {
        discoveryLog.add("=== DISCOVERY SUMMARY ===");
        discoveryLog.add("Total devices mapped: " + deviceMap.size());
        discoveryLog.add("Manufacturer distribution:");

        for (Map.Entry<String, Integer> entry : manufacturerStats.entrySet()) {
            discoveryLog.add("  " + entry.getKey() + ": " + entry.getValue() + " devices");
        }

        // Log to system as well
        AppLogger.i(TAG, "Device discovery completed: " + deviceMap.size() + " devices mapped");
        for (String logEntry : discoveryLog) {
            AppLogger.i(TAG, logEntry);
        }
    }
}