package org.ojrandom.paiesque;

import android.animation.AnimatorSet;
import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.animation.BounceInterpolator;
import android.webkit.WebView;
import android.widget.TextView;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.appcompat.app.AppCompatActivity;

import com.google.android.material.floatingactionbutton.FloatingActionButton;
import com.google.android.material.snackbar.Snackbar;

public class ImportActivity extends AppCompatActivity {

    private ActivityResultLauncher<Intent> filePickerLauncher;
    private FloatingActionButton fab;
    private TextView fabLabel;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_import);

        // Setup WebView
        WebView webView = findViewById(R.id.webview_import);

        // Load appropriate language HTML - using modern API
        String currentLanguage = getResources().getConfiguration().getLocales().get(0).getLanguage();
        if (currentLanguage.equals("de")) {
            webView.loadUrl("file:///android_asset/gadgetbridge_import_de.html");
        } else {
            webView.loadUrl("file:///android_asset/gadgetbridge_import.html");
        }

        // Get FAB and label
        fab = findViewById(R.id.fab_choose_db);
        fabLabel = findViewById(R.id.fab_label);

        // Setup FAB click listener
        fab.setOnClickListener(v -> openFilePicker());

        // Make FAB super prominent
        makeFabProminent();

        // Initialize file picker launcher
        filePickerLauncher = registerForActivityResult(
                new ActivityResultContracts.StartActivityForResult(),
                result -> {
                    if (result.getResultCode() == RESULT_OK && result.getData() != null) {
                        Uri uri = result.getData().getData();
                        if (uri != null) {
                            Intent resultIntent = new Intent();
                            resultIntent.setData(uri);
                            setResult(RESULT_OK, resultIntent);
                            finish();
                        }
                    }
                }
        );

        // Show instruction after a short delay to ensure UI is fully loaded
        // FIX: Use Handler with Looper to avoid deprecation
        new Handler(Looper.getMainLooper()).postDelayed(this::showInitialInstruction, 1000);
    }

    private void makeFabProminent() {
        // Initial animation
        animateFabEntrance();

        // Pulsating animation to draw attention
        startPulsatingAnimation();

        // Show temporary instruction snackbar
        showInitialInstruction();
    }

    private void animateFabEntrance() {
        fab.setScaleX(0);
        fab.setScaleY(0);
        fabLabel.setAlpha(0);

        // Use ObjectAnimator for entrance animation too
        ObjectAnimator scaleXAnim = ObjectAnimator.ofFloat(fab, "scaleX", 0f, 1f);
        ObjectAnimator scaleYAnim = ObjectAnimator.ofFloat(fab, "scaleY", 0f, 1f);

        AnimatorSet entranceSet = new AnimatorSet();
        entranceSet.playTogether(scaleXAnim, scaleYAnim);
        entranceSet.setDuration(800);
        entranceSet.setInterpolator(new BounceInterpolator());
        entranceSet.setStartDelay(500);
        entranceSet.start();

        // Label fade in
        ObjectAnimator labelFadeIn = ObjectAnimator.ofFloat(fabLabel, "alpha", 0f, 1f);
        labelFadeIn.setDuration(500);
        labelFadeIn.setStartDelay(1300); // After the fab animation
        labelFadeIn.start();
    }

    private void startPulsatingAnimation() {
        // Scale animation with ObjectAnimator
        ObjectAnimator scaleAnimatorX = ObjectAnimator.ofFloat(fab, "scaleX", 1.0f, 1.15f);
        ObjectAnimator scaleAnimatorY = ObjectAnimator.ofFloat(fab, "scaleY", 1.0f, 1.15f);
        ObjectAnimator alphaAnimator = ObjectAnimator.ofFloat(fab, "alpha", 1.0f, 0.7f);

        // Set repeat on individual animators
        scaleAnimatorX.setRepeatCount(ValueAnimator.INFINITE);
        scaleAnimatorX.setRepeatMode(ValueAnimator.REVERSE);
        scaleAnimatorY.setRepeatCount(ValueAnimator.INFINITE);
        scaleAnimatorY.setRepeatMode(ValueAnimator.REVERSE);
        alphaAnimator.setRepeatCount(ValueAnimator.INFINITE);
        alphaAnimator.setRepeatMode(ValueAnimator.REVERSE);

        AnimatorSet fabAnimatorSet = new AnimatorSet();
        fabAnimatorSet.playTogether(scaleAnimatorX, scaleAnimatorY, alphaAnimator);
        fabAnimatorSet.setDuration(800);
        fabAnimatorSet.setInterpolator(new BounceInterpolator());
        fabAnimatorSet.start();

        // Label animation
        ObjectAnimator labelAnimator = ObjectAnimator.ofFloat(fabLabel, "alpha", 0.6f, 1.0f);
        labelAnimator.setDuration(1200);
        labelAnimator.setInterpolator(new BounceInterpolator());
        labelAnimator.setRepeatCount(ValueAnimator.INFINITE);
        labelAnimator.setRepeatMode(ValueAnimator.REVERSE);
        labelAnimator.start();
    }

    private void showInitialInstruction() {
        String message = getString(R.string.import_prominent_instruction);
        String actionText = getString(R.string.import_snackbar_action);

        // Use the FAB label as anchor to position Snackbar above the FAB
        Snackbar snackbar = Snackbar.make(
                fabLabel, // Anchor to the label above the FAB
                message,
                Snackbar.LENGTH_INDEFINITE
        );

        snackbar.setAction(actionText, v -> {
            fab.clearAnimation();
            fabLabel.clearAnimation();
            snackbar.dismiss();
        });

        // Set anchor view to position Snackbar above the FAB area
        snackbar.setAnchorView(fabLabel);

        snackbar.show();
    }

    private void openFilePicker() {
        // Stop animations when user interacts
        fab.clearAnimation();
        fabLabel.clearAnimation();

        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.setType("*/*");

        String[] mimeTypes = {
                "application/octet-stream",
                "application/sqlite",
                "application/x-sqlite3",
                "application/vnd.sqlite3"
        };
        intent.putExtra(Intent.EXTRA_MIME_TYPES, mimeTypes);
        intent.putExtra(Intent.EXTRA_TITLE, getString(R.string.import_file_picker_title));

        filePickerLauncher.launch(intent);
    }

    @Override
    protected void onResume() {
        super.onResume();
        // Restart animations if they were stopped
        if (fab != null && fab.getAnimation() == null) {
            startPulsatingAnimation();
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        // Clean up animations
        if (fab != null) {
            fab.clearAnimation();
        }
        if (fabLabel != null) {
            fabLabel.clearAnimation();
        }
    }
}