package org.nuclearfog.smither.utils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.Log;

import org.nuclearfog.smither.backend.proxy.ProxyAuthenticator;
import org.nuclearfog.smither.backend.proxy.UserProxy;
import org.nuclearfog.smither.config.GlobalSettings;

import java.io.File;
import java.security.cert.X509Certificate;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import okhttp3.Cache;
import okhttp3.OkHttpClient;

/**
 * OkHttp client builder
 *
 * @author nuclearfog
 */
public class ConnectionBuilder {

	/**
	 * cache folder name
	 */
	private static final String CACHE_FOLDER = "cache";


	private ConnectionBuilder() {
	}

	/**
	 * create OkHttp instance
	 *
	 * @param context application context to initialize
	 * @return OkHttpClient instance
	 */
	public static OkHttpClient create(Context context) {
		return create(context, 0);
	}

	/**
	 * create OkHttpClient instance with cache enabled
	 *
	 * @param context   application context to initialize
	 * @param cacheSize cache size
	 * @return OkHttpClient instance
	 */
	public static OkHttpClient create(Context context, int cacheSize) {
		// init okhttp client builder
		GlobalSettings settings = GlobalSettings.get(context);
		OkHttpClient.Builder builder = new OkHttpClient.Builder();
		builder.writeTimeout(60, TimeUnit.SECONDS).readTimeout(60, TimeUnit.SECONDS).connectTimeout(60, TimeUnit.SECONDS);
		// setup cache
		if (cacheSize > 0) {
			File cacheFolder = new File(context.getExternalCacheDir(), CACHE_FOLDER);
			if (!cacheFolder.exists())
				cacheFolder.mkdir();
			builder.cache(new Cache(cacheFolder, cacheSize));
		}
		// setup proxy
		builder.proxy(UserProxy.get(settings));
		builder.proxyAuthenticator(new ProxyAuthenticator(settings));
		if (settings.sslVerificationDisabled())
			allowAllCertificates(builder);
		return builder.build();
	}

	/**
	 * disable SSL verification for using private server instances (only by user)
	 */
	@SuppressLint("CustomX509TrustManager,TrustAllX509TrustManager")
	private static void allowAllCertificates(OkHttpClient.Builder builder) {
		try {
			// Create a trust manager that does not validate certificate chains
			TrustManager[] trustAllCerts = new TrustManager[]{
					new X509TrustManager() {
						@Override
						public void checkClientTrusted(X509Certificate[] chain, String authType) {
							Log.d("Cert", chain[0].toString());
						}

						@Override
						public void checkServerTrusted(X509Certificate[] chain, String authType) {
							Log.d("Cert", chain[0].toString());
						}

						@Override
						public X509Certificate[] getAcceptedIssuers() {
							return new X509Certificate[]{};
						}
					}
			};
			// Install the all-trusting trust manager
			final SSLContext sslContext = SSLContext.getInstance("SSL");
			sslContext.init(null, trustAllCerts, new java.security.SecureRandom());
			// Create an ssl socket factory with our all-trusting manager
			final SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();
			builder.sslSocketFactory(sslSocketFactory, (X509TrustManager) trustAllCerts[0]);

		} catch (Exception e) {
			// ignore
		}
	}
}