package org.nuclearfog.smither.ui.views;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.Matrix;
import android.graphics.drawable.BitmapDrawable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.view.View.OnLayoutChangeListener;
import android.widget.ImageView;

import androidx.annotation.IdRes;
import androidx.annotation.Nullable;

import org.nuclearfog.smither.R;

import jp.wasabeef.blurry.Blurry;

/**
 * Overlay view to blur a covered view
 *
 * @author nuclearfog
 */
@SuppressLint("AppCompatCustomView")
public class BlurView extends ImageView implements OnLayoutChangeListener {

	private static final String TAG = "BlurView";

	/**
	 * minimum required bitmap size
	 */
	private static final int MIN_SIZE = 32;

	@IdRes
	private int viewToBlurId;

	/**
	 *
	 */
	public BlurView(Context context) {
		this(context, null);
	}

	/**
	 *
	 */
	public BlurView(Context context, @Nullable AttributeSet attrs) {
		super(context, attrs);
		if (attrs != null) {
			TypedArray attrArray = context.obtainStyledAttributes(attrs, R.styleable.BlurView);
			viewToBlurId = attrArray.getResourceId(R.styleable.BlurView_to_blur, NO_ID);
			attrArray.recycle();
		}
		setScaleType(ScaleType.MATRIX);
	}


	@Override
	protected void onFinishInflate() {
		super.onFinishInflate();
		getRootView().post(() -> {
			View view = getRootView().findViewById(viewToBlurId);
			if (view != null) {
				view.setDrawingCacheEnabled(true);
				view.addOnLayoutChangeListener(BlurView.this);
			}
		});
	}

	/**
	 * Sets the scale type to "CropTop" which is similar to CenterCrop
	 *
	 * @see this <a href="https://gist.github.com/arriolac/3843346">source</a>.
	 */
	@Override
	protected boolean setFrame(int l, int t, int r, int b) {
		if (getDrawable() != null) {
			Matrix matrix = getImageMatrix();
			float scale;
			float viewWidth = getWidth() - getPaddingLeft() - getPaddingRight();
			float viewHeight = getHeight() - getPaddingTop() - getPaddingBottom();
			float drawableWidth = getDrawable().getIntrinsicWidth();
			float drawableHeight = getDrawable().getIntrinsicHeight();
			if (drawableWidth * viewHeight > drawableHeight * viewWidth) {
				scale = viewHeight / drawableHeight;
			} else {
				scale = viewWidth / drawableWidth;
			}
			matrix.setScale(scale, scale);
			setImageMatrix(matrix);
		}
		return super.setFrame(l, t, r, b);
	}

	/**
	 * Applies blurring background to this ImageView
	 */
	@Override
	public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
		if (v instanceof ImageView) {
			ImageView imageView = (ImageView) v;
			if (imageView.getDrawable() instanceof BitmapDrawable) {
				BitmapDrawable drawable = (BitmapDrawable) imageView.getDrawable();
				if (drawable.getBitmap() != null && drawable.getBitmap().getWidth() > MIN_SIZE && drawable.getBitmap().getHeight() > MIN_SIZE) {
					try {
						Bitmap image = drawable.getBitmap().copy(Bitmap.Config.ARGB_8888, false);
						float viewRatio = imageView.getMeasuredWidth() / (float) imageView.getMeasuredHeight();
						// center crop bitmap image
						if (image.getWidth() / (float) image.getHeight() < viewRatio) {
							int h = Math.round(image.getWidth() / viewRatio);
							int y = Math.round((image.getHeight() - h) / 2f);
							image = Bitmap.createBitmap(image, 0, y, image.getWidth(), h);
						} else {
							int w = Math.round(image.getHeight() * viewRatio);
							int x = Math.round((image.getWidth() - w) / 2f);
							image = Bitmap.createBitmap(image, x, 0, w, image.getHeight());
						}
						// blur image and apply to this view
						Blurry.with(getContext()).from(image).into(this);
					} catch (Exception exception) {
						// if an error occurred, remove background image and reset to default state
						setImageResource(0);
						Log.e(TAG, "error while creating blurred background!", exception);
					}
				}
			}
		}
	}
}