package org.nuclearfog.smither.ui.fragments;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.view.View;
import android.widget.Toast;

import androidx.activity.result.ActivityResult;
import androidx.activity.result.ActivityResultCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.backend.api.ConnectionException;
import org.nuclearfog.smither.backend.async.AsyncExecutor.AsyncCallback;
import org.nuclearfog.smither.backend.async.FollowRequestAction;
import org.nuclearfog.smither.backend.async.NotificationAction;
import org.nuclearfog.smither.backend.async.NotificationLoader;
import org.nuclearfog.smither.model.Notification;
import org.nuclearfog.smither.model.lists.Notifications;
import org.nuclearfog.smither.ui.activities.ProfileActivity;
import org.nuclearfog.smither.ui.activities.StatusActivity;
import org.nuclearfog.smither.ui.adapter.recyclerview.NotificationAdapter;
import org.nuclearfog.smither.ui.adapter.recyclerview.NotificationAdapter.OnNotificationClickListener;
import org.nuclearfog.smither.ui.dialogs.ConfirmDialog;
import org.nuclearfog.smither.ui.dialogs.ConfirmDialog.OnConfirmListener;
import org.nuclearfog.smither.utils.ErrorUtils;

/**
 * fragment to show notifications
 *
 * @author nuclearfog
 */
public class NotificationFragment extends ListFragment implements OnNotificationClickListener, OnConfirmListener, ActivityResultCallback<ActivityResult> {

	/**
	 * Bundle key used to save adapter items
	 * value type is {@link Notification[]}
	 */
	private static final String KEY_DATA = "notification-data";

	private ActivityResultLauncher<Intent> activityResultLauncher = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), this);
	private AsyncCallback<NotificationAction.Result> notificationActionCallback = this::onDismiss;
	private AsyncCallback<NotificationLoader.Result> notificationLoaderCallback = this::onNotificationResult;
	private AsyncCallback<FollowRequestAction.Result> followRequestCallback = this::onFollowRequestResult;

	private NotificationLoader notificationLoader;
	private NotificationAction notificationAction;
	private FollowRequestAction followAction;

	private NotificationAdapter adapter;

	@Nullable
	private Notification select;


	@Override
	public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
		super.onViewCreated(view, savedInstanceState);
		notificationLoader = new NotificationLoader(requireContext());
		notificationAction = new NotificationAction(requireContext());
		followAction = new FollowRequestAction(requireContext());
		adapter = new NotificationAdapter(this, settings.chronologicalTimelineEnabled());
		setAdapter(adapter, settings.chronologicalTimelineEnabled());

		if (savedInstanceState != null) {
			Object data = savedInstanceState.getSerializable(KEY_DATA);
			if (data instanceof Notifications) {
				adapter.setItems((Notifications) data);
			}
		}
	}


	@Override
	public void onSaveInstanceState(@NonNull Bundle outState) {
		if (adapter.getItemCount() < 100)
			outState.putSerializable(KEY_DATA, adapter.getItems());
		super.onSaveInstanceState(outState);
	}


	@Override
	public void onStart() {
		super.onStart();
		if (adapter.isEmpty()) {
			load(0L, 0L, 0);
			setRefresh(true);
		}
	}


	@Override
	public void onDestroy() {
		notificationLoader.cancel();
		notificationAction.cancel();
		followAction.cancel();
		super.onDestroy();
	}


	@Override
	protected void onReload() {
		if (isReversed()) {
			load(0, adapter.getTopItemId(), adapter.getItemCount());
		} else {
			load(adapter.getTopItemId(), 0, 0);
		}
	}


	@Override
	protected void onReset() {
		adapter = new NotificationAdapter(this, settings.chronologicalTimelineEnabled());
		setAdapter(adapter, settings.chronologicalTimelineEnabled());
		notificationLoader = new NotificationLoader(requireContext());
		notificationAction = new NotificationAction(requireContext());
		followAction = new FollowRequestAction(requireContext());
		load(0, 0, 0);
		setRefresh(true);
	}


	@Override
	public void onNotificationClick(Notification notification, int action) {
		if (!isRefreshing()) {
			switch (action) {
				case OnNotificationClickListener.NOTIFICATION_VIEW_STATUS:
					Intent intent = new Intent(requireContext(), StatusActivity.class);
					intent.putExtra(StatusActivity.KEY_DATA, notification);
					activityResultLauncher.launch(intent);
					break;

				case OnNotificationClickListener.NOTIFICATION_DISMISS:
					if (notificationAction.isIdle()) {
						if (ConfirmDialog.show(this, ConfirmDialog.NOTIFICATION_DISMISS, null)) {
							select = notification;
						}
					}
					break;

				case OnNotificationClickListener.NOTIFICATION_USER:
					if (notification.getType() == Notification.TYPE_REQUEST) {
						if (ConfirmDialog.show(this, ConfirmDialog.FOLLOW_REQUEST, null)) {
							select = notification;
						}
						break;
					}
					// fall through

				case OnNotificationClickListener.NOTIFICATION_VIEW_USER:
					intent = new Intent(requireContext(), ProfileActivity.class);
					intent.putExtra(ProfileActivity.KEY_USER, notification.getUser());
					startActivity(intent);
					break;
			}
		}
	}


	@Override
	public boolean onPlaceholderClick(long sinceId, long maxId, int position) {
		if (!isRefreshing() && notificationLoader.isIdle()) {
			load(sinceId, maxId, position);
			return true;
		}
		return false;
	}


	@Override
	public void onActivityResult(ActivityResult result) {
		Intent intent = result.getData();
		if (intent != null) {
			switch (result.getResultCode()) {
				case StatusActivity.RETURN_NOTIFICATION_UPDATE:
					Object data = intent.getSerializableExtra(StatusActivity.KEY_DATA);
					if (data instanceof Notification) {
						Notification update = (Notification) data;
						adapter.updateItem(update);
					}
					break;

				case StatusActivity.RETURN_NOTIFICATION_REMOVED:
					long notificationId = intent.getLongExtra(StatusActivity.KEY_NOTIFICATION_ID, 0L);
					adapter.removeItem(notificationId);
					break;
			}
		}
	}


	@Override
	public void onConfirm(int type) {
		if (type == ConfirmDialog.NOTIFICATION_DISMISS) {
			if (select != null) {
				NotificationAction.Param param = new NotificationAction.Param(NotificationAction.Param.DISMISS, select.getId());
				notificationAction.execute(param, notificationActionCallback);
			}
		} else if (type == ConfirmDialog.FOLLOW_REQUEST) {
			if (select != null && select.getUser() != null) {
				FollowRequestAction.Param param = new FollowRequestAction.Param(FollowRequestAction.Param.ACCEPT, select.getUser().getId(), select.getId());
				followAction.execute(param, followRequestCallback);
			}
		}
	}

	/**
	 * used by {@link NotificationLoader} to set notification items
	 */
	private void onNotificationResult(@NonNull NotificationLoader.Result result) {
		if (result.notifications != null) {
			adapter.addItems(result.notifications, result.position);
		} else {
			Context context = getContext();
			if (context != null) {
				ErrorUtils.showErrorMessage(context, result.exception);
			}
			adapter.disableLoading();
		}
		setRefresh(false);
	}

	/**
	 * used by {@link FollowRequestAction} to accept a follow request
	 */
	private void onFollowRequestResult(FollowRequestAction.Result result) {
		Context context = getContext();
		if (context != null) {
			if (result.action == FollowRequestAction.Result.ACCEPT) {
				Toast.makeText(context, R.string.info_follow_request_accepted, Toast.LENGTH_SHORT).show();
				adapter.removeItem(result.notification_id);
			} else if (result.action == FollowRequestAction.Result.ERROR) {
				ErrorUtils.showErrorMessage(context, result.exception);
			}
		}
	}

	/**
	 * called by {@link NotificationAction} to take action on a selected notification
	 */
	private void onDismiss(@NonNull NotificationAction.Result result) {
		if (result.action == NotificationAction.Result.DISMISS) {
			adapter.removeItem(result.id);
		} else if (result.action == NotificationAction.Result.ERROR) {
			Context context = getContext();
			if (context != null) {
				ErrorUtils.showErrorMessage(context, result.exception);
			}
			if (result.exception != null && result.exception.getErrorCode() == ConnectionException.RESOURCE_NOT_FOUND) {
				adapter.removeItem(result.id);
			}
		}
	}

	/**
	 * @param minId lowest notification ID to load
	 * @param maxId highest notification Id to load
	 * @param pos   index to insert the new items
	 */
	private void load(long minId, long maxId, int pos) {
		NotificationLoader.Param param = new NotificationLoader.Param(NotificationLoader.Param.LOAD_ALL, pos, minId, maxId);
		notificationLoader.execute(param, notificationLoaderCallback);
	}
}