package org.nuclearfog.smither.ui.adapter.recyclerview.holder;

import android.graphics.drawable.ColorDrawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.ImageView;

import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.RecyclerView.ViewHolder;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.Media;
import org.nuclearfog.smither.utils.AppStyles;
import org.nuclearfog.smither.utils.BlurHashDecoder;
import org.nuclearfog.smither.utils.ImageWorker;

/**
 * holder class for {@link org.nuclearfog.smither.ui.adapter.recyclerview.PreviewAdapter}
 *
 * @author nucleaerfog
 */
public class MediaHolder extends ViewHolder implements OnClickListener {

	/**
	 * empty placeholder image color
	 */
	private static final int EMPTY_COLOR = 0x1f000000;

	private ImageView previewImage, playIcon;

	private ImageWorker imageWorker;
	private GlobalSettings settings;
	private OnHolderClickListener listener;

	private Media media;

	/**
	 *
	 */
	public MediaHolder(ViewGroup parent, OnHolderClickListener listener) {
		super(LayoutInflater.from(parent.getContext()).inflate(R.layout.item_preview, parent, false));
		settings = GlobalSettings.get(parent.getContext());
		imageWorker = ImageWorker.get(parent.getContext());
		previewImage = itemView.findViewById(R.id.item_preview_image);
		playIcon = itemView.findViewById(R.id.item_preview_play);
		this.listener = listener;

		previewImage.getLayoutParams().width = parent.getMeasuredHeight();

		previewImage.setOnClickListener(this);
	}


	@Override
	public void onClick(View v) {
		int pos = getLayoutPosition();
		if (pos != RecyclerView.NO_POSITION) {
			if (v == previewImage) {
				listener.onItemClick(pos, OnHolderClickListener.PREVIEW_CLICK);
			}
		}
	}

	/**
	 * set view content
	 *
	 * @param media       media content
	 * @param blurPreview true to blur the preview image
	 */
	public void setContent(Media media, boolean blurPreview) {
		// skip if same media is already set
		if (!media.equals(this.media)) {
			if (settings.imagesEnabled() && media.getMediaType() != Media.AUDIO && media.getMediaType() != Media.UNDEFINED && !media.getPreviewUrl().isEmpty()) {
				// calculate image ratio
				float ratio = 1.0f;
				if (media.getMeta() != null && media.getMeta().getWidthPreview() > 0 && media.getMeta().getHeightPreview() > 0) {
					ratio = media.getMeta().getWidthPreview() / (float) media.getMeta().getHeightPreview();
				}
				if (blurPreview) {
					// use integrated blur generator
					if (!media.getBlurHash().isEmpty()) {
						previewImage.setImageBitmap(BlurHashDecoder.decode(media.getBlurHash(), ratio));
					} else {
						previewImage.setImageDrawable(new ColorDrawable(EMPTY_COLOR));
					}
				} else {
					imageWorker.loadMediaImage(previewImage, media.getPreviewUrl(), media.getBlurHash(), ratio);
				}
			} else {
				previewImage.setImageDrawable(new ColorDrawable(EMPTY_COLOR));
			}
			switch (media.getMediaType()) {
				case Media.AUDIO:
				case Media.VIDEO:
					playIcon.setVisibility(View.VISIBLE);
					playIcon.setImageResource(R.drawable.play);
					break;

				case Media.GIF:
					playIcon.setVisibility(View.VISIBLE);
					playIcon.setImageResource(R.drawable.gif);
					break;

				default:
					playIcon.setVisibility(View.GONE);
					break;
			}
			AppStyles.setDrawableColor(playIcon, settings.getIconColor());
			this.media = media;
		}
	}
}