package org.nuclearfog.smither.ui.adapter.recyclerview.holder;

import android.content.res.Resources;
import android.graphics.Typeface;
import android.graphics.drawable.ColorDrawable;
import android.text.SpannableStringBuilder;
import android.text.style.StyleSpan;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.RecyclerView.ViewHolder;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.Card;
import org.nuclearfog.smither.utils.BlurHashDecoder;
import org.nuclearfog.smither.utils.ImageWorker;

/**
 * adapter item for {@link org.nuclearfog.smither.ui.adapter.recyclerview.PreviewAdapter}
 *
 * @author nuclearfog
 */
public class CardHolder extends ViewHolder implements OnClickListener {

	/**
	 * empty placeholder image color
	 */
	private static final int EMPTY_COLOR = 0x1f000000;

	/**
	 * link text background transparency
	 */
	private static final int TEXT_TRANSPARENCY = 0xafffffff;

	/**
	 * maximum char count of the title before truncating
	 */
	private static final int TITLE_MAX_LEN = 30;

	private TextView linkText;
	private ImageView preview;

	private ImageWorker imageWorker;
	private GlobalSettings settings;
	private OnHolderClickListener listener;

	private Card card;

	/**
	 *
	 */
	public CardHolder(ViewGroup parent, OnHolderClickListener listener) {
		super(LayoutInflater.from(parent.getContext()).inflate(R.layout.item_card, parent, false));
		linkText = itemView.findViewById(R.id.link_preview_text);
		preview = itemView.findViewById(R.id.link_preview_image);
		settings = GlobalSettings.get(parent.getContext());
		imageWorker = ImageWorker.get(parent.getContext());
		this.listener = listener;

		itemView.getLayoutParams().width = Resources.getSystem().getDisplayMetrics().widthPixels * 2 / 3;
		linkText.setTypeface(settings.getTypeFace());
		linkText.setTextColor(settings.getTextColor());
		linkText.setBackgroundColor(settings.getBackgroundColor() & TEXT_TRANSPARENCY);

		linkText.setOnClickListener(this);
		preview.setOnClickListener(this);
	}


	@Override
	public void onClick(View v) {
		int pos = getLayoutPosition();
		if (pos != RecyclerView.NO_POSITION) {
			if (v == linkText) {
				listener.onItemClick(pos, OnHolderClickListener.CARD_LINK);
			} else if (v == preview) {
				listener.onItemClick(pos, OnHolderClickListener.CARD_IMAGE);
			}
		}
	}

	/**
	 * set view content
	 *
	 * @param card        card content
	 * @param blurPreview true to blur the preview image
	 */
	public void setContent(Card card, boolean blurPreview) {
		if (!card.equals(this.card)) {
			SpannableStringBuilder urlDescription = new SpannableStringBuilder();
			// set url preview image
			if (settings.imagesEnabled() && !card.getImageUrl().isEmpty()) {
				if (blurPreview) {
					if (!card.getBlurHash().isEmpty()) {
						preview.setImageBitmap(BlurHashDecoder.decode(card.getBlurHash()));
					} else {
						preview.setImageDrawable(new ColorDrawable(EMPTY_COLOR));
					}
				} else {
					imageWorker.loadMediaImage(preview, card.getImageUrl(), card.getBlurHash(), 1.0f);
				}
			} else {
				preview.setImageDrawable(new ColorDrawable(EMPTY_COLOR));
			}
			// set url title and truncate if needed
			if (!card.getTitle().isEmpty()) {
				// truncate title
				if (card.getTitle().length() > TITLE_MAX_LEN) {
					urlDescription.append(card.getTitle().substring(0, TITLE_MAX_LEN - 3));
					urlDescription.append("...");
					urlDescription.setSpan(new StyleSpan(Typeface.BOLD), 0, TITLE_MAX_LEN, 0);
				} else {
					urlDescription.append(card.getTitle());
					urlDescription.setSpan(new StyleSpan(Typeface.BOLD), 0, card.getTitle().length(), 0);
				}
			}
			// set url description
			if (!card.getDescription().isEmpty()) {
				urlDescription.append('\n');
				urlDescription.append(card.getDescription());
			}
			// apply description
			linkText.setText(urlDescription);
			this.card = card;
		}
	}
}