package org.nuclearfog.smither.ui.activities;

import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuItem;
import android.view.ViewGroup;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.SearchView;
import androidx.appcompat.widget.SearchView.OnQueryTextListener;
import androidx.appcompat.widget.Toolbar;
import androidx.viewpager2.widget.ViewPager2;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.backend.api.ConnectionException;
import org.nuclearfog.smither.backend.async.AsyncExecutor.AsyncCallback;
import org.nuclearfog.smither.backend.async.UserListAction;
import org.nuclearfog.smither.backend.async.UserListManager;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.UserList;
import org.nuclearfog.smither.ui.adapter.viewpager.UserListAdapter;
import org.nuclearfog.smither.ui.dialogs.ConfirmDialog;
import org.nuclearfog.smither.ui.dialogs.ConfirmDialog.OnConfirmListener;
import org.nuclearfog.smither.ui.dialogs.UserListDialog;
import org.nuclearfog.smither.ui.dialogs.UserListDialog.UserListUpdatedCallback;
import org.nuclearfog.smither.ui.views.TabSelector;
import org.nuclearfog.smither.ui.views.TabSelector.OnTabSelectedListener;
import org.nuclearfog.smither.utils.AppStyles;
import org.nuclearfog.smither.utils.ErrorUtils;
import org.nuclearfog.smither.utils.LandscapePageTransformer;

import java.util.regex.Pattern;

/**
 * This activity shows content of an user list
 * like timeline, list member and subscriber
 *
 * @author nuclearfog
 */
public class UserListActivity extends AppCompatActivity implements OnTabSelectedListener, OnQueryTextListener, OnConfirmListener, UserListUpdatedCallback {

	/**
	 * key to add list information
	 * value type is {@link UserList}
	 */
	public static final String KEY_DATA = "list_data";

	/**
	 * result key to return the ID of a removed list
	 * value type is {@link UserList}
	 */
	public static final String KEY_ID = "removed-list-id";

	/**
	 * return code when an user list was deleted
	 */
	public static final int RETURN_LIST_REMOVED = 0xDAD518B4;

	/**
	 * return code when an user list was deleted
	 */
	public static final int RETURN_LIST_UPDATED = 0x5D0F5E8D;

	/**
	 * regex pattern to validate username
	 * e.g. username, @username or @username@instance.social
	 */
	private static final Pattern USERNAME_PATTERN = Pattern.compile("@?\\w{1,20}(@[\\w.]{1,50})?");

	private AsyncCallback<UserListAction.Result> userListRemoveCallback = this::onUserListRemoved;
	private AsyncCallback<UserListManager.Result> userListUpdateCallback = this::onUserListUpdate;


	private UserListAdapter adapter;
	private UserListAction listLoaderAsync;
	private UserListManager listManagerAsync;

	private GlobalSettings settings;

	private ViewPager2 viewPager;
	private Toolbar toolbar;

	@Nullable
	private UserList userList;


	@Override
	protected void attachBaseContext(Context newBase) {
		super.attachBaseContext(AppStyles.setFontScale(newBase));
	}


	@Override
	protected void onCreate(@Nullable Bundle b) {
		super.onCreate(b);
		setContentView(R.layout.page_tab_view);
		ViewGroup root = findViewById(R.id.page_tab_view_root);
		TabSelector tabSelector = findViewById(R.id.page_tab_view_tabs);
		toolbar = findViewById(R.id.page_tab_view_toolbar);
		viewPager = findViewById(R.id.page_tab_view_pager);

		settings = GlobalSettings.get(this);
		listLoaderAsync = new UserListAction(this);
		listManagerAsync = new UserListManager(this);
		adapter = new UserListAdapter(this);

		Object data = getIntent().getSerializableExtra(KEY_DATA);
		if (data instanceof UserList) {
			userList = (UserList) data;
			toolbar.setTitle(userList.getTitle());
			adapter.setId(userList.getId());
			viewPager.setAdapter(adapter);
			if (adapter.getItemCount() == 2) {
				tabSelector.addTabIcons(R.array.list_tab_icons);
			} else if (adapter.getItemCount() == 3) {
				tabSelector.addTabIcons(R.array.list_subscriber_tab_icons);
			}
		}
		viewPager.setOffscreenPageLimit(3);
		viewPager.setPageTransformer(new LandscapePageTransformer());
		tabSelector.setLargeIndicator(getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE);
		setSupportActionBar(toolbar);
		AppStyles.setTheme(root);

		tabSelector.addOnTabSelectedListener(this);
	}


	@Override
	protected void onDestroy() {
		listLoaderAsync.cancel();
		super.onDestroy();
	}


	@Override
	public boolean onCreateOptionsMenu(@NonNull Menu m) {
		getMenuInflater().inflate(R.menu.userlist, m);
		AppStyles.setMenuIconColor(m, settings.getIconColor());
		AppStyles.setOverflowIcon(toolbar, settings.getIconColor());
		return true;
	}


	@Override
	public boolean onPrepareOptionsMenu(Menu m) {
		MenuItem search = m.findItem(R.id.menu_list_add_user);
		SearchView searchUser = (SearchView) search.getActionView();
		if (searchUser != null) {
			AppStyles.setTheme(searchUser);
			searchUser.setQueryHint(getString(R.string.menu_add_user));
			searchUser.setOnQueryTextListener(this);
		}
		return true;
	}


	@Override
	public boolean onOptionsItemSelected(@NonNull MenuItem item) {
		if (userList != null && listLoaderAsync.isIdle()) {
			// open user list editor
			if (item.getItemId() == R.id.menu_list_edit) {
				UserListDialog.show(this, userList);
				return true;
			}
			// delete user list
			else if (item.getItemId() == R.id.menu_delete_list) {
				ConfirmDialog.show(this, ConfirmDialog.LIST_DELETE, null);
				return true;
			}
			// theme expanded search view
			else if (item.getItemId() == R.id.menu_list_add_user) {
				SearchView searchView = (SearchView) item.getActionView();
				if (searchView != null) {
					AppStyles.setTheme(searchView);
				}
				return true;
			}
		}
		return super.onOptionsItemSelected(item);
	}


	@Override
	public void onBackPressed() {
		if (viewPager.getCurrentItem() > 0) {
			viewPager.setCurrentItem(0);
		} else {
			Intent result = new Intent();
			result.putExtra(KEY_DATA, userList);
			setResult(RETURN_LIST_UPDATED, result);
			super.onBackPressed();
		}
	}


	@Override
	public void onConfirm(int type) {
		// delete user list
		if (type == ConfirmDialog.LIST_DELETE && userList != null) {
			if (listLoaderAsync.isIdle()) {
				UserListAction.Param param = new UserListAction.Param(UserListAction.Param.DELETE, userList.getId());
				listLoaderAsync.execute(param, userListRemoveCallback);
			}
		}
	}


	@Override
	public void onTabSelected() {
		adapter.scrollToTop();
	}


	@Override
	public boolean onQueryTextChange(String newText) {
		return false;
	}


	@Override
	public boolean onQueryTextSubmit(String query) {
		if (userList == null)
			return false;
		if (USERNAME_PATTERN.matcher(query).matches()) {
			if (listManagerAsync.isIdle()) {
				Toast.makeText(getApplicationContext(), R.string.info_adding_user_to_list, Toast.LENGTH_SHORT).show();
				UserListManager.Param param = new UserListManager.Param(UserListManager.Param.ADD, userList.getId(), query);
				listManagerAsync.execute(param, userListUpdateCallback);
				return true;
			}
		} else {
			Toast.makeText(getApplicationContext(), R.string.error_username_format, Toast.LENGTH_SHORT).show();
		}
		return false;
	}


	@Override
	public void onUserListUpdate(UserList userList) {
		this.userList = userList;
		toolbar.setTitle(this.userList.getTitle());
		invalidateOptionsMenu();
	}

	/**
	 * update user list member
	 */
	private void onUserListUpdate(@NonNull UserListManager.Result result) {
		switch (result.action) {
			case UserListManager.Result.ADD_USER:
				String name;
				if (!result.name.startsWith("@"))
					name = '@' + result.name;
				else
					name = result.name;
				String info = getString(R.string.info_user_added_to_list, name);
				Toast.makeText(getApplicationContext(), info, Toast.LENGTH_SHORT).show();
				invalidateOptionsMenu();
				break;

			case UserListManager.Result.ERROR:
				ErrorUtils.showErrorMessage(getApplicationContext(), result.exception);
				break;
		}
	}

	/**
	 * update user list content
	 */
	private void onUserListRemoved(@NonNull UserListAction.Result result) {
		if (result.action == UserListAction.Result.DELETE) {
			Intent intent = new Intent();
			intent.putExtra(KEY_ID, result.id);
			setResult(RETURN_LIST_REMOVED, intent);
			Toast.makeText(getApplicationContext(), R.string.info_list_removed, Toast.LENGTH_SHORT).show();
			finish();
		} else if (result.action == UserListAction.Result.ERROR) {
			ErrorUtils.showErrorMessage(getApplicationContext(), result.exception);
			if (result.exception != null && result.exception.getErrorCode() == ConnectionException.RESOURCE_NOT_FOUND) {
				// List does not exist
				Intent intent = new Intent();
				intent.putExtra(KEY_ID, result.id);
				setResult(RETURN_LIST_REMOVED, intent);
				finish();
			}
		}
	}
}