package org.nuclearfog.smither.database.impl;

import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.nuclearfog.smither.database.DatabaseAdapter.StatusPropertiesTable;
import org.nuclearfog.smither.database.DatabaseAdapter.StatusTable;
import org.nuclearfog.smither.model.Account;
import org.nuclearfog.smither.model.Card;
import org.nuclearfog.smither.model.Emoji;
import org.nuclearfog.smither.model.Location;
import org.nuclearfog.smither.model.Media;
import org.nuclearfog.smither.model.Poll;
import org.nuclearfog.smither.model.Status;
import org.nuclearfog.smither.model.User;

import java.util.regex.Pattern;

/**
 * Implementation of a database STATUS
 *
 * @author nuclearfog
 */
public class DatabaseStatus implements Status, StatusTable, StatusPropertiesTable {

	private static final long serialVersionUID = -5957556706939766801L;

	private static final Pattern KEY_SEPARATOR = Pattern.compile(";");

	private long id, time, embeddedId, replyID, replyUserId, myRepostId, locationId, pollId, editedAt;
	private int repostCount, favoriteCount, replyCount, visibility;
	private boolean reposted, favored, bookmarked, sensitive, spoiler, isHidden, isPinned;
	private Status embedded;
	private Poll poll;
	private User author;
	private Location location;
	private String[] mediaKeys = {};
	private String[] emojiKeys = {};
	private Media[] medias = {};
	private Emoji[] emojis = {};
	private Card[] cards = {};
	private String replyName = "";
	private String text = "";
	private String source = "";
	private String userMentions = "";
	private String statusUrl = "";
	private String language = "";

	/**
	 * @param cursor  database cursor
	 * @param account current user login information
	 */
	public DatabaseStatus(Cursor cursor, Account account) {
		author = new DatabaseUser(cursor, account);
		time = cursor.getLong(cursor.getColumnIndexOrThrow(TIME));
		repostCount = cursor.getInt(cursor.getColumnIndexOrThrow(REPOST));
		favoriteCount = cursor.getInt(cursor.getColumnIndexOrThrow(FAVORITE));
		replyCount = cursor.getInt(cursor.getColumnIndexOrThrow(REPLY));
		id = cursor.getLong(cursor.getColumnIndexOrThrow(ID));
		replyID = cursor.getLong(cursor.getColumnIndexOrThrow(REPLYSTATUS));
		locationId = cursor.getLong(cursor.getColumnIndexOrThrow(LOCATION));
		pollId = cursor.getLong(cursor.getColumnIndexOrThrow(POLL));
		replyUserId = cursor.getLong(cursor.getColumnIndexOrThrow(REPLYUSER));
		embeddedId = cursor.getLong(cursor.getColumnIndexOrThrow(EMBEDDED));
		myRepostId = cursor.getLong(cursor.getColumnIndexOrThrow(REPOST_ID));
		editedAt = cursor.getLong(cursor.getColumnIndexOrThrow(EDITED_AT));
		String statusUrl = cursor.getString(cursor.getColumnIndexOrThrow(URL));
		String language = cursor.getString(cursor.getColumnIndexOrThrow(LANGUAGE));
		String source = cursor.getString(cursor.getColumnIndexOrThrow(SOURCE));
		String text = cursor.getString(cursor.getColumnIndexOrThrow(TEXT));
		String replyName = cursor.getString(cursor.getColumnIndexOrThrow(REPLYNAME));
		String userMentions = cursor.getString(cursor.getColumnIndexOrThrow(MENTIONS));
		String mediaKeys = cursor.getString(cursor.getColumnIndexOrThrow(MEDIA));
		String emojiKeys = cursor.getString(cursor.getColumnIndexOrThrow(EMOJI));
		int flags = cursor.getInt(cursor.getColumnIndexOrThrow(FLAGS));

		favored = (flags & MASK_STATUS_FAVORED) != 0;
		reposted = (flags & MASK_STATUS_REPOSTED) != 0;
		sensitive = (flags & MASK_STATUS_SENSITIVE) != 0;
		isHidden = (flags & MASK_STATUS_HIDDEN) != 0;
		bookmarked = (flags & MASK_STATUS_BOOKMARKED) != 0;
		spoiler = (flags & MASK_STATUS_SPOILER) != 0;
		isPinned = (flags & MASK_STATUS_PINNED) != 0;
		if ((flags & MASK_STATUS_VISIBILITY_DIRECT) != 0)
			visibility = VISIBLE_DIRECT;
		else if ((flags & MASK_STATUS_VISIBILITY_PRIVATE) != 0)
			visibility = VISIBLE_PRIVATE;
		else if ((flags & MASK_STATUS_VISIBILITY_UNLISTED) != 0)
			visibility = VISIBLE_UNLISTED;
		else
			visibility = VISIBLE_PUBLIC;
		if (mediaKeys != null && !mediaKeys.isEmpty())
			this.mediaKeys = KEY_SEPARATOR.split(mediaKeys);
		if (emojiKeys != null && !emojiKeys.isEmpty())
			this.emojiKeys = KEY_SEPARATOR.split(emojiKeys);
		if (statusUrl != null)
			this.statusUrl = statusUrl;
		if (language != null)
			this.language = language;
		if (source != null)
			this.source = source;
		if (text != null)
			this.text = text;
		if (replyName != null)
			this.replyName = replyName;
		if (userMentions != null)
			this.userMentions = userMentions;
	}


	@Override
	public long getId() {
		return id;
	}


	@Override
	public String getText() {
		return text;
	}


	@Override
	public User getAuthor() {
		return author;
	}


	@Override
	public long getTimestamp() {
		return time;
	}


	@Override
	public String getSource() {
		return source;
	}


	@Nullable
	@Override
	public Status getEmbeddedStatus() {
		return embedded;
	}


	@Override
	public String getReplyName() {
		return replyName;
	}


	@Override
	public long getRepliedUserId() {
		return replyUserId;
	}


	@Override
	public long getRepliedStatusId() {
		return replyID;
	}


	@Override
	public long getRepostId() {
		return myRepostId;
	}


	@Override
	public int getRepostCount() {
		return repostCount;
	}


	@Override
	public int getFavoriteCount() {
		return favoriteCount;
	}


	@Override
	public int getReplyCount() {
		return replyCount;
	}


	@Override
	public int getVisibility() {
		return visibility;
	}


	@Override
	public long editedAt() {
		return editedAt;
	}


	@NonNull
	@Override
	public Media[] getMedia() {
		return medias;
	}


	@NonNull
	@Override
	public Emoji[] getEmojis() {
		return emojis;
	}


	@Override
	public String getUserMentions() {
		return userMentions;
	}


	@Override
	public String getLanguage() {
		return language;
	}


	@Override
	public boolean isSensitive() {
		return sensitive;
	}


	@Override
	public boolean isSpoiler() {
		return spoiler;
	}


	@Override
	public boolean isReposted() {
		return reposted;
	}


	@Override
	public boolean isFavored() {
		return favored;
	}


	@Override
	public boolean isBookmarked() {
		return bookmarked;
	}


	@Override
	public boolean isPinned() {
		return isPinned;
	}


	@Override
	@Nullable
	public Location getLocation() {
		return location;
	}


	@Override
	public boolean isHidden() {
		return isHidden;
	}


	@Override
	public String getUrl() {
		return statusUrl;
	}


	@NonNull
	@Override
	public Card[] getCards() {
		return cards;
	}


	@Nullable
	@Override
	public Poll getPoll() {
		return poll;
	}


	@Override
	public boolean equals(@Nullable Object obj) {
		if (!(obj instanceof Status))
			return false;
		Status status = ((Status) obj);
		return status.getId() == getId() && status.getTimestamp() == getTimestamp() && status.getAuthor().equals(getAuthor());
	}


	@NonNull
	@Override
	public String toString() {
		return "from=\"" + getAuthor().getScreenName() + "\" text=\"" + getText() + "\"";
	}

	/**
	 * get ID of the embedded status
	 *
	 * @return ID of the
	 */
	public long getEmbeddedStatusId() {
		return embeddedId;
	}

	/**
	 * @return media keys
	 */
	public String[] getMediaKeys() {
		return mediaKeys;
	}

	/**
	 * @return emoji keys
	 */
	public String[] getEmojiKeys() {
		return emojiKeys;
	}

	/**
	 * @return location ID
	 */
	public long getLocationId() {
		return locationId;
	}

	/**
	 * @return ID of an attached poll or '0'
	 */
	public long getPollId() {
		return pollId;
	}

	/**
	 * attach status referenced by {@link #embeddedId}
	 *
	 * @param embedded embedded status
	 */
	public void setEmbeddedStatus(Status embedded) {
		this.embedded = embedded;
	}

	/**
	 * add status media
	 *
	 * @param medias media array
	 */
	public void addMedia(@NonNull Media[] medias) {
		this.medias = medias;
	}

	/**
	 * add status emojis
	 *
	 * @param emojis emoji array
	 */
	public void addEmojis(@NonNull Emoji[] emojis) {
		this.emojis = emojis;
	}

	/**
	 * add location information
	 *
	 * @param location location item
	 */
	public void addLocation(Location location) {
		this.location = location;
	}

	/**
	 * add poll
	 *
	 * @param poll poll item
	 */
	public void addPoll(@Nullable Poll poll) {
		this.poll = poll;
	}
}