package org.nuclearfog.smither.config;

import androidx.annotation.ArrayRes;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.model.Account;

/**
 * Configurations for different networks, containing static configuration
 *
 * @author nuclearfog
 */
public enum Configuration {

	/**
	 * configurations for Mastodon
	 */
	MASTODON(Account.API_MASTODON);

	private final String name;
	private final int accountType;
	private final boolean searchFilterEnabled;
	private final boolean profileLocationEnabled;
	private final boolean profileUrlEnabled;
	private final boolean postLocationSupported;
	private final boolean notificationDismissSupported;
	private final boolean statusSpoilerSupported;
	private final boolean statusVisibilitySupported;
	private final boolean emojiSupported;
	private final boolean statusEditSupported;
	private final boolean webPushSupported;
	private final boolean filterSupported;
	private final boolean publicTimelineSupported;
	private final boolean userListSubscriberSupported;
	private final boolean userListMembershipSupported;
	private final boolean outgoingFollowRequestSupported;
	private final int arrayResHome;

	/**
	 * @param accountType account login type, see {@link Account}
	 */
	Configuration(int accountType) {
		this.accountType = accountType;
		switch (accountType) {
			default:
			case Account.API_MASTODON:
				name = "Mastodon";
				searchFilterEnabled = false;
				profileLocationEnabled = false;
				profileUrlEnabled = false;
				postLocationSupported = false;
				notificationDismissSupported = true;
				statusSpoilerSupported = true;
				statusVisibilitySupported = true;
				emojiSupported = true;
				statusEditSupported = true;
				webPushSupported = true;
				filterSupported = true;
				publicTimelineSupported = true;
				userListSubscriberSupported = false;
				userListMembershipSupported = false;
				outgoingFollowRequestSupported = false;
				arrayResHome = R.array.home_mastodon_tab_icons;
				break;
		}
	}

	/**
	 * @return network name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @return account login type, see {@link Account}
	 */
	public int getAccountType() {
		return accountType;
	}

	/**
	 * @return true if search filter option is enabled
	 */
	public boolean filterEnabled() {
		return searchFilterEnabled;
	}

	/**
	 * @return true if network supports profile location information
	 */
	public boolean profileLocationEnabled() {
		return profileLocationEnabled;
	}

	/**
	 * @return true if network supports profile url information
	 */
	public boolean profileUrlEnabled() {
		return profileUrlEnabled;
	}

	/**
	 * @return true if posting location is supported
	 */
	public boolean locationSupported() {
		return postLocationSupported;
	}

	/**
	 * @return true if notification dismiss is supported
	 */
	public boolean notificationDismissEnabled() {
		return notificationDismissSupported;
	}

	/**
	 * @return true if login type supports warning for status spoiler
	 */
	public boolean statusSpoilerSupported() {
		return statusSpoilerSupported;
	}

	/**
	 * @return true if login type supports status visibility states
	 */
	public boolean statusVisibilitySupported() {
		return statusVisibilitySupported;
	}

	/**
	 * @return true if text emojis are supported
	 */
	public boolean isEmojiSupported() {
		return emojiSupported;
	}

	/**
	 * @return true if status edit is supported
	 */
	public boolean isStatusEditSupported() {
		return statusEditSupported;
	}

	/**
	 * @return true if network supports push subscription
	 */
	public boolean isWebPushSupported() {
		return webPushSupported;
	}

	/**
	 * @return true if a public timeline is supported
	 */
	public boolean isPublicTimelineSupported() {
		return publicTimelineSupported;
	}

	/**
	 * @return true if showing subscribers of an user list is supported
	 */
	public boolean isUserListSubscriberSupported() {
		return userListSubscriberSupported;
	}

	/**
	 * @return true if showing user list members is supported
	 */
	public boolean isUserListMembershipSupported() {
		return userListMembershipSupported;
	}

	/**
	 * @return true if showing outgoing follow request is supported
	 */
	public boolean isOutgoingFollowRequestSupported() {
		return outgoingFollowRequestSupported;
	}

	/**
	 * @return true if status filter is supported
	 */
	public boolean isFilterSupported() {
		return filterSupported;
	}

	/**
	 * get home tab items drawable IDs
	 *
	 * @return Integer array resource containing drawable IDs
	 */
	@ArrayRes
	public int getHomeTabIcons() {
		return arrayResHome;
	}
}