package org.nuclearfog.smither.backend.api.mastodon.impl;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.json.JSONException;
import org.json.JSONObject;
import org.nuclearfog.smither.model.Notification;
import org.nuclearfog.smither.model.Status;
import org.nuclearfog.smither.model.User;
import org.nuclearfog.smither.utils.StringUtils;

/**
 * notification implementation for Mastodon
 *
 * @author nuclearfog
 */
public class MastodonNotification implements Notification {

	private static final long serialVersionUID = 4113306729125959429L;

	private long id;
	private long timestamp;
	private int type;
	private User user;
	private Status status;

	/**
	 * @param json      notification json object
	 * @param currentId Id of the current user
	 */
	public MastodonNotification(JSONObject json, long currentId) throws JSONException {
		String idStr = json.getString("id");
		String typeStr = json.getString("type");
		JSONObject statusJson = json.optJSONObject("status");
		JSONObject userJson = json.getJSONObject("account");
		timestamp = StringUtils.getIsoTime(json.getString("created_at"));
		user = new MastodonUser(userJson, currentId);

		switch (typeStr) {
			case "mention":
				type = TYPE_MENTION;
				break;

			case "status":
				type = TYPE_STATUS;
				break;

			case "reblog":
				type = TYPE_REPOST;
				break;

			case "follow":
				type = TYPE_FOLLOW;
				break;

			case "follow_request":
				type = TYPE_REQUEST;
				break;

			case "favourite":
				type = TYPE_FAVORITE;
				break;

			case "poll":
				type = TYPE_POLL;
				break;

			case "update":
				type = TYPE_UPDATE;
				break;

			case "annual_report":
				type = TYPE_ANNUAL_REPORT;
				break;
		}
		if (statusJson != null) {
			status = new MastodonStatus(statusJson, currentId);
		}
		try {
			id = Long.parseLong(idStr);
		} catch (NumberFormatException e) {
			throw new JSONException("bad ID:" + idStr);
		}
	}


	@Override
	public long getId() {
		return id;
	}


	@Override
	public int getType() {
		return type;
	}


	@Override
	public long getTimestamp() {
		return timestamp;
	}


	@Override
	public User getUser() {
		return user;
	}


	@Nullable
	@Override
	public Status getStatus() {
		return status;
	}


	@Override
	public boolean equals(@Nullable Object obj) {
		if (!(obj instanceof Notification))
			return false;
		return ((Notification) obj).getId() == getId();
	}


	@NonNull
	@Override
	public String toString() {
		return "id=" + getId() + " " + getUser();
	}
}