package org.nuclearfog.smither.utils;

import android.content.Context;
import android.widget.ImageView;

import androidx.annotation.Nullable;

import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.config.GlobalSettings.SettingsChangeObserver;

import coil3.BitmapImage;
import coil3.ComponentRegistry;
import coil3.ImageLoader;
import coil3.Uri;
import coil3.gif.GifDecoder;
import coil3.network.okhttp.OkHttpNetworkFetcher;
import coil3.request.CachePolicy;
import coil3.request.ImageRequest;
import coil3.target.ImageViewTarget;
import kotlin.jvm.JvmClassMappingKt;
import kotlin.reflect.KClass;
import okhttp3.OkHttpClient;

/**
 * Create image loader instance with proxy connection
 *
 * @author nuclearfog
 */
public class ImageWorker implements SettingsChangeObserver {


	private static final KClass<Uri> URI = JvmClassMappingKt.getKotlinClass(Uri.class);

	private static ImageWorker instance;
	private static boolean notifySettingsChange = false;

	private ImageLoader imageLoader;

	/**
	 *
	 */
	private ImageWorker(Context context) {
		notifySettingsChange = false;
		GlobalSettings settings = GlobalSettings.get(context);
		OkHttpClient client = ConnectionBuilder.create(context);
		ComponentRegistry componentRegistry = new ComponentRegistry().newBuilder()
				.add(new GifDecoder.Factory()).add(OkHttpNetworkFetcher.factory(client), URI).build();
		imageLoader = new ImageLoader.Builder(context).components(componentRegistry).build();
		settings.registerObserver(this);
	}

	/**
	 * @param context Context used to initialize singleton instance
	 * @return instance of Picasso with custom downloader
	 */
	public static ImageWorker get(Context context) {
		if (notifySettingsChange || instance == null) {
			instance = new ImageWorker(context);
		}
		return instance;
	}

	/**
	 * clear memory cache
	 */
	public static void clear() {
		if (instance != null && instance.imageLoader.getMemoryCache() != null) {
			instance.imageLoader.getMemoryCache().clear();
		}
	}

	/**
	 *
	 */
	@Override
	public void onSettingsChange() {
		notifySettingsChange = true;
	}

	/**
	 * download an image from online source, then add to imageview
	 *
	 * @param imageView ImageView to add the downloaded image
	 * @param url       Url of the image
	 */
	public void load(ImageView imageView, String url) {
		load(imageView, url, null, 1.0f, false);
	}

	/**
	 * download a profile (avatar) image from online source, then add to imageview
	 *
	 * @param imageView ImageView to add the downloaded image
	 * @param url       Url of the image
	 */
	public void loadProfileImage(ImageView imageView, String url) {
		load(imageView, url, null, 1.0f, false);
	}

	/**
	 * download a profile banner image from online source, then add to imageview
	 *
	 * @param imageView ImageView to add the downloaded image
	 * @param url       Url of the image
	 */
	public void loadBannerImage(ImageView imageView, String url) {
		load(imageView, url, null, 1.0f, false);
	}

	/**
	 * load media image from posts
	 *
	 * @param imageView ImageView to add the downloaded image
	 * @param url       Url of the image
	 * @param blurHash  blur hash to create the placeholder image
	 * @param ratio     ratio of the image
	 */
	public void loadMediaImage(ImageView imageView, String url, @Nullable String blurHash, float ratio) {
		load(imageView, url, blurHash, ratio, true);
	}

	/**
	 * download an image from online source, then add to imageview.
	 * while downloading image, use a blurred image as placeholder
	 *
	 * @param imageView ImageView to add the downloaded image
	 * @param url       Url of the image
	 * @param blurHash  blur hash to create the placeholder image
	 * @param dontCache true to disable image caching
	 */
	private void load(ImageView imageView, String url, @Nullable String blurHash, float ratio, boolean dontCache) {
		BitmapImage blur = null;
		if (blurHash != null && !blurHash.isEmpty()) {
			blur = new BitmapImage(BlurHashDecoder.decode(blurHash, ratio), false);
		}
		ImageRequest.Builder requestBuilder = new ImageRequest.Builder(imageView.getContext());
		requestBuilder.data(url).target(new ImageViewTarget(imageView)).placeholder(blur);
		if (dontCache)
			requestBuilder.memoryCachePolicy(CachePolicy.DISABLED).diskCachePolicy(CachePolicy.DISABLED);
		imageLoader.enqueue(requestBuilder.build());
	}
}