package org.nuclearfog.smither.ui.adapter.listview;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.TextView;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.Media;
import org.nuclearfog.smither.utils.StringUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * ListView adapter used to show Meta items
 *
 * @author nuclearfog
 * @see org.nuclearfog.smither.ui.dialogs.MetaDialog
 */
public class MetaAdapter extends BaseAdapter {

	private List<String> keys = new ArrayList<>();
	private List<String> values = new ArrayList<>();

	private Context context;
	private GlobalSettings settings;

	/**
	 *
	 */
	public MetaAdapter(Context context) {
		this.context = context;
		settings = GlobalSettings.get(context);
	}


	@Override
	public int getCount() {
		return Math.min(keys.size(), values.size());
	}


	@Override
	public Object getItem(int position) {
		return new String[]{keys.get(position), values.get(position)};
	}


	@Override
	public long getItemId(int position) {
		return keys.get(position).hashCode();
	}


	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		TextView key_text;
		TextView value_text;
		if (convertView == null) {
			convertView = LayoutInflater.from(context).inflate(R.layout.item_meta_field, parent, false);
			key_text = convertView.findViewById(R.id.item_meta_field_key);
			value_text = convertView.findViewById(R.id.item_meta_field_value);
			key_text.setTextColor(settings.getTextColor());
			value_text.setTextColor(settings.getTextColor());
		} else {
			key_text = convertView.findViewById(R.id.item_meta_field_key);
			value_text = convertView.findViewById(R.id.item_meta_field_value);
		}
		key_text.setText(keys.get(position));
		value_text.setText(values.get(position));
		return convertView;
	}

	/**
	 * set adapter items
	 *
	 * @param meta media meta information
	 */
	public void setItems(Media.Meta meta) {
		if (meta.getWidth() > 1 && meta.getHeight() > 1) {
			keys.add(context.getString(R.string.dialog_meta_width));
			values.add(StringUtils.NUMBER_FORMAT.format(meta.getWidth()));
			keys.add(context.getString(R.string.dialog_meta_height));
			values.add(StringUtils.NUMBER_FORMAT.format(meta.getHeight()));
		}
		if (meta.getBitrate() > 0) {
			keys.add(context.getString(R.string.dialog_meta_bitrate));
			values.add(StringUtils.NUMBER_FORMAT.format(meta.getBitrate()) + " kBit/s");
		}
		if (meta.getFrameRate() > 0) {
			keys.add(context.getString(R.string.dialog_meta_framerate));
			values.add(StringUtils.NUMBER_FORMAT.format(meta.getFrameRate()) + " fps");
		}
		if (meta.getDuration() > 0) {
			StringBuilder durationValue = new StringBuilder();
			keys.add(context.getString(R.string.dialog_meta_duration));
			long hour = Math.round(meta.getDuration() / 3600);
			long min = Math.round(meta.getDuration() / 60) % 60L;
			long sec = Math.round(meta.getDuration() % 60.0);
			if (hour > 0) {
				if (hour < 10)
					durationValue.append('0');
				durationValue.append(hour).append(':');
			}
			if (min < 10)
				durationValue.append('0');
			durationValue.append(min).append(':');
			if (sec < 10)
				durationValue.append('0');
			durationValue.append(sec).append('\n');
			values.add(durationValue.toString());
		}
		notifyDataSetChanged();
	}
}