package org.nuclearfog.smither.backend.api.mastodon.impl;

import androidx.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.nuclearfog.smither.model.Media;
import org.nuclearfog.smither.model.Poll;
import org.nuclearfog.smither.model.ScheduledStatus;
import org.nuclearfog.smither.model.Status;
import org.nuclearfog.smither.utils.StringUtils;

/**
 * Mastodon implementation of a scheduled status
 *
 * @author nuclearfog
 */
public class ScheduledMastodonStatus implements ScheduledStatus {

	private static final long serialVersionUID = -1340937182294786469L;

	private long id;
	private long time;
	private String text;
	private String language = "";
	private int visibility;
	private boolean sensitive;
	private boolean spoiler;
	private Media[] medias = {};
	private Poll poll;

	/**
	 *
	 */
	public ScheduledMastodonStatus(JSONObject json) throws JSONException {
		JSONObject params = json.getJSONObject("params");
		JSONObject pollJson = params.optJSONObject("poll");
		JSONArray mediaArray = json.optJSONArray("media_attachments");
		String idStr = json.getString("id");
		String visibilityStr = params.optString("visibility", "");
		text = StringUtils.extractText(params.optString("text", ""));
		time = StringUtils.getIsoTime(json.optString("scheduled_at", ""));
		sensitive = params.optBoolean("sensitive", false);
		spoiler = params.optBoolean("spoiler_text", false);

		if (!params.isNull("language")) {
			language = params.optString("language");
		}
		if (pollJson != null) {
			poll = new MastodonPoll(pollJson);
		}
		if (mediaArray != null && mediaArray.length() > 0) {
			medias = new Media[mediaArray.length()];
			for (int i = 0; i < mediaArray.length(); i++) {
				JSONObject mediaItem = mediaArray.getJSONObject(i);
				medias[i] = new MastodonMedia(mediaItem);
			}
		}
		switch (visibilityStr) {
			case "public":
				visibility = Status.VISIBLE_PUBLIC;
				break;

			case "private":
				visibility = Status.VISIBLE_PRIVATE;
				break;

			case "direct":
				visibility = Status.VISIBLE_DIRECT;
				break;

			case "unlisted":
				visibility = Status.VISIBLE_UNLISTED;
				break;

			default:
				visibility = Status.VISIBLE_DEFAULT;
				break;
		}
		try {
			id = Long.parseLong(idStr);
		} catch (NumberFormatException exception) {
			throw new JSONException("Bad ID: " + idStr);
		}
	}


	@Override
	public long getId() {
		return id;
	}


	@Override
	public long getPublishTime() {
		return time;
	}


	@Override
	public String getText() {
		return text;
	}


	@Override
	public String getLanguage() {
		return language;
	}


	@Override
	public Media[] getMedia() {
		return medias;
	}


	@Nullable
	@Override
	public Poll getPoll() {
		return poll;
	}


	@Override
	public int getVisibility() {
		return visibility;
	}


	@Override
	public boolean isSensitive() {
		return sensitive;
	}


	@Override
	public boolean isSpoiler() {
		return spoiler;
	}


	@Override
	public boolean equals(@Nullable Object obj) {
		if (!(obj instanceof ScheduledStatus))
			return false;
		return ((ScheduledStatus) obj).getId() == getId();
	}
}