package org.nuclearfog.smither.backend.api.mastodon.impl;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.nuclearfog.smither.model.Card;
import org.nuclearfog.smither.model.Emoji;
import org.nuclearfog.smither.model.Location;
import org.nuclearfog.smither.model.Media;
import org.nuclearfog.smither.model.Poll;
import org.nuclearfog.smither.model.Status;
import org.nuclearfog.smither.model.User;
import org.nuclearfog.smither.utils.StringUtils;

import java.util.Locale;

/**
 * Status implementation for Mastodon
 *
 * @author nuclearfog
 */
public class MastodonStatus implements Status {

	private static final long serialVersionUID = 1184375228249441241L;

	private long id;
	private long replyId;
	private long replyUserId;
	private long createdAt;
	private long editedAt;

	private int replyCount;
	private int favoriteCount;
	private int reblogCount;
	private int visibility;
	private boolean favored;
	private boolean reposted;
	private boolean bookmarked;
	private boolean sensitive;
	private boolean spoiler;
	private boolean muted;
	private boolean isPinned;

	private String text;
	private String mentions = "";
	private String language = "";
	private String source = "";
	private String url = "";

	private User author;
	private Poll poll;
	private Status embeddedStatus;
	private Card[] cards = {};
	private Media[] medias = {};
	private Emoji[] emojis = {};

	/**
	 * @param json          Mastodon status json object
	 * @param currentUserId Id of the current user
	 */
	public MastodonStatus(JSONObject json, long currentUserId) throws JSONException {
		JSONObject embeddedJson = json.optJSONObject("reblog");
		JSONObject appJson = json.optJSONObject("application");
		JSONObject cardJson = json.optJSONObject("card");
		JSONObject pollJson = json.optJSONObject("poll");
		JSONArray mentionsJson = json.optJSONArray("mentions");
		JSONArray mediaArray = json.optJSONArray("media_attachments");
		JSONArray emojiArray = json.optJSONArray("emojis");
		String replyIdStr = json.optString("in_reply_to_id", "0");
		String replyUserIdStr = json.optString("in_reply_to_account_id", "0");
		String idStr = json.getString("id");
		String visibilityStr = json.getString("visibility");
		String language = json.optString("language", "");
		String editedAtStr = json.optString("edited_at");

		author = new MastodonUser(json.getJSONObject("account"), currentUserId);
		createdAt = StringUtils.getIsoTime(json.optString("created_at"));
		replyCount = json.optInt("replies_count");
		reblogCount = json.optInt("reblogs_count");
		favoriteCount = json.optInt("favourites_count");
		muted = json.optBoolean("muted", false);
		favored = json.optBoolean("favourited", false);
		reposted = json.optBoolean("reblogged", false);
		sensitive = json.optBoolean("sensitive", false);
		spoiler = json.optBoolean("spoiler_text", false);
		bookmarked = json.optBoolean("bookmarked", false);
		isPinned = json.optBoolean("pinned", false);
		text = StringUtils.extractText(json.optString("content", ""));

		if (!editedAtStr.isEmpty() && !editedAtStr.equals("null"))
			editedAt = StringUtils.getIsoTime(editedAtStr);
		if (author.getId() != currentUserId)
			mentions = author.getScreenName() + ' ';
		if (embeddedJson != null) {
			embeddedStatus = new MastodonStatus(embeddedJson, currentUserId);
			this.url = embeddedStatus.getUrl();
		} else if (!json.isNull("url")) {
			this.url = json.optString("url", "");
		}
		if (pollJson != null) {
			poll = new MastodonPoll(pollJson);
		}
		if (mediaArray != null && mediaArray.length() > 0) {
			medias = new Media[mediaArray.length()];
			for (int i = 0; i < mediaArray.length(); i++) {
				JSONObject mediaItem = mediaArray.getJSONObject(i);
				medias[i] = new MastodonMedia(mediaItem);
			}
		}
		if (mentionsJson != null && mentionsJson.length() > 0) {
			StringBuilder mentionsBuilder = new StringBuilder(mentions);
			for (int i = 0; i < mentionsJson.length(); i++) {
				long mentionUserId = 0L;
				JSONObject mentionJson = mentionsJson.getJSONObject(i);
				String mention = '@' + mentionJson.getString("acct");
				String mentionedUserIdStr = mentionJson.optString("id", "0");
				if (mentionedUserIdStr.matches("\\d+"))
					mentionUserId = Long.parseLong(mentionsJson.getJSONObject(i).getString("id"));
				if (mentionUserId != 0L && mentionUserId != currentUserId) {
					mentionsBuilder.append(mention).append(' ');
				}
			}
			mentions = mentionsBuilder.toString();
		}
		if (emojiArray != null && emojiArray.length() > 0) {
			emojis = new Emoji[emojiArray.length()];
			for (int i = 0; i < emojis.length; i++) {
				JSONObject emojiJson = emojiArray.getJSONObject(i);
				emojis[i] = new MastodonEmoji(emojiJson);
			}
		}
		if (appJson != null) {
			source = appJson.optString("name", "");
		} else if (embeddedStatus != null) {
			source = embeddedStatus.getSource();
		}
		if (cardJson != null) {
			cards = new Card[]{new MastodonCard(cardJson)};
		}
		if (!language.isEmpty() && !language.equals("null") && !language.equals(Locale.getDefault().getLanguage())) {
			this.language = language;
		}
		switch (visibilityStr) {
			case "public":
				visibility = VISIBLE_PUBLIC;
				break;

			case "private":
				visibility = VISIBLE_PRIVATE;
				break;

			case "direct":
				visibility = VISIBLE_DIRECT;
				break;

			case "unlisted":
				visibility = VISIBLE_UNLISTED;
				break;

			default:
				visibility = VISIBLE_DEFAULT;
				break;
		}
		try {
			id = Long.parseLong(idStr);
			if (!replyIdStr.equals("null"))
				replyId = Long.parseLong(replyIdStr);
			if (!replyUserIdStr.equals("null"))
				replyUserId = Long.parseLong(replyUserIdStr);
		} catch (NumberFormatException e) {
			throw new JSONException("bad ID:" + idStr + ' ' + replyIdStr + ' ' + replyUserIdStr);
		}
	}


	@Override
	public long getId() {
		return id;
	}


	@Override
	public String getText() {
		return text;
	}


	@Override
	public User getAuthor() {
		return author;
	}


	@Override
	public long getTimestamp() {
		return createdAt;
	}


	@Override
	public String getSource() {
		return source;
	}


	@Nullable
	@Override
	public Status getEmbeddedStatus() {
		return embeddedStatus;
	}


	@Override
	public String getReplyName() {
		return "";
	}


	@Override
	public long getRepliedUserId() {
		return replyUserId;
	}


	@Override
	public long getRepliedStatusId() {
		return replyId;
	}


	@Override
	public long getRepostId() {
		return 0L;
	}


	@Override
	public int getRepostCount() {
		return reblogCount;
	}


	@Override
	public int getFavoriteCount() {
		return favoriteCount;
	}


	@Override
	public int getReplyCount() {
		return replyCount;
	}


	@Override
	public int getVisibility() {
		return visibility;
	}


	@NonNull
	@Override
	public Media[] getMedia() {
		return medias;
	}


	@NonNull
	@Override
	public Emoji[] getEmojis() {
		return emojis;
	}


	@Override
	public String getUserMentions() {
		return mentions;
	}


	@Override
	public long editedAt() {
		return editedAt;
	}


	@Override
	public boolean isSensitive() {
		return sensitive;
	}


	@Override
	public boolean isSpoiler() {
		return spoiler;
	}


	@Override
	public boolean isReposted() {
		return reposted;
	}


	@Override
	public boolean isFavored() {
		return favored;
	}


	@Override
	public boolean isBookmarked() {
		return bookmarked;
	}


	@Override
	public boolean isPinned() {
		return isPinned;
	}


	@Override
	public boolean isHidden() {
		return muted;
	}


	@Override
	public String getUrl() {
		return url;
	}


	@Override
	@Nullable
	public Location getLocation() {
		return null; // todo add implementation if supported by API
	}


	@Override
	public String getLanguage() {
		return language;
	}


	@NonNull
	@Override
	public Card[] getCards() {
		return cards;
	}


	@Nullable
	@Override
	public Poll getPoll() {
		return poll;
	}


	@Override
	public boolean equals(@Nullable Object obj) {
		if (!(obj instanceof Status))
			return false;
		Status status = ((Status) obj);
		return status.getId() == id && status.getTimestamp() == getTimestamp() && status.getAuthor().equals(getAuthor());
	}


	@NonNull
	@Override
	public String toString() {
		return "from=\"" + getAuthor().getScreenName() + "\" text=\"" + getText() + "\"";
	}

	/**
	 * set repost status
	 */
	public MastodonStatus setRepost(boolean enable) {
		this.reposted = enable;
		if (embeddedStatus instanceof MastodonStatus) {
			((MastodonStatus) embeddedStatus).setRepost(enable);
		}
		return this;
	}

	/**
	 * set favorite status
	 */
	public MastodonStatus setFavorite(boolean enable) {
		this.favored = enable;
		if (embeddedStatus instanceof MastodonStatus) {
			((MastodonStatus) embeddedStatus).setFavorite(enable);
		}
		return this;
	}

	/**
	 * set bookmark status
	 */
	public MastodonStatus setBookmark(boolean enable) {
		this.bookmarked = enable;
		if (embeddedStatus instanceof MastodonStatus) {
			((MastodonStatus) embeddedStatus).setBookmark(enable);
		}
		return this;
	}

	/**
	 * set status pinned
	 */
	public MastodonStatus setPined(boolean enable) {
		this.isPinned = enable;
		if (embeddedStatus instanceof MastodonStatus) {
			((MastodonStatus) embeddedStatus).setPined(enable);
		}
		return this;
	}
}