package org.nuclearfog.smither.ui.dialogs;

import android.content.Context;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemSelectedListener;
import android.widget.Button;
import android.widget.CompoundButton;
import android.widget.CompoundButton.OnCheckedChangeListener;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.backend.async.AsyncExecutor.AsyncCallback;
import org.nuclearfog.smither.backend.async.UserListUpdater;
import org.nuclearfog.smither.backend.helper.update.UserListUpdate;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.UserList;
import org.nuclearfog.smither.ui.adapter.listview.DropdownAdapter;
import org.nuclearfog.smither.ui.views.InputView;
import org.nuclearfog.smither.ui.views.InputView.OnTextChangeListener;
import org.nuclearfog.smither.utils.AppStyles;
import org.nuclearfog.smither.utils.ErrorUtils;

/**
 * dialog used to create or update an user list
 *
 * @author nuclearfog
 */
public class UserListDialog extends DialogFragment implements OnClickListener, OnItemSelectedListener, OnCheckedChangeListener, OnTextChangeListener, AsyncCallback<UserListUpdater.Result> {

	/**
	 *
	 */
	private static final String TAG = "UserListDialog";

	/**
	 * Bundle key used to set/restore user list configuration
	 *
	 * @value type is {@link UserList} or {@link UserListUpdate}
	 */
	private static final String KEY_USERLIST = "userlist";

	private InputView title_input;

	private UserListUpdater listUpdater;

	private UserListUpdate userListUpdater = new UserListUpdate();

	/**
	 *
	 */
	public UserListDialog() {
		setStyle(STYLE_NO_TITLE, R.style.DefaultDialog);
	}


	@Nullable
	@Override
	public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
		View view = inflater.inflate(R.layout.dialog_userlist, container, false);
		Button apply = view.findViewById(R.id.dialog_userlist_apply);
		View button_cancel = view.findViewById(R.id.dialog_userlist_cancel);
		TextView title_dialog = view.findViewById(R.id.dialog_userlist_title_dialog);
		title_input = view.findViewById(R.id.dialog_userlist_title_input);
		CompoundButton exclusive = view.findViewById(R.id.dialog_userlist_exclusive);
		Spinner policy = view.findViewById(R.id.dialog_userlist_replies_selector);

		listUpdater = new UserListUpdater(requireContext());
		GlobalSettings settings = GlobalSettings.get(requireContext());
		DropdownAdapter adapter = new DropdownAdapter(requireContext());

		adapter.setItems(R.array.userlist_policy);
		policy.setAdapter(adapter);
		AppStyles.setTheme((ViewGroup) view, settings.getPopupColor());

		if (savedInstanceState == null)
			savedInstanceState = getArguments();
		if (savedInstanceState != null) {
			Object data = savedInstanceState.getSerializable(KEY_USERLIST);
			if (data instanceof UserListUpdate) {
				userListUpdater = (UserListUpdate) data;
			} else if (data instanceof UserList) {
				userListUpdater = new UserListUpdate((UserList) data);
			}
		}
		if (userListUpdater.getId() != 0L) {
			title_dialog.setText(R.string.userlist_update_list);
			apply.setText(R.string.userlist_update);
		} else {
			title_dialog.setText(R.string.userlist_create_new_list);
			apply.setText(R.string.userlist_create);
		}
		title_input.setText(userListUpdater.getTitle());
		exclusive.setChecked(userListUpdater.isExclusive());
		if (userListUpdater.getPolicy() == UserList.REPLIES_NONE) {
			policy.setSelection(0);
		} else if (userListUpdater.getPolicy() == UserList.REPLIES_FOLLOWING) {
			policy.setSelection(1);
		} else if (userListUpdater.getPolicy() == UserList.REPLIES_MEMBER) {
			policy.setSelection(2);
		}
		apply.setOnClickListener(this);
		button_cancel.setOnClickListener(this);
		policy.setOnItemSelectedListener(this);
		exclusive.setOnCheckedChangeListener(this);
		title_input.setOnTextChangeListener(this);
		return view;
	}


	@Override
	public void onSaveInstanceState(@NonNull Bundle outState) {
		outState.putSerializable(KEY_USERLIST, userListUpdater);
		super.onSaveInstanceState(outState);
	}


	@Override
	public void onDestroyView() {
		listUpdater.cancel();
		super.onDestroyView();
	}


	@Override
	public void onClick(View v) {
		if (v.getId() == R.id.dialog_userlist_apply) {
			if (userListUpdater.getTitle().isEmpty()) {
				title_input.setError(title_input.getContext().getString(R.string.error_list_title_empty));
			} else if (listUpdater.isIdle()) {
				listUpdater.execute(userListUpdater, this);
				title_input.setError(null);
			}
		} else if (v.getId() == R.id.dialog_userlist_cancel) {
			dismiss();
		}
	}


	@Override
	public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
		if (buttonView.getId() == R.id.dialog_userlist_exclusive) {
			userListUpdater.setExclusive(isChecked);
		}
	}


	@Override
	public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
		if (parent.getId() == R.id.dialog_userlist_replies_selector) {
			if (position == 0) {
				userListUpdater.setPolicy(UserList.REPLIES_NONE);
			} else if (position == 1) {
				userListUpdater.setPolicy(UserList.REPLIES_FOLLOWING);
			} else if (position == 2) {
				userListUpdater.setPolicy(UserList.REPLIES_MEMBER);
			}
		}
	}


	@Override
	public void onNothingSelected(AdapterView<?> parent) {
	}


	@Override
	public void onTextChanged(InputView inputView, String text) {
		if (inputView.getId() == R.id.dialog_userlist_title_input) {
			userListUpdater.setTitle(text);
		}
	}


	@Override
	public void onResult(@NonNull UserListUpdater.Result result) {
		Context context = getContext();
		if (result.action == UserListUpdater.Result.CREATED) {
			if (context != null)
				Toast.makeText(context, R.string.info_list_created, Toast.LENGTH_SHORT).show();
			if (getActivity() instanceof UserListUpdatedCallback)
				((UserListUpdatedCallback) getActivity()).onUserListUpdate(result.userlist);
			dismiss();
		} else if (result.action == UserListUpdater.Result.UPDATED) {
			if (context != null)
				Toast.makeText(context, R.string.info_list_updated, Toast.LENGTH_SHORT).show();
			if (getActivity() instanceof UserListUpdatedCallback)
				((UserListUpdatedCallback) getActivity()).onUserListUpdate(result.userlist);
			dismiss();
		} else if (result.action == UserListUpdater.Result.ERROR) {
			if (context != null) {
				ErrorUtils.showErrorMessage(context, result.exception);
			}
		}
	}

	/**
	 *
	 */
	public static void show(FragmentActivity activity, UserList userlist) {
		Fragment dialogFragment = activity.getSupportFragmentManager().findFragmentByTag(TAG);
		if (dialogFragment == null) {
			UserListDialog dialog = new UserListDialog();
			Bundle param = new Bundle();
			param.putSerializable(KEY_USERLIST, userlist);
			dialog.setArguments(param);
			dialog.show(activity.getSupportFragmentManager(), TAG);
		}
	}

	/**
	 * Callback interface used to update user list information
	 */
	public interface UserListUpdatedCallback {

		/**
		 * called if the user list is successfully updated
		 *
		 * @param userList new/updated userList
		 */
		void onUserListUpdate(UserList userList);
	}
}