package org.nuclearfog.smither.ui.adapter.recyclerview.holder;

import android.graphics.Color;
import android.text.Spannable;
import android.text.SpannableString;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.SeekBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.RecyclerView.ViewHolder;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.backend.async.AsyncExecutor.AsyncCallback;
import org.nuclearfog.smither.backend.async.TextEmojiLoader;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.Emoji;
import org.nuclearfog.smither.model.PollOption;
import org.nuclearfog.smither.utils.AppStyles;
import org.nuclearfog.smither.utils.EmojiUtils;
import org.nuclearfog.smither.utils.StringUtils;

/**
 * This holder if for a single poll option
 *
 * @author nuclearfog
 * @see org.nuclearfog.smither.ui.adapter.recyclerview.OptionsAdapter
 */
public class OptionHolder extends ViewHolder implements OnClickListener, AsyncCallback<TextEmojiLoader.Result> {

	private SeekBar voteProgress;
	private TextView optionName, optionVotes;
	private ImageView checkIcon;

	private OnHolderClickListener listener;
	private TextEmojiLoader emojiLoader;
	private GlobalSettings settings;

	private long tagId;

	/**
	 *
	 */
	public OptionHolder(ViewGroup parent, OnHolderClickListener listener) {
		super(LayoutInflater.from(parent.getContext()).inflate(R.layout.item_option, parent, false));
		settings = GlobalSettings.get(parent.getContext());
		emojiLoader = new TextEmojiLoader(parent.getContext());
		optionName = itemView.findViewById(R.id.item_option_name);
		checkIcon = itemView.findViewById(R.id.item_option_voted_icon);
		voteProgress = itemView.findViewById(R.id.item_option_count_bar);
		optionVotes = itemView.findViewById(R.id.item_option_count_text);
		this.listener = listener;

		voteProgress.setPadding(0, 0, 0, 0);
		AppStyles.setTheme((ViewGroup) itemView, Color.TRANSPARENT);

		checkIcon.setOnClickListener(this);
	}


	@Override
	public void onClick(View v) {
		int position = getLayoutPosition();
		if (position != RecyclerView.NO_POSITION) {
			if (v == checkIcon) {
				listener.onItemClick(position, OnHolderClickListener.POLL_OPTION);
			}
		}
	}


	@Override
	public void onResult(@NonNull TextEmojiLoader.Result result) {
		if (result.images != null && result.id == tagId) {
			Spannable spannable = EmojiUtils.addEmojis(optionName.getContext(), result.spannable, result.images);
			optionName.setText(spannable);
		}
	}

	/**
	 * set view holder content
	 *
	 * @param option     poll option content
	 * @param selected   true if option is selected
	 * @param totalCount total vote count
	 */
	public void setContent(PollOption option, Emoji[] emojis, boolean selected, int totalCount) {
		voteProgress.setMax(Math.max(totalCount, 1));
		voteProgress.setProgress(option.getVotes());
		optionVotes.setText(StringUtils.NUMBER_FORMAT.format(option.getVotes()));
		if (emojis.length > 0 && settings.imagesEnabled()) {
			tagId = option.getTitle().hashCode();
			SpannableString optionSpan = new SpannableString(option.getTitle());
			TextEmojiLoader.Param param = new TextEmojiLoader.Param(tagId, emojis, optionSpan, optionName.getResources().getDimensionPixelSize(R.dimen.item_option_emoji_size));
			optionName.setText(EmojiUtils.removeTags(optionSpan));
			emojiLoader.execute(param, this);
		} else {
			optionName.setText(option.getTitle());
		}
		if (option.isSelected() | selected) {
			checkIcon.setImageResource(R.drawable.check);
		} else {
			checkIcon.setImageResource(R.drawable.circle);
		}
		AppStyles.setDrawableColor(checkIcon, settings.getIconColor());
	}
}