package org.nuclearfog.smither.ui.adapter.recyclerview.holder;

import android.text.Spannable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.RecyclerView.ViewHolder;

import org.nuclearfog.smither.R;
import org.nuclearfog.smither.backend.async.AsyncExecutor.AsyncCallback;
import org.nuclearfog.smither.backend.async.TextEmojiLoader;
import org.nuclearfog.smither.config.GlobalSettings;
import org.nuclearfog.smither.model.Announcement;
import org.nuclearfog.smither.ui.adapter.recyclerview.ReactionAdapter;
import org.nuclearfog.smither.ui.adapter.recyclerview.ReactionAdapter.OnReactionSelected;
import org.nuclearfog.smither.utils.AppStyles;
import org.nuclearfog.smither.utils.EmojiUtils;
import org.nuclearfog.smither.utils.StringUtils;
import org.nuclearfog.smither.utils.Tagger;

/**
 * View holder for {@link org.nuclearfog.smither.ui.adapter.recyclerview.AnnouncementAdapter}
 *
 * @author nuclearfog
 */
public class AnnouncementHolder extends ViewHolder implements OnClickListener, OnReactionSelected, AsyncCallback<TextEmojiLoader.Result> {

	private TextView time, content;
	private View dismissButton;

	private OnHolderClickListener listener;
	private GlobalSettings settings;
	private TextEmojiLoader emojiLoader;
	private ReactionAdapter adapter;
	private int iconSize;

	private long tagId = 0L;

	/**
	 *
	 */
	public AnnouncementHolder(ViewGroup parent, OnHolderClickListener listener) {
		super(LayoutInflater.from(parent.getContext()).inflate(R.layout.item_announcement, parent, false));
		settings = GlobalSettings.get(parent.getContext());
		emojiLoader = new TextEmojiLoader(parent.getContext());
		adapter = new ReactionAdapter(this);
		this.listener = listener;
		ViewGroup container = itemView.findViewById(R.id.item_announcement_container);
		dismissButton = itemView.findViewById(R.id.item_announcement_dismiss);
		RecyclerView reactionList = itemView.findViewById(R.id.item_announcement_list_reactions);
		time = itemView.findViewById(R.id.item_announcement_timestamp);
		content = itemView.findViewById(R.id.item_announcement_content);
		iconSize = parent.getResources().getDimensionPixelSize(R.dimen.item_announcement_icon_size);

		reactionList.setLayoutManager(new LinearLayoutManager(parent.getContext(), LinearLayoutManager.HORIZONTAL, false));
		reactionList.setAdapter(adapter);
		AppStyles.setTheme((ViewGroup) itemView);

		container.setOnClickListener(this);
		dismissButton.setOnClickListener(this);
	}


	@Override
	public void onClick(View v) {
		int position = getLayoutPosition();
		if (position != RecyclerView.NO_POSITION) {
			if (v.getId() == R.id.item_announcement_container) {
				listener.onItemClick(position, OnHolderClickListener.ANNOUNCEMENT_CLICK);
			} else if (v.getId() == R.id.item_announcement_dismiss) {
				listener.onItemClick(position, OnHolderClickListener.ANNOUNCEMENT_DISMISS);
			}
		}
	}


	@Override
	public void onReactionClick(int index) {
		int position = getLayoutPosition();
		if (position != RecyclerView.NO_POSITION) {
			listener.onItemClick(position, OnHolderClickListener.ANNOUNCEMENT_REACTION, index);
		}
	}


	@Override
	public void onResult(@NonNull TextEmojiLoader.Result result) {
		if (result.id == tagId && result.images != null) {
			Spannable spannable = EmojiUtils.addEmojis(content.getContext(), result.spannable, result.images);
			content.setText(spannable);
		}
	}

	/**
	 * set holder content
	 */
	public void setContent(Announcement announcement) {
		Spannable textSpan = Tagger.makeTextWithLinks(announcement.getMessage(), settings.getHighlightColor());
		if (announcement.getEmojis().length > 0 && settings.imagesEnabled()) {
			TextEmojiLoader.Param param = new TextEmojiLoader.Param(tagId, announcement.getEmojis(), textSpan, iconSize);
			emojiLoader.execute(param, this);
			textSpan = EmojiUtils.removeTags(textSpan);
		}
		if (announcement.isDismissed()) {
			dismissButton.setVisibility(View.GONE);
		} else {
			dismissButton.setVisibility(View.VISIBLE);
		}
		content.setText(textSpan);
		time.setText(StringUtils.formatCreationTime(time.getResources(), announcement.getTimestamp()));
		adapter.setItems(announcement.getReactions(), announcement.getEmojis());
	}
}